/****************************************************************************
*                   hfield.c
*
*	This file implements the height field shape primitive.  The shape is
*	implemented as a collection of triangles which are calculated as
*	needed.  The basic intersection routine first computes the rays
*	intersection with the box marking the limits of the shape, then
*	follows the line from one intersection point to the other, testing the
*	two triangles which form the pixel for an intersection with the ray at
*	each step.
*		height field added by Doug Muir
*		with lots of advice and support from David Buck 
*			and Drew Wells.
*
*  from Persistence of Vision Raytracer 
*  Copyright 1991 Persistence of Vision Team
*---------------------------------------------------------------------------
*                       *IMPORTANT!*
*  This copyrighted software is freely distributable. The source and/or
* object code may be copied or uploaded to communications services so long as
* this notice remains at the top of each file.
* 
*  If any changes are made to the program, you must clearly indicate in the
* documentation and in the program startup message who it was who made the
* changes. The documentation should also describe what those changes were.
* 
*  This software may not be included in whole or in part into any commercial
* package without the express written consent of the PV-Team. It may,
* however, be included in other freely distributed software so long as proper
* credit for the software is given. No more than five dollars U.S. ($5) can
* be charged for the copying of this software and the media it is provided on,
* i.e. a shareware distribution company may only charge five U.S dollars or
* less for providing this software.
* 
*  This software is provided as is without any guarantees or warranty.
* Although the authors have attempted to find and correct any bugs in the
* software, they are not responsible for any damage caused by the use of the
* software. The authors are under no obligation to provide service,
* corrections, or upgrades to this package.
*-----------------------------------------------------------------------------
*  Despite all the legal stuff above, if you have any problems with the
* program the PV-Team would like to hear about them. Also, if you have any
* comments, questions or enhancements, please contact the PV-Team on the
* Compuserve Online Service in the COMART forum message section 16 (!GO
* COMART). The CIS COMART forum is devoted to computer generated artwork like
* raytracing, animation and fractals. For more information regarding the PV
* team see the file PVINF.TXT. For more information on Compuserve call
* (in the U.S.) 1-800-848-8990.
* 
*       Drew Wells
*       PV-Team Leader
*       CIS: 73767,1244
* 
* 
*  This program is based on the popular DKB raytracer version 2.12 written by
* David Buck, a PV-Team member.
*  (David Buck CIS: 70521,1371 Internet: dbuck@ccs.carleton.ca)
* 
*****************************************************************************/
 
 
#include "frame.h"
#include "vector.h"
#include "pvproto.h"

#define sign(x) ((x) > 0.0 ? 1: ((x) == 0.0 ? 0: (-1)))

#ifndef min
#define min(x,y) ((x) > (y) ? (y) : (x))
#endif

METHODS Height_Field_Methods = {
	Object_Intersect, All_HeightFld_Intersections,
	Inside_HeightFld, HeightFld_Normal,
	Copy_HeightFld,
	Translate_HeightFld, Rotate_HeightFld,
	Scale_HeightFld, Invert_HeightFld};

extern HEIGHT_FIELD *Get_Height_Field_Shape();

extern long Ray_Ht_Field_Tests, Ray_Ht_Field_Tests_Succeeded;
extern int Options;

int Intersect_Box(H_Field, Ray, depth1, depth2)
     HEIGHT_FIELD *H_Field;
     RAY *Ray;
     DBL *depth1;
     DBL *depth2;
{
   DBL dot, pos, depth, s, t;
   int hit_box;

   hit_box = 0;

   if(hit_box < 2)
      {
      VDot(dot,H_Field->n1,Ray->Direction);
      if((dot > Small_Tolerance) || (dot < -Small_Tolerance))
	 {
	 VDot(pos,H_Field->n1,Ray->Initial);
	 pos += H_Field->d1;

	 depth = -pos / dot;
	 if((depth > Small_Tolerance) && (depth < Max_Distance))
	    {
	    s = Ray->Initial.y + depth * Ray->Direction.y;
	    t = Ray->Initial.z + depth * Ray->Direction.z;

	    if((s >= H_Field->d4) && (s < -H_Field->d3) && (t >= 0.0) && (t < -H_Field->d5))
	       {
	       if (hit_box==0)
		  *depth1 = depth;
	       else
		  *depth2 = depth;
	       hit_box++;
	       }
	    }
	 }
      }

      if(hit_box < 2)
      {
      VDot(dot,H_Field->n2,Ray->Direction);
      if((dot > Small_Tolerance) || (dot < -Small_Tolerance))
	 {
	 VDot(pos,H_Field->n2,Ray->Initial);
	 pos += H_Field->d2;

	 depth = -pos / dot;
	 if((depth > Small_Tolerance) && (depth < Max_Distance))
	    {
	    s = Ray->Initial.y + depth * Ray->Direction.y;
	    t = Ray->Initial.z + depth * Ray->Direction.z;

	    if((s >= H_Field->d4) && (s < -H_Field->d3) && (t >= 0.0) && (t < -H_Field->d5))
	       {
	       if (hit_box==0)
		  *depth1 = depth;
	       else
		  *depth2 = depth;
	       hit_box++;
	       }
	    }
	 }
      }

      if(hit_box < 2)
      {
      VDot(dot,H_Field->n3,Ray->Direction);
      if((dot > Small_Tolerance) || (dot < -Small_Tolerance))
	 {
	 VDot(pos,H_Field->n3,Ray->Initial);
	 pos += H_Field->d3;

	 depth = -pos / dot;
	 if((depth > Small_Tolerance) && (depth < Max_Distance))
	    {
	    s = Ray->Initial.x + depth * Ray->Direction.x;
	    t = Ray->Initial.z + depth * Ray->Direction.z;

	    if((s >= 0.0) && (s < -H_Field->d1) && (t >= 0.0) && (t < -H_Field->d5))
	       {
	       if (hit_box==0)
		  *depth1 = depth;
	       else
		  *depth2 = depth;
	       hit_box++;
	       }
	    }
	 }
      }

      if(hit_box < 2)
      {
      VDot(dot,H_Field->n4,Ray->Direction);
      if((dot > Small_Tolerance) || (dot < -Small_Tolerance))
	 {
	 VDot(pos,H_Field->n4,Ray->Initial);
	 pos += H_Field->d4;

	 depth = -pos / dot;
	 if((depth > Small_Tolerance) && (depth < Max_Distance))
	    {
	    s = Ray->Initial.x + depth * Ray->Direction.x;
	    t = Ray->Initial.z + depth * Ray->Direction.z;

	    if((s >= 0.0) && (s < -H_Field->d1) && (t >= 0.0) && (t < -H_Field->d5))
	       {
	       if (hit_box==0)
		  *depth1 = depth;
	       else
		  *depth2 = depth;
	       hit_box++;
	       }
	    }
	 }
      }

      if(hit_box < 2)
      {
      VDot(dot,H_Field->n5,Ray->Direction);
      if((dot > Small_Tolerance) || (dot < -Small_Tolerance))
	 {
	 VDot(pos,H_Field->n5,Ray->Initial);
	 pos += H_Field->d5;

	 depth = -pos / dot;
	 if((depth > Small_Tolerance) && (depth < Max_Distance))
	    {
	    s = Ray->Initial.x + depth * Ray->Direction.x;
	    t = Ray->Initial.y + depth * Ray->Direction.y;

	    if((s >= 0.0) && (s < -H_Field->d1) && (t >= H_Field->d4) && (t < -H_Field->d3))
	       {
	       if (hit_box==0)
		  *depth1 = depth;
	       else
		  *depth2 = depth;
	       hit_box++;
	       }
	    }
	 }
      }

      if(hit_box < 2)
      {
      VDot(dot,H_Field->n6,Ray->Direction);
      if((dot > Small_Tolerance) || (dot < -Small_Tolerance))
	 {
	 VDot(pos,H_Field->n6,Ray->Initial);
	 pos += H_Field->d6;

	 depth = -pos / dot;
	 if((depth > Small_Tolerance) && (depth < Max_Distance))
	    {
	    s = Ray->Initial.x + depth * Ray->Direction.x;
	    t = Ray->Initial.y + depth * Ray->Direction.y;

	    if((s >= 0.0) && (s < -H_Field->d1) && (t >= H_Field->d4) && (t < -H_Field->d3))
	       {
	       if (hit_box==0)
		  *depth1 = depth;
	       else
		  *depth2 = depth;
	       hit_box++;
	       }
	    }
	 }
      }
   if (hit_box == 0)
      return(FALSE);
   if (hit_box == 2)
      return(TRUE);

   if ((Ray->Initial.x >= 0.0) && (Ray->Initial.x < -H_Field->d1) && (Ray->Initial.y >= H_Field->d4) && (Ray->Initial.y < -H_Field->d3) && (Ray->Initial.z >= 0.0) && (Ray->Initial.z < -H_Field->d5))
      {
      *depth2 = 0.0;
      return(TRUE);
      }
   return(FALSE);
}

DBL Get_Height(x, y, H_Field)
     int x;
     int y;
     HEIGHT_FIELD *H_Field;
{
	unsigned int temp1, temp2;
	int iwidth;
	DBL dtemp=0.0;
	IMAGE *Map;

	y = (H_Field->iheight - 1)-y; /*Account for images being upside down*/

	if(((x >= 0) && (x < H_Field->iwidth)) && ((y >= 0) && (y < H_Field->iheight)))
	{
		iwidth = H_Field->iwidth;
		Map = H_Field->Map;
		switch (H_Field->Map_Type)
		{
		    case GIF:
			dtemp  = (DBL)Map->data.map_lines[y][x];
			break;
		    case POT:
			temp1 = Map->data.map_lines[y][x];
			temp2 = Map->data.map_lines[y][x+iwidth];
			dtemp = (DBL)((DBL)temp1 + (DBL)temp2/256.0);
			break;
		}
	}
        if(Options & DEBUGGING)
	   printf("Returning vertex height as %f",dtemp);
	return dtemp;
}

int Intersect_Pixel(x, z, Ray, H_Field, height1, height2, Depth)
     int x;
     int z;
     RAY *Ray;
     HEIGHT_FIELD *H_Field;
     DBL height1;
     DBL height2;
     DBL *Depth;
{
	VECTOR T1V1,T1V2,T1V3,T2V1,T2V2,T2V3,Local_Normal;
	DBL pos1,pos2,dot,depth1,depth2,s,t;

	depth1 = Max_Distance;
	depth2 = Max_Distance;

	Make_Vector(&T1V1,(DBL)x,Get_Height(x,z,H_Field),(DBL)z);
	Make_Vector(&T1V2,(DBL)(x+1),Get_Height(x+1,z,H_Field),(DBL)z);
	Make_Vector(&T1V3,(DBL)x,Get_Height(x,z+1,H_Field),(DBL)(z+1));
	Make_Vector(&T2V1,(DBL)(x+1),Get_Height(x+1,z+1,H_Field),(DBL)(z+1));
	Make_Vector(&T2V2,(DBL)x,Get_Height(x,z+1,H_Field),(DBL)(z+1));
	Make_Vector(&T2V3,(DBL)(x+1),Get_Height(x+1,z,H_Field),(DBL)z);

   if((max(T1V1.y,max(T1V2.y,T1V3.y)) >= height1) && (min(T1V1.y,min(T1V2.y,T1V3.y)) <= height2))
   {
        VSub(T1V3,T1V3,T1V1);
	VSub(T1V2,T1V2,T1V1);
	VCross(Local_Normal,T1V3,T1V2);
	VDot(dot,Local_Normal,Ray->Direction);

	if((dot > Small_Tolerance) || (dot < -Small_Tolerance))
	{
		VDot(pos1,Local_Normal,T1V1);

		VDot(pos2,Local_Normal,Ray->Initial);
		pos2 -= pos1;

		depth1 = -pos2 / dot;

		if((depth1 > Small_Tolerance) && (depth1 < Max_Distance))
		{
			s = Ray->Initial.x+(depth1*Ray->Direction.x)-(DBL)x;
			t = Ray->Initial.z+(depth1*Ray->Direction.z)-(DBL)z;

			if((s<0.0) || (t<0.0) || ((s+t)>1.0))
				depth1 = Max_Distance;
		}
		else
			depth1 = Max_Distance;
	}
   }

   if((max(T2V1.y,max(T2V2.y,T2V3.y)) >= height1) && (min(T2V1.y,min(T2V2.y,T2V3.y)) <= height2))
   {
        VSub(T2V3,T2V3,T2V1);
	VSub(T2V2,T2V2,T2V1);
	VCross(Local_Normal,T2V3,T2V2);
	VDot(dot,Local_Normal,Ray->Direction);

	if((dot > Small_Tolerance) || (dot < -Small_Tolerance))
	{
		VDot(pos1,Local_Normal,T2V1);

		VDot(pos2,Local_Normal,Ray->Initial);
		pos2 -= pos1;

		depth2 = -pos2 / dot;

		if((depth2 > Small_Tolerance) && (depth2 < Max_Distance))
		{
			s = Ray->Initial.x+(depth2*Ray->Direction.x)-(DBL)x;
			t = Ray->Initial.z+(depth2*Ray->Direction.z)-(DBL)z;

			if((s>1.0) || (t>1.0) || ((s+t)<1.0))
				depth2 = Max_Distance;
		}
		else
			depth2 = Max_Distance;
	}
   }

	if((depth1 >= Max_Distance) && (depth2 >= Max_Distance))
		return(FALSE);

	if(depth2 < depth1)
		*Depth = depth2;
	else
		*Depth = depth1;

	return(TRUE);
}

int All_HeightFld_Intersections (Object, Ray, Depth_Queue)
     OBJECT *Object;
     RAY *Ray;
     PRIOQ *Depth_Queue;
{
	HEIGHT_FIELD *Shape = (HEIGHT_FIELD *) Object;
	DBL Depth;
	VECTOR Intersection_Point;
	INTERSECTION Local_Element;

	if (Intersect_HeightFld(Ray, Shape, &Depth))
	{
		Local_Element.Depth = Depth;
		Local_Element.Object = Shape -> Parent_Object;
		VScale (Intersection_Point, Ray->Direction, Depth);
		VAdd (Intersection_Point, Intersection_Point, Ray->Initial);
		Local_Element.Point = Intersection_Point;
		Local_Element.Shape = (SHAPE *)Shape;
		pq_add (Depth_Queue, &Local_Element);
		return (TRUE);
	}
	return (FALSE);
}

int Intersect_HeightFld(Ray, H_Field, Depth)
     RAY *Ray;
     HEIGHT_FIELD *H_Field;
     DBL *Depth;
{
	DBL ymin, ymax, sdy, dx, dz, depth1, depth2, length;
	VECTOR Temp1, Temp2, New_Ray_Direction;
	RAY Temp_Ray;
	int i,steps;
	int px,pz,sdx,sdz;
	long x,z,dxabs,dzabs;

	Ray_Ht_Field_Tests++;
	Temp_Ray = *Ray;

	VAdd(New_Ray_Direction,Temp_Ray.Initial,Temp_Ray.Direction);

	MInverseTransformVector(&Temp_Ray.Initial,&Temp_Ray.Initial,H_Field->Transformation);
	MInverseTransformVector(&New_Ray_Direction,&New_Ray_Direction,H_Field->Transformation);

	VSub(Temp_Ray.Direction,New_Ray_Direction,Temp_Ray.Initial);
	VLength(length,Temp_Ray.Direction);
	VScale(Temp_Ray.Direction,Temp_Ray.Direction,1.0/length);

	if(!Intersect_Box(H_Field,&Temp_Ray,&depth1,&depth2))
		return(FALSE);

	if(depth1>depth2)
	      {
	      VScale(Temp1,Temp_Ray.Direction,depth2);
	      VAdd(Temp1,Temp1,Temp_Ray.Initial);
	      VScale(Temp2,Temp_Ray.Direction,depth1);
	      VAdd(Temp2,Temp2,Temp_Ray.Initial);
	      }
	else
	      {
	      VScale(Temp2,Temp_Ray.Direction,depth2);
	      VAdd(Temp2,Temp2,Temp_Ray.Initial);
	      VScale(Temp1,Temp_Ray.Direction,depth1);
	      VAdd(Temp1,Temp1,Temp_Ray.Initial);
	      }

	px = (int)Temp1.x;
	pz = (int)Temp1.z;
	dx = Temp2.x - Temp1.x;
	dz = Temp2.z - Temp1.z;
	sdx = sign(dx);
	sdz = sign(dz);
	dxabs = labs((long)(dx*10000.0));
	dzabs = labs((long)(dz*10000.0));
	steps = max(abs((int)dx),abs((int)dz));

	if(dx >= 0.0)
	  x = (long)((DBL)max(dxabs,dzabs) * (Temp1.x - (DBL)px));
	else
	  x = (long)((DBL)max(dxabs,dzabs) * ((DBL)(px + 1) - Temp1.x));
	if(dz >= 0.0)
	  z = (long)((DBL)max(dxabs,dzabs) * (Temp1.z - (DBL)pz));
	else
	  z = (long)((DBL)max(dxabs,dzabs) * ((DBL)(pz + 1) - Temp1.z));

	if(dxabs >= dzabs)
	   {
	   sdy = Temp_Ray.Direction.y / Temp_Ray.Direction.x;
	   ymin = Temp1.y + min(sdy, 0.0 - sdy);
	   ymax = Temp1.y + max(sdy, 0.0 - sdy);
	   for(i=0; i<=steps; i++)
	      {
	      if(Intersect_Pixel(px,pz,&Temp_Ray,H_Field,ymin,ymax,Depth))
		 {
		 *Depth /= length;
		 Ray_Ht_Field_Tests_Succeeded++;
		 return(TRUE);
		 }
	      z += dzabs;
	      if(z >= dxabs)
		 {
		 z -= dxabs;
		 if(Intersect_Pixel(px,pz+sdz,&Temp_Ray,H_Field,ymin,ymax,Depth))
		    {
		    *Depth /= length;
		    Ray_Ht_Field_Tests_Succeeded++;
		    return(TRUE);
		    }
		 if(Intersect_Pixel(px+sdx,pz,&Temp_Ray,H_Field,ymin,ymax,Depth))
		    {
		    *Depth /= length;
		    Ray_Ht_Field_Tests_Succeeded++;
		    return(TRUE);
		    }
		 pz += sdz;
		 }
	      px += sdx;
	      ymin +=sdy;
	      ymax +=sdy;
	      }
	   }
	else
	   {
	   sdy = Temp_Ray.Direction.y / Temp_Ray.Direction.z;
	   ymin = Temp1.y + min(sdy, 0.0 - sdy);
	   ymax = Temp1.y + max(sdy, 0.0 - sdy);
	   for(i=0; i<=steps; i++)
	      {
	      if(Intersect_Pixel(px,pz,&Temp_Ray,H_Field,ymin,ymax,Depth))
		 {
		 *Depth /= length;
		 Ray_Ht_Field_Tests_Succeeded++;
		 return(TRUE);
		 }
	      x += dxabs;
	      if (x>=dzabs)
		 {
		 x -= dzabs;
		 if(Intersect_Pixel(px+sdx,pz,&Temp_Ray,H_Field,ymin,ymax,Depth))
		    {
		    *Depth /= length;
		    Ray_Ht_Field_Tests_Succeeded++;
		    return(TRUE);
		    }
		 if(Intersect_Pixel(px,pz+sdz,&Temp_Ray,H_Field,ymin,ymax,Depth))
		    {
		    *Depth /= length;
		    Ray_Ht_Field_Tests_Succeeded++;
		    return(TRUE);
		    }
		 px += sdx;
		 }
	      pz += sdz;
	      ymin += sdy;
	      ymax += sdy;
	      }
	   }
	return(FALSE);
	}

int Inside_HeightFld (Test_Point, Object)
   VECTOR *Test_Point;
   OBJECT *Object;
   {
   return (FALSE);
   }

void HeightFld_Normal (Result, Object, Intersection_Point)
   OBJECT *Object;
   VECTOR *Result, *Intersection_Point;
   {
   HEIGHT_FIELD *H_Field = (HEIGHT_FIELD *) Object;
   int px,pz;
   DBL x,z;
   VECTOR Local_Origin, Temp1, Temp2;

   MInverseTransformVector(&Local_Origin, Intersection_Point, H_Field->Transformation);

   px = (int)Local_Origin.x;
   pz = (int)Local_Origin.z;
   x = Local_Origin.x - (DBL)px;
   z = Local_Origin.z - (DBL)pz;

   if((x+z)<=1)
   {
	Local_Origin.x = (DBL)px;
	Local_Origin.z = (DBL)pz;
	Local_Origin.y = Get_Height(px,pz,H_Field);
	Temp1.x = (DBL)(px + 1);
	Temp1.z = (DBL)pz;
	Temp1.y = Get_Height(px+1,pz,H_Field);
	Temp2.x = (DBL)px;
	Temp2.z = (DBL)(pz + 1);
	Temp2.y = Get_Height(px,pz+1,H_Field);
   }
   else
   {
   	Local_Origin.x = (DBL)(px+1);
	Local_Origin.z = (DBL)(pz+1);
	Local_Origin.y = Get_Height(px+1,pz+1,H_Field);
	Temp1.x = (DBL)px;
	Temp1.z = (DBL)(pz + 1);
	Temp1.y = Get_Height(px,pz+1,H_Field);
	Temp2.x = (DBL)(px + 1);
	Temp2.z = (DBL)pz;
	Temp2.y = Get_Height(px+1,pz,H_Field);
   }

   MTransformVector(&Local_Origin,&Local_Origin,H_Field->Transformation);
   MTransformVector(&Temp1,&Temp1,H_Field->Transformation);
   MTransformVector(&Temp2,&Temp2,H_Field->Transformation);
   VSub(Temp1,Temp1,Local_Origin);
   VSub(Temp2,Temp2,Local_Origin);
   VCross(*Result, Temp2, Temp1);
   VNormalize(*Result,*Result);
   return;
   }

void *Copy_HeightFld (Object)
   OBJECT *Object;
   {
   HEIGHT_FIELD *New_Shape;

   New_Shape = Get_Height_Field_Shape();
   *New_Shape = * ((HEIGHT_FIELD *)Object);
   New_Shape -> Next_Object = NULL;

   if (New_Shape->Shape_Texture != NULL)
      New_Shape->Shape_Texture = Copy_Texture (New_Shape->Shape_Texture);

   return (New_Shape);
   }

void Translate_HeightFld (Object, Vector)
   OBJECT *Object;
   VECTOR *Vector;
   {
   HEIGHT_FIELD *H_Field = (HEIGHT_FIELD *) Object;
   TRANSFORMATION Transformation;

   if (! H_Field -> Transformation)
      H_Field -> Transformation = Get_Transformation ();
   Get_Translation_Transformation(&Transformation,Vector);
   Compose_Transformations(H_Field->Transformation,&Transformation);

   Translate_Texture (&((HEIGHT_FIELD *) Object)->Shape_Texture, Vector);
}

void Rotate_HeightFld (Object, Vector)
   OBJECT *Object;
   VECTOR *Vector;
   {
   TRANSFORMATION Transformation;
   HEIGHT_FIELD *H_Field = (HEIGHT_FIELD *)Object;

   if (! H_Field -> Transformation)
      H_Field -> Transformation = Get_Transformation ();
   Get_Rotation_Transformation(&Transformation,Vector);
   Compose_Transformations(H_Field->Transformation,&Transformation);

   Rotate_Texture (&((HEIGHT_FIELD *) Object)->Shape_Texture, Vector);
   }

void Scale_HeightFld (Object, Vector)
   OBJECT *Object;
   VECTOR *Vector;
   {
   HEIGHT_FIELD *H_Field = (HEIGHT_FIELD *) Object;
   TRANSFORMATION Transformation;

   if (! H_Field -> Transformation)
      H_Field -> Transformation = Get_Transformation ();
   Get_Scaling_Transformation(&Transformation,Vector);
   Compose_Transformations(H_Field->Transformation,&Transformation);

   Scale_Texture (&((HEIGHT_FIELD *) Object)->Shape_Texture, Vector);
   }

void Invert_HeightFld (Object)
   OBJECT *Object;
   {
   HEIGHT_FIELD *H_Field = (HEIGHT_FIELD *) Object;

   }
