/****************************************************************************
*                pvray.c
*
*  This module contains the entry routine for the raytracer and the code to
*  parse the parameters on the command line.
*
*  from Persistence of Vision Raytracer 
*       Copyright 1991 Persistence of Vision Team
*---------------------------------------------------------------------------
*                       *IMPORTANT!*
*  This copyrighted software is freely distributable. The source and/or
* object code may be copied or uploaded to communications services so long as
* this notice remains at the top of each file.
* 
*  If any changes are made to the program, you must clearly indicate in the
* documentation and in the program startup message who it was who made the
* changes. The documentation should also describe what those changes were.
* 
*  This software may not be included in whole or in part into any commercial
* package without the express written consent of the PV-Team. It may,
* however, be included in other freely distributed software so long as proper
* credit for the software is given. No more than five dollars U.S. ($5) can
* be charged for the copying of this software and the media it is provided on,
* i.e. a shareware distribution company may only charge five U.S dollars or
* less for providing this software.
* 
*  This software is provided as is without any guarantees or warranty.
* Although the authors have attempted to find and correct any bugs in the
* software, they are not responsible for any damage caused by the use of the
* software. The authors are under no obligation to provide service,
* corrections, or upgrades to this package.
*-----------------------------------------------------------------------------
*  Despite all the legal stuff above, if you have any problems with the
* program the PV-Team would like to hear about them. Also, if you have any
* comments, questions or enhancements, please contact the PV-Team on the
* Compuserve Online Service in the COMART forum message section 16 (!GO
* COMART). The CIS COMART forum is devoted to computer generated artwork like
* raytracing, animation and fractals. For more information regarding the PV
* team see the file PVINF.TXT. For more information on Compuserve call
* (in the U.S.) 1-800-848-8990.
* 
*       Drew Wells
*       PV-Team Leader
*       CIS: 73767,1244
* 
* 
*  This program is based on the popular DKB raytracer version 2.12 written by
* David Buck, a PV-Team member.
*  (David Buck CIS: 70521,1371 Internet: dbuck@ccs.carleton.ca)
* 
*****************************************************************************/

#include <ctype.h>
#include <time.h>          /* BP */
#include "frame.h"		/* common to ALL modules in this program */
#include "pvproto.h"

#define MAX_FILE_NAMES 1
unsigned int Options;
int Quality;
int Case_Sensitive_Flag = CASE_SENSITIVE_DEFAULT;

FILE *bfp;

extern FRAME Frame;

char Input_File_Name[FILE_NAME_LENGTH], Output_File_Name[FILE_NAME_LENGTH];

#define MAX_LIBRARIES 10
char *Library_Paths[MAX_LIBRARIES];
int Library_Path_Index;

FILE_HANDLE *Output_File_Handle;
int File_Buffer_Size;
static int Number_Of_Files;
DBL VTemp;
DBL Antialias_Threshold;
int First_Line, Last_Line;
int Display_Started = FALSE;
int Shadow_Test_Flag = FALSE;

/* Stats kept by the ray tracer: */
long Number_Of_Pixels, Number_Of_Rays, Number_Of_Pixels_Supersampled;
long Ray_Sphere_Tests, Ray_Sphere_Tests_Succeeded;
long Ray_Plane_Tests, Ray_Plane_Tests_Succeeded;
long Ray_Triangle_Tests, Ray_Triangle_Tests_Succeeded;
long Ray_Quadric_Tests, Ray_Quadric_Tests_Succeeded;
long Ray_Poly_Tests, Ray_Poly_Tests_Succeeded;
long Ray_Bicubic_Tests, Ray_Bicubic_Tests_Succeeded;
long Ray_Ht_Field_Tests, Ray_Ht_Field_Tests_Succeeded;
long Bounding_Region_Tests, Bounding_Region_Tests_Succeeded;
long Clipping_Region_Tests, Clipping_Region_Tests_Succeeded;
long Calls_To_Noise, Calls_To_DNoise;
long Shadow_Ray_Tests, Shadow_Rays_Succeeded;
long Reflected_Rays_Traced, Refracted_Rays_Traced;
long Transmitted_Rays_Traced;

char DisplayFormat, OutputFormat, VerboseFormat, PaletteOption, Color_Bits;

DBL    tused;             /* Trace timer variables. - BP */ 
time_t tstart, tstop;



#ifdef NOCMDLINE	/* a main() by any other name... */
#ifdef ALTMAIN
void alt_main()
#else
void main()
#endif
#else
#ifdef ALTMAIN
void alt_main(argc, argv)
#else
void main(argc, argv)
#endif
  int argc;
  char **argv;
#endif			/* ...would be a lot less hassle!! :-) AAC */
  {
  register int i;

  STARTUP_PVRAY
  
  print_credits();
  
  PRINT_OTHER_CREDITS

/* Parse the command line parameters */
#ifndef NOCMDLINE
  if (argc == 1)
     usage();
#endif

  init_vars();

  Output_File_Name[0]='\0';

  Library_Paths[0] = NULL;
  Library_Path_Index = 0;

/* Read the default parameters from pvray.def */
  get_defaults();

#ifndef NOCMDLINE
  for (i = 1 ; i < argc ; i++ )
    if ((*argv[i] == '+') || (*argv[i] == '-'))
      parse_option(argv[i]);
    else
      parse_file_name(argv[i]);
#endif

   if (Last_Line == -1)
      Last_Line = Frame.Screen_Height;

   if (Options & DISKWRITE) {
      switch (OutputFormat) {
         case '\0':
         case 'd':
         case 'D':
                   if ((Output_File_Handle = Get_Dump_File_Handle()) == NULL) {
                      close_all();
                      exit(1);
                      }
                   break;
/*
         case 'i':
         case 'I':
                   if ((Output_File_Handle = Get_Iff_File_Handle()) == NULL) {
                      close_all();
                      exit(1);
                      }
                   break;

*/
         case 'r':
         case 'R':
                   if ((Output_File_Handle = Get_Raw_File_Handle()) == NULL) {
                      close_all();
                      exit(1);
                      }
                   break;

         case 't':
         case 'T':
                   if ((Output_File_Handle = Get_Targa_File_Handle()) == NULL) {
                      close_all();
                      exit(1);
                      }
                   break;

         default:
                   fprintf (stderr, "Unrecognized output file format %c\n", OutputFormat);
                   exit(1);
         }
      if (Output_File_Name[0] == '\0')
         strcpy (Output_File_Name, Default_File_Name (Output_File_Handle));
      }

   Print_Options();

   Initialize_Tokenizer(Input_File_Name);
   fprintf (stderr,"Parsing...");
   Parse (&Frame);
   Terminate_Tokenizer();
   /* fprintf (stderr,"\n"); */

  if (Options & DISPLAY)
    {
    printf ("Displaying...\n");
    display_init(Frame.Screen_Width, Frame.Screen_Height);
    Display_Started = TRUE;
    }

/* Get things ready for ray tracing */
   if (Options & DISKWRITE)
      if (Options & CONTINUE_TRACE) {
         if (Open_File (Output_File_Handle, Output_File_Name,
                 &Frame.Screen_Width, &Frame.Screen_Height, File_Buffer_Size,
                 READ_MODE) != 1) {
            fprintf (stderr, "Error opening output file\n");
            close_all();
            exit(1);
            }

         Initialize_Renderer();
         Read_Rendered_Part();
         }
      else {
         if (Open_File (Output_File_Handle, Output_File_Name,
                 &Frame.Screen_Width, &Frame.Screen_Height, File_Buffer_Size,
                 WRITE_MODE) != 1) {
            fprintf (stderr, "Error opening output file\n");
            close_all();
            exit(1);
            }

         Initialize_Renderer();
         }
  else
     Initialize_Renderer();

  pq_init();
  Initialize_Noise();
  
  START_TIME  /* Store start time for trace. Timer macro in CONFIG.H */
  

/* Ok, go for it - trace the picture */
  if (Options & VERBOSE && VerboseFormat!='1')
      printf ("Rendering...\n");
  else if (Options & VERBOSE && VerboseFormat=='1')
      fprintf (stderr,"PV-Ray rendering %s to %s :\n",Input_File_Name,Output_File_Name);
    
  Start_Tracing ();
  
  if (Options & VERBOSE && VerboseFormat=='1')
      fprintf (stderr,"\n");
  
/* Record the time so well spent... */
  STOP_TIME                  /* Get trace done time. */
  tused = TIME_ELAPSED       /* Calc. elapsed time. Define TIME_ELAPSED as */
                             /* 0 in your specific CONFIG.H if unsupported */
   
/* Clean up and leave */
  display_finished();

  close_all ();
  print_stats();

  FINISH_PVRAY
  }

/* Print out usage error message */

void usage ()
    {
    fprintf (stderr,"\nUsage:");
    fprintf (stderr,"\n   trace  [+/-] Option1 [+/-] Option2");
    fprintf (stderr,"\n");
    fprintf (stderr,"\n Options:");
    fprintf (stderr,"\n    t  = CASE SENSITIVE y = yes(old style) n=no o=opt(default)");
    fprintf (stderr,"\n    dxy = display in format x, using palette option y");
    fprintf (stderr,"\n    vx  = verbose in format x");
    fprintf (stderr,"\n    p  = prompt exit");
    fprintf (stderr,"\n    x  = enable early exit by key hit");
    fprintf (stderr,"\n    fx = write output file in format x");
    fprintf (stderr,"\n         ft - Uncompressed Targa-24  fd - DKB/QRT Dump  fr - 3 Raw Files");
    fprintf (stderr,"\n    a  = perform antialiasing");
    fprintf (stderr,"\n    c  = continue aborted trace");
    fprintf (stderr,"\n    qx = image quality 0=rough, 9=full");
    fprintf (stderr,"\n    lxxx = library path prefix");
    fprintf (stderr,"\n    wxxx = width of the screen");
    fprintf (stderr,"\n    hxxx = height of the screen");
    fprintf (stderr,"\n    sxxx = start at line number xxx");
    fprintf (stderr,"\n    exxx = end at line number xxx");
    fprintf (stderr,"\n    bxxx = Use xxx kilobytes for output file buffer space");
    fprintf (stderr,"\n    ifilename = input file name");
    fprintf (stderr,"\n    ofilename = output file name");
    fprintf (stderr,"\n");
    exit(1);
    }

void init_vars()
  {
  Output_File_Handle = NULL;
  File_Buffer_Size = 0;
  Options = 0;
  Quality = 9;
  Number_Of_Files = 0;
  First_Line = 0;
  Last_Line = -1;
  Color_Bits = 8;

  Number_Of_Pixels = 0L;
  Number_Of_Rays = 0L;
  Number_Of_Pixels_Supersampled = 0L;
  Ray_Ht_Field_Tests = 0L;
  Ray_Ht_Field_Tests_Succeeded = 0L;
  Ray_Bicubic_Tests = 0L;
  Ray_Bicubic_Tests_Succeeded = 0L;
  Ray_Sphere_Tests = 0L;
  Ray_Sphere_Tests_Succeeded = 0L;
  Ray_Plane_Tests = 0L;
  Ray_Plane_Tests_Succeeded = 0L;
  Ray_Triangle_Tests = 0L;
  Ray_Triangle_Tests_Succeeded = 0L;
  Ray_Quadric_Tests = 0L;
  Ray_Quadric_Tests_Succeeded = 0L;
  Ray_Poly_Tests = 0L;
  Ray_Poly_Tests_Succeeded = 0L;
  Bounding_Region_Tests = 0L;
  Bounding_Region_Tests_Succeeded = 0L;
  Clipping_Region_Tests = 0L;
  Clipping_Region_Tests_Succeeded = 0L;
  Calls_To_Noise = 0L;
  Calls_To_DNoise = 0L;
  Shadow_Ray_Tests = 0L;
  Shadow_Rays_Succeeded = 0L;
  Reflected_Rays_Traced = 0L;
  Refracted_Rays_Traced = 0L;
  Transmitted_Rays_Traced = 0L;

  Frame.Screen_Height = 100;
  Frame.Screen_Width = 100;

  Antialias_Threshold = 0.3;
  strcpy (Input_File_Name, "object.dat");
  return;
  }

/* Close all the stuff that has been opened. */
void close_all ()
   {
   if ((Options & DISPLAY) && Display_Started)
     display_close();

   if (Output_File_Handle)
      Close_File (Output_File_Handle);
   }

/* Read the default parameters from pvray.def */
void get_defaults()
  {
  FILE *defaults_file;
  char Option_String[256], *Option_String_Ptr;

  if ((defaults_file = fopen("pvray.def", "r")) != NULL) {
     while (fgets(Option_String, 256, defaults_file) != NULL)
        read_options(Option_String);
     fclose (defaults_file);
     }

  if ((Option_String_Ptr = getenv("PVRAYOPT")) != NULL)
     read_options(Option_String_Ptr);
  }

void read_options (Option_Line)
  char *Option_Line;
  {
  register int c, String_Index, Option_Started;
  short Option_Line_Index = 0;
  char Option_String[80];

  String_Index = 0;
  Option_Started = FALSE;
  while ((c = Option_Line[Option_Line_Index++]) != '\0')
    {
    if (Option_Started)
      if (isspace(c))
        {
        Option_String[String_Index] = '\0';
        parse_option (Option_String);
        Option_Started = FALSE;
	String_Index = 0;
        }
     else
       Option_String[String_Index++] = (char) c;

    else /* Option_Started */
      if ((c == (int) '-') || (c == (int) '+'))
        {
        String_Index = 0;
        Option_String[String_Index++] = (char) c;
        Option_Started = TRUE;
        }
      else
        if (!isspace(c))
          {
          fprintf (stderr, "\nBad default file format.  Offending char: (%c), val: %d.\n", (char) c, c);
          exit (1);
          }
    }

  if (Option_Started)
    {
    Option_String[String_Index] = '\0';
    parse_option (Option_String);
    }
  }

/* parse the command line parameters */
void parse_option (Option_String)
  char *Option_String;
  {
  register int Add_Option;
  unsigned int Option_Number;
  DBL threshold;

  if (*(Option_String++) == '-')
    Add_Option = FALSE;
  else
    Add_Option = TRUE;

  switch (*Option_String)
    {
    case 'B':
    case 'b':  sscanf (&Option_String[1], "%d", &File_Buffer_Size);
               File_Buffer_Size *= 1024;
               if (File_Buffer_Size < BUFSIZ)
                  File_Buffer_Size = BUFSIZ;
               Option_Number = 0;
               break;

    case 'C':
    case 'c':  Option_Number = CONTINUE_TRACE;
               break;

    case 'D':
    case 'd':  Option_Number = DISPLAY;
               DisplayFormat = '0';
               if (Option_String[1] != '\0')
		              DisplayFormat = (char)toupper(Option_String[1]);
	                PaletteOption = '0';
               if (Option_String[2] != '\0')
		              PaletteOption = (char)toupper(Option_String[2]);
	             break;

    case 'V':
    case 'v':  Option_Number = VERBOSE;
               VerboseFormat = (char)toupper(Option_String[1]);
               if (VerboseFormat == '\0')
                   VerboseFormat = '1';
               break;

    case 'W':
    case 'w':  sscanf (&Option_String[1], "%d", &Frame.Screen_Width);
	             Option_Number = 0;
               break;

    case 'H':
    case 'h':  sscanf (&Option_String[1], "%d", &Frame.Screen_Height);
	             Option_Number = 0;
               break;

    case 'F':
    case 'f':  Option_Number = DISKWRITE;
               if (isupper(Option_String[1]))
                  OutputFormat = (char)tolower(Option_String[1]);
               else
                  OutputFormat = Option_String[1];

             /* Default the output format to the default in the config file */
               if (OutputFormat == '\0')
                  OutputFormat = DEFAULT_OUTPUT_FORMAT;
               break;

    case 'P':
    case 'p':  Option_Number = PROMPTEXIT;
               break;

    case 'I':
    case 'i':  strncpy (Input_File_Name, &Option_String[1], FILE_NAME_LENGTH);
	             Option_Number = 0;
               break;

    case 'O':
    case 'o':  strncpy (Output_File_Name, &Option_String[1], FILE_NAME_LENGTH);
	             Option_Number = 0;
               break;

    case 'A':
    case 'a':  Option_Number = ANTIALIAS;
               if (sscanf (&Option_String[1], DBL_FORMAT_STRING, &threshold) != EOF)
                   Antialias_Threshold = threshold;
               break;

    case 'X':
    case 'x':  Option_Number = EXITENABLE;
               break;


    case 'L':
    case 'l':  if (Library_Path_Index >= MAX_LIBRARIES) {
                  fprintf (stderr, "Too many library directories specified\n");
                  exit(1);
                  }
                Library_Paths [Library_Path_Index] = malloc (strlen(Option_String));
	             if (Library_Paths [Library_Path_Index] == NULL) {
		              fprintf (stderr, "Cannot allocate memory for library pathname\n");
		              exit(1);
		              }
               strcpy (Library_Paths [Library_Path_Index], &Option_String[1]);
               Library_Path_Index++;
	             Option_Number = 0;
               break;
    case 'T':
    case 't':  switch (toupper(Option_String[1])){
                  case 'Y':
                    Case_Sensitive_Flag = 0;
                     break;
                  case 'N':
                    Case_Sensitive_Flag = 1;
                     break;
                  case 'O':
                    Case_Sensitive_Flag = 2;
                     break;
                  default:
                    Case_Sensitive_Flag = 2;
                     break;
                }
               break;
              
    case 'S':
    case 's':  sscanf (&Option_String[1], "%d", &First_Line);
	       Option_Number = 0;
               break;

    case 'E':
    case 'e':  sscanf (&Option_String[1], "%d", &Last_Line);
	       Option_Number = 0;
               break;

    case 'Q':
    case 'q':  sscanf (&Option_String[1], "%d", &Quality);
	       Option_Number = 0;
               break;

               /* Turn on debugging print statements. */
    case 'Z':
    case 'z':  Option_Number = DEBUGGING;
               break;

    default:   fprintf (stderr, "\nInvalid option: %s\n\n", --Option_String);
	       Option_Number = 0;
    }

  if (Option_Number != 0)
      if (Add_Option)
           Options |= Option_Number;
      else Options &= ~Option_Number;
  }

void Print_Options()
   {
   int i;
   
   fprintf (stderr,"Options in effect: ");

   if (Options & CONTINUE_TRACE)
      fprintf (stderr,"+c ");
   else
      fprintf (stderr,"-c ");

   if (Options & DISPLAY)
      fprintf (stderr,"+d%c%c ", DisplayFormat, PaletteOption);
   else
      fprintf (stderr,"-d ");

   if (Options & VERBOSE)
      fprintf (stderr,"+v%c ", VerboseFormat);
   else
      fprintf (stderr,"-v ");

   if (Options & DISKWRITE)
      fprintf (stderr,"+f%c ", OutputFormat);
   else
      fprintf (stderr,"-f ");

   if (Options & PROMPTEXIT)
      fprintf (stderr,"+p ");
   else
      fprintf (stderr,"-p ");

   if (Options & EXITENABLE)
      fprintf (stderr,"+x ");
   else
      fprintf (stderr,"-x ");

   if (Options & ANTIALIAS)
      fprintf (stderr,"+a%f ", Antialias_Threshold);
   else
      fprintf (stderr,"-a ");

   if (Options & DEBUGGING)
      fprintf (stderr,"+z ");

   if (File_Buffer_Size != 0)
      fprintf (stderr,"-b%d ", File_Buffer_Size/1024);

   fprintf (stderr,"-q%d -w%d -h%d -s%d -e%d\n-i%s ",
           Quality, Frame.Screen_Width, Frame.Screen_Height,
           First_Line, Last_Line, Input_File_Name);

   if (Options & DISKWRITE)
      fprintf (stderr,"-o%s ", Output_File_Name);

   for (i = 0 ; i < Library_Path_Index ; i++)
      fprintf (stderr,"-l%s ", Library_Paths[i]);

   fprintf (stderr,"\n");
   }

void parse_file_name (File_Name)
  char *File_Name;
  {
  FILE *defaults_file;
  char Option_String[256];

  if (++Number_Of_Files > MAX_FILE_NAMES)
    {
    fprintf (stderr, "\nOnly %d file names are allowed in a command line.", 
             MAX_FILE_NAMES);
    exit(1);
    }

  if ((defaults_file = fopen(File_Name, "r")) != NULL) {
     while (fgets(Option_String, 256, defaults_file) != NULL)
        read_options(Option_String);
     fclose (defaults_file);
     }
  }

void print_stats()
   {
   int hours,min;
   DBL sec;
   
   fprintf (stdout,"%s Statistics\n",Input_File_Name);
   fprintf (stdout,"--------------------------------------\n");
   fprintf (stdout,"# Rays:  %10ld    # Pixels:  %10ld  # Pixels supersampled: %10ld\n",
            Number_Of_Rays, Number_Of_Pixels, Number_Of_Pixels_Supersampled);

   fprintf (stdout," Intersection Tests:\n");
   fprintf (stdout,"   Type       Tests    Succeeded\n");
   fprintf (stdout,"   ----    ----------  ----------\n");
   if(Ray_Sphere_Tests)
   fprintf (stdout,"  Sphere   %10ld  %10ld\n", Ray_Sphere_Tests, Ray_Sphere_Tests_Succeeded);
   if(Ray_Plane_Tests)
   fprintf (stdout,"  Plane    %10ld  %10ld\n", Ray_Plane_Tests, Ray_Plane_Tests_Succeeded);
   if(Ray_Triangle_Tests)
   fprintf (stdout,"  Triangle %10ld  %10ld\n", Ray_Triangle_Tests, Ray_Triangle_Tests_Succeeded);
   if(Ray_Quadric_Tests)
   fprintf (stdout,"  Quadric  %10ld  %10ld\n", Ray_Quadric_Tests, Ray_Quadric_Tests_Succeeded);
   if(Ray_Poly_Tests)
   fprintf (stdout,"  Poly     %10ld  %10ld\n", Ray_Poly_Tests, Ray_Poly_Tests_Succeeded);
   if(Ray_Bicubic_Tests)
   fprintf (stdout,"  Bicubic  %10ld  %10ld\n", Ray_Bicubic_Tests, Ray_Bicubic_Tests_Succeeded);
   if(Ray_Ht_Field_Tests)
   fprintf (stdout,"  Hght_Fld %10ld  %10ld\n", Ray_Ht_Field_Tests, Ray_Ht_Field_Tests_Succeeded);
   if(Bounding_Region_Tests)
   fprintf (stdout,"  Bounds   %10ld  %10ld\n", Bounding_Region_Tests, Bounding_Region_Tests_Succeeded);
   if(Clipping_Region_Tests)
   fprintf (stdout,"  Clips    %10ld  %10ld\n", Clipping_Region_Tests, Clipping_Region_Tests_Succeeded);
   if(Calls_To_Noise)
   fprintf (stdout,"  Calls to Noise:   %10ld\n", Calls_To_Noise);
   if(Calls_To_DNoise)
   fprintf (stdout,"  Calls to DNoise:  %10ld\n", Calls_To_DNoise);
   if(Shadow_Ray_Tests)
   fprintf (stdout,"  Shadow Ray Tests: %10ld     Blocking Objects Found:  %10ld\n",
              Shadow_Ray_Tests, Shadow_Rays_Succeeded);
   if(Reflected_Rays_Traced)
   fprintf (stdout,"  Reflected Rays:   %10ld\n", Reflected_Rays_Traced);
   if(Refracted_Rays_Traced)
   fprintf (stdout,"  Refracted Rays:   %10ld\n", Refracted_Rays_Traced);
   if(Transmitted_Rays_Traced)
   fprintf (stdout,"  Transmitted Rays: %10ld\n", Transmitted_Rays_Traced);
   
   if (tused != 0){
     /* Convert seconds to hours, min & sec. CdW */
     hours = (int) tused/3600;
     min = (int) (tused - hours*3600)/60;
     sec = tused - (DBL) (hours*3600 + min*60);
     fprintf (stdout,"  Time For Trace:   %2d hours %2d minutes %4.2f seconds\n", hours,min,sec); 
     }
   }

/* Find a file in the search path. */

FILE *Locate_File (filename, mode)
   char *filename, *mode;
   {
   FILE *f;
   int i;
   char pathname[FILE_NAME_LENGTH];

   /* Check the current directory first. */
   if ((f = fopen (filename, mode)) != NULL)
      return (f);

   for (i = 0 ; i < Library_Path_Index ; i++) {
      strcpy (pathname, Library_Paths[i]);
      if (FILENAME_SEPARATOR != NULL)
         strcat (pathname, FILENAME_SEPARATOR);
      strcat (pathname, filename);
      if ((f = fopen (pathname, mode)) != NULL)
         return (f);
      }

   return (NULL);
   }
void print_credits()
  {
  fprintf (stderr,"\n");
  fprintf (stderr,"  Persistence of Vision Raytracer\n");
  fprintf (stderr,"  BETA Version 0.5\n");
  fprintf (stderr,"  -------------------------------\n");
  fprintf (stderr,"  Copyright (c) 1991 PV-Team\n");
  fprintf (stderr,"  Contact the PV-Team on Compuserve's COMART forum.\n");
  fprintf (stderr,"  PV-Ray is based on DKBTrace 2.12 written by David K. Buck.\n");
  fprintf (stderr,"  This program is freely distributable subject to restrictions.\n\n");

  fprintf (stderr,"  Contributing Authors: (Alphabetically)\n");
  fprintf (stderr,"  ------------------------------------------------------------\n");
  fprintf (stderr,"  Steve A. Bennett   David K. Buck      Aaron A. Collins\n");
  fprintf (stderr,"  Alexander Enzmann  Girish T. Hagan    Douglas Muir \n");
  fprintf (stderr,"  Bill Pulver        Robert Skinner     Scott Taylor\n");
  fprintf (stderr,"  Drew Wells         (Did I miss anyone?)\n");
  fprintf (stderr," -------------------------------------------------------------\n");
  
  }
