/****************************************************************************
*                txtcolor.c
*
*  This module implements solid texturing functions that modify the color
*  transparency of an object's surface.
*
*  from Persistence of Vision Raytracer 
*  Copyright 1991 Persistence of Vision Team
*---------------------------------------------------------------------------
*                       *IMPORTANT!*
*  This copyrighted software is freely distributable. The source and/or
* object code may be copied or uploaded to communications services so long as
* this notice remains at the top of each file.
* 
*  If any changes are made to the program, you must clearly indicate in the
* documentation and in the program startup message who it was who made the
* changes. The documentation should also describe what those changes were.
* 
*  This software may not be included in whole or in part into any commercial
* package without the express written consent of the PV-Team. It may,
* however, be included in other freely distributed software so long as proper
* credit for the software is given. No more than five dollars U.S. ($5) can
* be charged for the copying of this software and the media it is provided on,
* i.e. a shareware distribution company may only charge five U.S dollars or
* less for providing this software.
* 
*  This software is provided as is without any guarantees or warranty.
* Although the authors have attempted to find and correct any bugs in the
* software, they are not responsible for any damage caused by the use of the
* software. The authors are under no obligation to provide service,
* corrections, or upgrades to this package.
*-----------------------------------------------------------------------------
*  Despite all the legal stuff above, if you have any problems with the
* program the PV-Team would like to hear about them. Also, if you have any
* comments, questions or enhancements, please contact the PV-Team on the
* Compuserve Online Service in the COMART forum message section 16 (!GO
* COMART). The CIS COMART forum is devoted to computer generated artwork like
* raytracing, animation and fractals. For more information regarding the PV
* team see the file PVINF.TXT. For more information on Compuserve call
* (in the U.S.) 1-800-848-8990.
* 
*       Drew Wells
*       PV-Team Leader
*       CIS: 73767,1244
* 
* 
*  This program is based on the popular DKB raytracer version 2.12 written by
* David Buck, a PV-Team member.
*  (David Buck CIS: 70521,1371 Internet: dbuck@ccs.carleton.ca)
* 
*****************************************************************************/

/*
   Some texture ideas garnered from SIGGRAPH '85 Volume 19 Number 3, 
   "An Image Synthesizer" By Ken Perlin.
   Further Ideas Garnered from "The RenderMan Companion" (Addison Wesley).
*/

#include "frame.h"
#include "vector.h"
#include "pvproto.h"
#include "texture.h"


void agate (x, y, z, Texture, colour)
   DBL x, y, z;
   TEXTURE *Texture;
   COLOUR *colour;
   {
   register DBL noise, hue;
   COLOUR New_Colour;

   noise = cycloidal(1.3 * Turbulence(x, y, z) + 1.1 * z) + 1;
   noise *= 0.5;
   noise = pow(noise, 0.77);

   if (Options & DEBUGGING)
      printf ("agate %g %g %g noise %g\n", x, y, z, noise);

   if (Texture -> Colour_Map != NULL)
      {
      Compute_Colour (&New_Colour, Texture->Colour_Map, noise);
      colour -> Red += New_Colour.Red;
      colour -> Green += New_Colour.Green;
      colour -> Blue += New_Colour.Blue;
      colour -> Alpha += New_Colour.Alpha;
      return;
      }

   hue = 1.0 - noise;

   if (noise < 0.5)
      {
      colour -> Red += (1.0 - (noise / 10));
      colour -> Green += (1.0 - (noise / 5));
      colour -> Blue += hue;
      }
   else if (noise < 0.6)
      {
      colour -> Red += 0.9;
      colour -> Green += 0.7;
      colour -> Blue += hue;
      }
   else
      {
      colour -> Red += (0.6 + hue);
      colour -> Green += (0.3 + hue);
      colour -> Blue += hue;
      }
   return;
   }


void bozo (x, y, z, Texture, colour)
DBL x, y, z;
TEXTURE *Texture;
COLOUR *colour;
   {
   register DBL noise, turb;
   COLOUR New_Colour;
   VECTOR BozoTurbulence;


   if (Options & DEBUGGING)
      printf ("bozo %g %g %g ", x, y, z);

   if ((turb = Texture->Turbulence) != 0.0)
      {
      DTurbulence (&BozoTurbulence, x, y, z);
      x += BozoTurbulence.x * turb;
      y += BozoTurbulence.y * turb;
      z += BozoTurbulence.z * turb;
      }

   noise = Noise (x, y, z);

   if (Options & DEBUGGING)
      printf ("noise %g\n", noise);

   if (Texture -> Colour_Map != NULL) {
      Compute_Colour (&New_Colour, Texture->Colour_Map, noise);
      colour -> Red += New_Colour.Red;
      colour -> Green += New_Colour.Green;
      colour -> Blue += New_Colour.Blue;
      colour -> Alpha += New_Colour.Alpha;
      return ;
      }

   if (noise < 0.4) {
      colour -> Red += 1.0;
      colour -> Green += 1.0;
      colour -> Blue += 1.0;
      return ;
      }

   if (noise < 0.6) {
      colour -> Green += 1.0;
      return ;
      }

   if (noise < 0.8) {
      colour -> Blue += 1.0;
      return ;
      }

   colour -> Red += 1.0;
   return ;
   }

void checker (x, y, z, Texture, colour)
   DBL x, y, z;
   TEXTURE *Texture;
   COLOUR *colour;
   {
   int brkindx;

   brkindx = (int)(FLOOR(x) + FLOOR(y) + FLOOR(z));  /* AAC: was just x + z */

   if (Options & DEBUGGING)
      printf ("checker %g %g %g\n", x, y, z);

   if (brkindx & 1)
      *colour = *Texture -> Colour1;
   else
      *colour = *Texture -> Colour2;
   return;
   }



void checker_texture (x, y, z, Texture, colour)
   DBL x, y, z;
   TEXTURE *Texture;
   COLOUR *colour;
   {
   int brkindx;
   VECTOR Point;

   brkindx = (int)(FLOOR(x) + FLOOR(y) + FLOOR(z));

   if (Options & DEBUGGING)
      printf ("checker_texture %g %g %g\n", x, y, z);

   Make_Vector (&Point, x, y, z);

   if (brkindx & 1)
      Colour_At (colour, ((TEXTURE *) Texture -> Colour1), &Point);
   else
      Colour_At (colour, ((TEXTURE *) Texture -> Colour2), &Point);
   return;
   }

/*
   Color Gradient Texture - gradient based on the fractional values of x, y or
   z, based on whether or not the given directional vector is a 1.0 or a 0.0.
   Note - ONLY works with colour maps, preferably one that is circular - i.e.
   the last defined colour (value 1.001) is the same as the first colour (with
   a value of 0.0) in the map.  The basic concept of this is from DBW Render,
   but Dave Wecker's only supports simple Y axis gradients.
*/

void gradient (x, y, z, Texture, colour)
   DBL x, y, z;
   TEXTURE *Texture;
   COLOUR *colour;
   {
   COLOUR New_Colour;
   DBL value = 0.0, turb;
   VECTOR GradTurbulence;

   if ((turb = Texture->Turbulence) != 0.0)
      {
      DTurbulence (&GradTurbulence, x, y, z);
      x += GradTurbulence.x * turb;
      y += GradTurbulence.y * turb;
      z += GradTurbulence.z * turb;
      }

   if (Texture -> Colour_Map == NULL)
      return;
   if (Texture -> Texture_Gradient.x != 0.0)
      {
      x = FABS(x);
      value += x - FLOOR(x);	/* obtain fractional X component */
      }
   if (Texture -> Texture_Gradient.y != 0.0)
      {
      y = FABS(y);
      value += y - FLOOR(y);	/* obtain fractional Y component */
   }
   if (Texture -> Texture_Gradient.z != 0.0)
      {
      z = FABS(z);
      value += z - FLOOR(z);	/* obtain fractional Z component */
      }
   value = ((value > 1.0) ? fmod(value, 1.0) : value); /* clamp to 1.0 */

   if (Options & DEBUGGING)
      printf ("gradient %g %g %g value %g\n", x, y, z, value);

   Compute_Colour(&New_Colour, Texture->Colour_Map, value);
   colour -> Red += New_Colour.Red;
   colour -> Green += New_Colour.Green;
   colour -> Blue += New_Colour.Blue;
   colour -> Alpha += New_Colour.Alpha;
   return;
   }


/*
   Granite - kind of a union of the "spotted" and the "dented" textures,
   using a 1/f fractal noise function for color values.  Typically used
   w/ small scaling values.  Should work with colour maps for pink granite...
*/


void granite (x, y, z, Texture, colour)
   DBL x, y, z;
   TEXTURE *Texture;
   COLOUR *colour;
   {
   register int i;
   register DBL temp, noise = 0.0, freq = 1.0;
   COLOUR New_Colour;

   for (i = 0; i < 6 ; freq *= 2.0, i++)
      {
      temp = 0.5 - Noise (x * 4 * freq, y * 4 * freq, z * 4 * freq);
      temp = FABS(temp);
      noise += temp / freq;
      }

   if (Options & DEBUGGING)
      printf ("granite %g %g %g noise %g\n", x, y, z, noise);

   if (Texture -> Colour_Map != NULL)
      {
      Compute_Colour (&New_Colour, Texture->Colour_Map, noise);
      colour -> Red += New_Colour.Red;
      colour -> Green += New_Colour.Green;
      colour -> Blue += New_Colour.Blue;
      colour -> Alpha += New_Colour.Alpha;
      return;
      }

   colour -> Red += noise;                  /* "white (1.0) * noise" */
   colour -> Green += noise;
   colour -> Blue += noise;

   return;
   }


void marble (x, y, z, Texture, colour)
   DBL x, y, z;
   TEXTURE *Texture;
   COLOUR *colour;
   {
   register DBL noise, hue;
   COLOUR New_Colour;

   noise = Triangle_Wave(x + Turbulence(x, y, z) * Texture -> Turbulence);

   if (Options & DEBUGGING)
      printf ("marble %g %g %g noise %g \n", x, y, z, noise);

   if (Texture -> Colour_Map != NULL)
      {
      Compute_Colour (&New_Colour, Texture->Colour_Map, noise);
      colour -> Red += New_Colour.Red;
      colour -> Green += New_Colour.Green;
      colour -> Blue += New_Colour.Blue;
      colour -> Alpha += New_Colour.Alpha;
      return ;
      }

   if (noise < 0.0)
      {
      colour -> Red += 0.9;
      colour -> Green += 0.8;
      colour -> Blue += 0.8;
      }
   else if (noise < 0.9)
      {
      colour -> Red += 0.9;
      hue = 0.8 - noise * 0.8;
      colour -> Green += hue;
      colour -> Blue += hue;
      }
   return ;
   }

/*	
   With a little reflectivity and brilliance, can look like organ pipe
   metal.   With tiny scaling values can look like masonry or concrete.
   Works with color maps.
*/

void spotted (x, y, z, Texture, colour)
   DBL x, y, z;
   TEXTURE *Texture;
   COLOUR *colour;
   {
   register DBL noise;
   COLOUR New_Colour;

   noise = Noise (x, y, z);

   if (Options & DEBUGGING)
      printf ("spotted %g %g %g\n", x, y, z);

   if (Texture -> Colour_Map != NULL)
      {
      Compute_Colour (&New_Colour, Texture->Colour_Map, noise);
      colour -> Red += New_Colour.Red;
      colour -> Green += New_Colour.Green;
      colour -> Blue += New_Colour.Blue;
      colour -> Alpha += New_Colour.Alpha;
      return;
      }

   colour -> Red += noise;             /* "white (1.0) * noise" */
   colour -> Green += noise;
   colour -> Blue += noise;

   return;
   }

   



void wood (x, y, z, Texture, colour)
   DBL x, y, z;
   TEXTURE *Texture;
   COLOUR *colour;
   {
   register DBL noise, length;
   VECTOR WoodTurbulence;
   VECTOR point;
   COLOUR New_Colour;

   DTurbulence (&WoodTurbulence, x, y, z);

   if (Options & DEBUGGING)
      printf ("wood %g %g %g", x, y, z);

   point.x = cycloidal((x + WoodTurbulence.x)
               * Texture -> Turbulence);
   point.y = cycloidal((y + WoodTurbulence.y)
               * Texture -> Turbulence);
   point.z = 0.0;

   point.x += x;
   point.y += y;
   
   /*  point.z += z;       Deleted per David Buck --  BP 7/91 */

   VLength (length, point);

   noise = Triangle_Wave(length);

   if (Options & DEBUGGING)
      printf ("noise %g\n", noise);

   if (Texture -> Colour_Map != NULL) {
      Compute_Colour (&New_Colour, Texture->Colour_Map, noise);
      colour -> Red += New_Colour.Red;
      colour -> Green += New_Colour.Green;
      colour -> Blue += New_Colour.Blue;
      colour -> Alpha += New_Colour.Alpha;
      return ;
      }

   if (noise > 0.6) {
      colour -> Red += 0.4;
      colour -> Green += 0.133;
      colour -> Blue += 0.066;
      }
  else {
      colour -> Red += 0.666;
      colour -> Green += 0.312;
      colour -> Blue += 0.2;
      }
      return ;
   }

/* Two new textures by Scott Taylor LEOPARD & ONION */

void leopard (x, y, z, Texture, colour)      /* SWT 7/18/91 */
DBL x, y, z;
TEXTURE *Texture;
COLOUR *colour;
   {
   /* The variable noise is not used as noise in this function */
   register DBL noise, turb;
   COLOUR New_Colour;
   VECTOR LeopardTurbulence;


   if (Options & DEBUGGING)
      printf ("leopard %g %g %g ", x, y, z);

   if ((turb = Texture->Turbulence) != 0.0)
      {
      DTurbulence (&LeopardTurbulence, x, y, z);
      x += LeopardTurbulence.x * turb;
      y += LeopardTurbulence.y * turb;
      z += LeopardTurbulence.z * turb;
      }

   noise = Sqr((sin(x)+sin(y)+sin(z))/3);

   if (Options & DEBUGGING)
      printf ("noise %g\n", noise);

   if (Texture -> Colour_Map != NULL) {
      Compute_Colour (&New_Colour, Texture->Colour_Map, noise);
      colour -> Red += New_Colour.Red;
      colour -> Green += New_Colour.Green;
      colour -> Blue += New_Colour.Blue;
      colour -> Alpha += New_Colour.Alpha;
      return;
      }

   colour -> Red += noise;
   colour -> Green += noise;
   colour -> Blue += noise;
   return;
   }

void onion (x, y, z, Texture, colour)      /* SWT 7/18/91 */
DBL x, y, z;
TEXTURE *Texture;
COLOUR *colour;
   {
   /* The variable noise is not used as noise in this function */
   register DBL noise, turb;
   COLOUR New_Colour;
   VECTOR OnionTurbulence;


   if (Options & DEBUGGING)
      printf ("onion %g %g %g ", x, y, z);

   if ((turb = Texture->Turbulence) != 0.0)
      {
      DTurbulence (&OnionTurbulence, x, y, z);
      x += OnionTurbulence.x * turb;
      y += OnionTurbulence.y * turb;
      z += OnionTurbulence.z * turb;
      }

   /* This ramp goes 0-1,1-0,0-1,1-0...
   noise = (fmod(SQRT(Sqr(x)+Sqr(y)+Sqr(z)),2.0)-1.0);
   if (noise<0.0) {noise = 0.0-noise;}
   */

   /* This ramp goes 0-1,0-1,0-1,0-1... */
   noise = (fmod(SQRT(Sqr(x)+Sqr(y)+Sqr(z)),1.0));

   if (Options & DEBUGGING)
      printf ("noise %g\n", noise);

   if (Texture -> Colour_Map != NULL) {
      Compute_Colour (&New_Colour, Texture->Colour_Map, noise);
      colour -> Red += New_Colour.Red;
      colour -> Green += New_Colour.Green;
      colour -> Blue += New_Colour.Blue;
      colour -> Alpha += New_Colour.Alpha;
      return;
      }

   colour -> Red += noise;
   colour -> Green += noise;
   colour -> Blue += noise;
   return;
   }
