/****************************************************************************
*                   ibm.c
*
*  This module implements the IBM-PC specific routines for PV-Ray.
*
*  from Persistence of Vision Raytracer 
*  Copyright 1991 Persistence of Vision Team
*---------------------------------------------------------------------------
*                       *IMPORTANT!*
*  This copyrighted software is freely distributable. The source and/or
* object code may be copied or uploaded to communications services so long as
* this notice remains at the top of each file.
* 
*  If any changes are made to the program, you must clearly indicate in the
* documentation and in the program startup message who it was who made the
* changes. The documentation should also describe what those changes were.
* 
*  This software may not be included in whole or in part into any commercial
* package without the express written consent of the PV-Team. It may,
* however, be included in other freely distributed software so long as proper
* credit for the software is given. No more than five dollars U.S. ($5) can
* be charged for the copying of this software and the media it is provided on,
* i.e. a shareware distribution company may only charge five U.S dollars or
* less for providing this software.
* 
*  This software is provided as is without any guarantees or warranty.
* Although the authors have attempted to find and correct any bugs in the
* software, they are not responsible for any damage caused by the use of the
* software. The authors are under no obligation to provide service,
* corrections, or upgrades to this package.
*-----------------------------------------------------------------------------
*  Despite all the legal stuff above, if you have any problems with the
* program the PV-Team would like to hear about them. Also, if you have any
* comments, questions or enhancements, please contact the PV-Team on the
* Compuserve Online Service in the COMART forum message section 16 (!GO
* COMART). The CIS COMART forum is devoted to computer generated artwork like
* raytracing, animation and fractals. For more information regarding the PV
* team see the file PVINF.TXT. For more information on Compuserve call
* (in the U.S.) 1-800-848-8990.
* 
*       Drew Wells
*       PV-Team Leader
*       CIS: 73767,1244
* 
* 
*  This program is based on the popular DKB raytracer version 2.12 written by
* David Buck, a PV-Team member.
*  (David Buck CIS: 70521,1371 Internet: dbuck@ccs.carleton.ca)
* 
*****************************************************************************/

/* Original IBM VGA "colour" output routines for MS/DOS by Aaron A. Collins.

   This will deliver approximate colorings using HSV values for the selection.
   The palette map is divided into 4 parts - upper and lower half generated
   with full and half "value" (intensity), respectively.  These halves are
   further halved by full and half saturation values of each range (pastels).
   There are three constant colors, black, white, and grey.  They are used
   when the saturation is low enough that the hue becomes undefined, and which
   one is selected is based on a simple range map of "value".  Usage of the
   palette is accomplished by converting the requested color RGB into an HSV
   value.  If the saturation is too low (< .25) then black, white or grey is
   selected.  If there is enough saturation to consider looking at the hue,
   then the hue range of 1-63 is scaled into one of the 4 palette quadrants
   based on its "value" and "saturation" characteristics.

   Further SVGA, MVGA mods by Aaron A. Collins:
   SVGA, MVGA assembler routines originally by John Bridges.
   VESA assembler routines from FRACTINT, by The Stone Soup Group
   AT&T VDC600 SVGA mods to DKB Trace 2.01 module IBM.C by John Gooding

   This file now represents the collective wisdom of the VGAKIT34 package,
   with support for all of the SVGA types known to mankind.  Seriously,
   VGAKIT34 is an excellent technical reference for IBM programmers wishing
   to do any sort of SVGA video access, and it encompasses nearly all of the
   SVGA adapters in use today.  It was written by John Bridges, a.k.a.
   CIS:73307,606, GENIE:J.BRIDGES.  It was originally coded in IBM 80x86
   assembler, and since DKBTrace is a completely "C"-based package, I have
   converted John's assembler routines all into "C".  These may be a tad bit
   slower, but they should be compatible across a wide variety of 80x86/(S)VGA
   machines.  Note if you have a regular cheapo VGA card like myself, included
   is "MODE13x" or MVGA (modified VGA) mode (some call it "tweaked", but I
   call it "Simulated SVGA"), which gives 360x480 on any reasonably register-
   compatible plain vanilla VGA card.  This mode gives a good simulated 640 by
   480 screen resolution.  I did not implement all the neat hi-res modes of
   all the various SVGA adapters, if you select a trace size bigger than the
   program and/or card can handle (most likely 640x480), it is dynamically
   scaled to fit the available resolution, so you'll be able to see a rough
   approximation of an 800x600 trace even on any el-cheapo VGA card at 320x200
   resolution.  The VESA VGA mode was freely adapted from FRACTINT, whose GIF
   reading routines we are already using in DKBTrace.  I hope my conversion
   of it works properly.

   There is still a reported problem with the EVEREX autodetect returning
   TRIDENT.  In fact EVEREX uses a TRIDENT chip set, but apparently there
   is some difference in operation.  There are cryptic diagnostic messages
   such as T0000, etc. printed as a result of the autodetection routines
   to help track down why the error is happening.  If you are experiencing
   problems with EVEREX or TRIDENT, make note of the letter-4 digit code you
   are given.  There is now an autodetect for VDC600 that I hope will work
   universally.  A similar problem as the EVEREX exists, in that the VDC600
   is detected as a PARADISE because it uses the PARADISE chip set.  I am now
   looking for what I believe to be the model number in the BIOS ROM of the
   VDC600 to differentiate between the two.  I hope this works  with all
   VDC600's, as I only had one example to work from.  Please send all bug
   reports to Aaron Collins at the "You Can Call Me RAY" BBS, the number is
   in the authors.txt document.


   PV-Ray Enhancements:

   (S)VGA B&W 64 Greyscale code courtesy of Grant Hess 6/21/91

   16/24-bit HGWS/TIGA code courtesy of Jay S. Curtis 11/15/90

   VGA "332" palette routines courtesy of Alexander Enzmann 7/20/91

   Additional support for VESA and Tseng Labs 256 and 32768 color modes by
   Charles Marslett  (CIS: 73317,3662).  Resolution is selected to be the
   lowest possible and still fit the image entirely on the screen. 4/28/91

*/

#include <dos.h>	/* MS-DOS specific - for int86() REGS struct, etc. */
#include <time.h>
#include "frame.h"
#include "pvproto.h"
#include "tiga.h"	/* HGWS/TIGA function library include file */
#include "extend.h"	/* HGWS/TIGA function library include file */

#ifdef TURBOC
extern unsigned _stklen = 12288; /* fairly large stack for HEAVY recursion */
#define CLD asm cld
#define _asm asm
#else
#ifdef __WATCOMC__
#define CLD clear_direction()
void clear_direction(void);
#pragma aux clear_direction = 0xFC parm [] modify nomemory;
#else
#define CLD _asm cld
#endif
#endif

/* The supported VGA adapter types 1 - 9, A - G.  0 is for auto-detect. */

#define	BASIC_VGA	1		/* 1 - Tested: AAC */
#define	MODE13x		2		/* 2 - Tested: AAC */
#define	TSENG3		3		/* 3 - Tested: William Minus */
#define	TSENG4		4		/* 4 - Tested: William Minus */
#define	VDC600		5		/* 5 - Tested: John Gooding */
#define	OAKTECH		6		/* 6 - Untested */
#define	VIDEO7		7		/* 7 - Untested */
#define CIRRUS		8		/* 8 - Tested: AAC */
#define	PARADISE	9		/* 9 - Tested: John Degner */
#define	AHEADA		17		/* A - Untested */
#define	AHEADB		18		/* B - Untested */
#define	CHIPSTECH	19		/* C - Untested */
#define	ATIVGA		20		/* D - Tested: William Earl */
#define	EVEREX		21		/* E - Tested: A+B problem - Larry Minton */
#define	TRIDENT		22		/* F - Tested: A problem - Alexander Enzmann */
#define VESA		23		/* G - Tested: Charles Marslett/AAC */

#define TRUECOLOR	24		/* this # or higher isn't an (S)VGA */
#define TIGA_16		24		/* O - Untested */
#define TIGA_24		25		/* P - Untested */


#define	MISCOUT		0x3c2		/* VGA chip msic output reg. addr */
#define	SEQUENCER	0x3c4		/* VGA chip sequencer register addr */
#define	CRTC		0x3d4		/* VGA chip crt controller reg addr */


char *vga_names[] = 
    {
    "",
    "Standard VGA",
    "Simulated SVGA",
    "Tseng Labs 3000 SVGA",
    "Tseng Labs 4000 SVGA",
    "AT&T VDC600 SVGA",
    "Oak Technologies SVGA",
    "Video 7 SVGA",
    "Video 7 Vega (Cirrus) VGA",
    "Paradise SVGA",
    "",
    "",
    "",
    "",	/* reserved */
    "",
    "",
    "",
    "Ahead Systems Ver. A SVGA",
    "Ahead Systems Ver. B SVGA",
    "Chips & Technologies SVGA",
    "ATI SVGA",
    "Everex SVGA",
    "Trident SVGA",
    "VESA Standard SVGA",
    "",
    "",
    "",
    "", /* spare SVGAs */
    "",
    "",
    "",
    "Hercules GWS/TIGA 16-bit",	/* here on is reserved for non-SVGA cards */
    "Hercules GWS/TIGA 24-bit"
    };


unsigned int vptbl[] =		/* CRTC register values for MODE13x */
    {
    0x6b00,	/* horz total */
    0x5901,	/* horz displayed */
    0x5a02,	/* start horz blanking */
    0x8e03U,	/* end horz blanking */
    0x5e04,	/* start h sync */
    0x8a05U,	/* end h sync */
    0x0d06,	/* vertical total */
    0x3e07,	/* overflow */
    0x4009,	/* cell height */
    0xea10U,	/* v sync start */
    0xac11U,	/* v sync end and protect cr0-cr7 */
    0xdf12U,	/* vertical displayed */
    0x2d13,	/* offset */
    0x0014,	/* turn off dword mode */
    0xe715U,	/* v blank start */
    0x0616,	/* v blank end */
    0xe317U	/* turn on byte mode */
};


int screen_height, screen_width;
int svga_width = 640;		/* default SVGA width/height, */
int svga_height = 480;		/* unless we find otherwise... */
int lastx, lasty, lastline;	/* Pixel / Line Caches */
int whichvga = BASIC_VGA;	/* BASIC_VGA mode by default */
int vga_512K = FALSE;		/* Flag for whether or not >= 512K VGA mem */
unsigned char cur_page = 255;	/* SVGA current page (bank) number */
unsigned int map_code = 0;	/* default map code is 0 */
unsigned long gran = 65536L;	/* SVGA granule size (64K by default) */
unsigned char vesa_granularity = 1;	/* VESA default to 64K granules */
/* void (_cdecl *vesa_bankswitch)(void); * ptr to VESA bankswitch function */
unsigned char answer[257];		/* answer area for VESA BIOS calls */
int vesamode;				/* cur. VESA BIOS supported mode */
int dither_matrix[4][4]={	{0,8,2,10},	/* HGWS/TIGA dither matrix */
				{12,4,14,6},	/* this may not be needed? */
				{3,11,1,9},
				{15,7,13,5}
			};
char hpt[3];			/* HWGS/TIGA information passing area */
char _far *hptr=hpt;		/* HWGS/TIGA information passing area ptr */


extern unsigned int Options;
extern char DisplayFormat, PaletteOption, Color_Bits;


int auto_detect_vga(void);
int cirrus(void);
int chkbank(unsigned int, unsigned int);
void palette_init(void);
void newbank(void);
void set_palette_register(unsigned, unsigned, unsigned, unsigned);
void hsv_to_rgb(DBL, DBL, DBL, unsigned *, unsigned *, unsigned *);
void rgb_to_hsv(unsigned, unsigned, unsigned, DBL *, DBL *, DBL *);
#ifdef __WATCOMC__
int matherr(struct exception *);
#else
int _cdecl matherr(struct exception *);
#endif

void display_init(width, height) /* Set video to requested or best mode */
int width, height;
   {
   union REGS inr, outr;
   struct SREGS segs;
   unsigned char _far *fp;
   unsigned int _far *mp;
   unsigned int u, granule, vesamodes[32];
/* unsigned int tmp_word, tmp_word1; */
   int i, maxvesamode;
   long l, lt;
   int show_display_type = FALSE;

   lastline = -1;		/* make sure we display the 1st line... */
   screen_height = height;	/* requested screen height and width */
   screen_width = width;
   if (DisplayFormat == '?'){  
     DisplayFormat = '0';
     show_display_type = TRUE;
     }
    
   if (DisplayFormat != '0')	/* if not 0, some display type specified */
	  whichvga = (int)(DisplayFormat - '0');	/* de-ASCII-fy selection */
   else
	{
	whichvga = auto_detect_vga();
	printf("Display detected: (%c) %s Adapter", whichvga + '0', vga_names[whichvga]);
	if (whichvga < TRUECOLOR)   /*  if not a True-Color graphics card */
	    printf(", with %s 512K RAM\n", vga_512K ? ">=" : "<");
	lt = l = time(&l);
  if(show_display_type)    /* If display format is ? */
	  while (time(&l) < lt + 5)	/* display detected VGA type for 5 secs */
	    ;
	if (!vga_512K)			/* not enough RAM for 640 x 480? */
	    whichvga = MODE13x;		/* then try for next best mode... */
	}

   if (whichvga < TRUECOLOR) /* If an (S)VGA (not True-Color graphics card) */
	{
	if (whichvga == CIRRUS)	/* Register Compatible VGA? */
	    whichvga = MODE13x;	/* MODE13x if > 320x200, else... */

	if (screen_height <= 200 && screen_width <= 320 &&
		PaletteOption != HICOLOR)
	    whichvga = BASIC_VGA;	/* BASIC_VGA if <= 320x200 */

	if (PaletteOption == HICOLOR)
	    {
	    inr.x.ax = 0x10F1;		/* check and see if it's true... */
	    int86(0x10, &inr, &inr);
	    if ((int) inr.h.al != 0x10 || (int)inr.h.bl == 0)
	        {
		printf("Error - High Color Palette Option Unavailable\n");
		exit(1);
		}
	    }
	}

   switch (whichvga)
	{
	case MODE13x:
	    inr.x.ax = 0x0013;   /* setup to VGA 360x480x256 (mode 13X) */
	    svga_width = 360;		/* Fake 640 mode actually is 360 */
	    break;
	case VDC600:
	    inr.x.ax = 0x005E;   /* setup to VGA 640x400x256 (mode 5EH) */
	    svga_height = 400;	 /* This is the only SVGA card w/400 Lines */
	    break;
	case OAKTECH:
	    inr.x.ax = 0x0053;   /* setup to VGA 640x480x256 most SVGAs */
	    break;
	case AHEADA:
	case AHEADB:
	    inr.x.ax = 0x0061;
	    break;
	case EVEREX:
	    inr.x.ax = 0x0070;	 /* BIOS Mode 0x16 for EV-678? */
	    inr.h.bl = 0x30;
	    break;
	case ATIVGA:
	    inr.x.ax = 0x0062;
	    break;
	case TRIDENT:
	    inr.x.ax = 0x005d;
	    break;
	case VIDEO7:
	    inr.x.ax = 0x6f05;
	    inr.h.bl = 0x67;
	    break;
	case CHIPSTECH:
            if (screen_height <= 400)
	    {
	       inr.x.ax = 0x0078;
               svga_height = 400;
            }
	    else inr.x.ax = 0x0079;
	    break;
	case PARADISE:
	    inr.x.ax = 0x005f;
	    break;
	case TSENG3:
	case TSENG4:
	    if ((screen_height <= 200 && screen_width <= 320) &&
		    PaletteOption == HICOLOR)
	    {
		 inr.x.ax = 0x0013;  /* setup to VGA 320x200 for 32K mode  */
		 svga_width = 320;   /* allow scaling to run at 320x200 */
		 svga_height = 200;
	    }
	    else if (screen_height <= 350 && screen_width <= 640)
	    {
		 inr.x.ax = 0x002D;
		 svga_height = 350;
	    }
	    else if (screen_height <= 400 && screen_width <= 640 && whichvga == TSENG4)
	    {
		 if (PaletteOption == HICOLOR)
		      inr.x.ax = 0x0078;
		 else inr.x.ax = 0x002F;
		 svga_height = 400;
	    }
	    else if (screen_height <= 480 && screen_width <= 640)
	    {
		 inr.x.ax = 0x002E;
	    }
	    else	/* 800 by 600 and beyond */
	    {
		 inr.x.ax = 0x0030;
		 svga_width = 800;
		 svga_height = 600;
	    }
	    if ((screen_height > 600 || screen_width > 800) && whichvga == TSENG4)
	    {
		if (PaletteOption == HICOLOR)
		{
		    inr.x.ax = 0x0030; /* Limit to 800x600 in HiColor mode */
		    svga_width = 800;
		    svga_height = 600;
		}
		else
		{
		    inr.x.ax = 0x0038;
		    svga_width = 1024;
		    svga_height = 768;
		}
	    }
	    break;
	case VESA:
	    if ((int)(DisplayFormat - '0') == whichvga) /* not autodetected */
		{
		inr.x.ax = 0x4F00;	/* VESA BIOS adapter identify call */
		inr.x.di = FP_OFF(answer);	/* deposit results here */
		segread(&segs);			/* get our DS, etc. */
		segs.es = FP_SEG(answer);	/* get segment of answer */
		int86x(0x10, &inr, &outr, &segs);
		if (outr.x.ax != 0x004F || strncmp(answer, "VESA", 4))
		    {			   /* if response unsuccessful */
		    printf("Error - VESA BIOS Extensions Not Available.\n");
		    exit(1);
		    }
		}

	    mp = *(unsigned int **)(answer + 14); /* get mode tbl. pointer */
	    for (i = 0; mp[i] != 0xFFFF; i++)	/* for all modes available */
		vesamodes[i] = mp[i];		/* copy to local mode list */
	    maxvesamode = i;			/* note end of list */

	    vesamode = 0x100;			/* start with 640x400x256 */
	    svga_height = 400;			/* (this is an assumption) */

/*
   The following wierd logic takes the minimum available resolution necessary
   to get the desired screen size.  Each successively higher resolution mode
   is tried until the desired screen size is matched or the higer res mode is
   not found.  If the mode was not found, it stays at the highest previously
   found mode and relies on auto-scaling.  Sorry about the clumsy flow and
   "goto"'s (yecch) but there was just no elegant way of putting it - AAC.
*/

	    if (screen_height > 400)
	    {
		for (i = 0; i < maxvesamode; i++) /* scan to see if exists */
		    if (vesamodes[i] == 0x101)
		    {
			vesamode = 0x101;	/* 640x480x256 */
			svga_height = 480;
			break;
		    }
		    if (i == maxvesamode)	/* if mode not found */
			goto gotmode;		/* then use prev. mode */
	    }
	    if (screen_height > 480 || screen_width > 640)
	    {
		 for (i = 0; i < maxvesamode; i++)
		    if (vesamodes[i] == 0x103)
		    {
			vesamode = 0x103;	/* 800x600x256 */
			svga_width = 800;
			svga_height = 600;
			break;
		    }
		    if (i == maxvesamode)
			goto gotmode;
	    }
	    if (screen_height > 600 || screen_width > 800)
	    {
		 for (i = 0; i < maxvesamode; i++)
		    if (vesamodes[i] == 0x105)
		    {
			vesamode = 0x105;	/* 1024x768x256 */
			svga_width = 1024;
			svga_height = 768;
			break;
		    }
		    if (i == maxvesamode)
			goto gotmode;
	    }
	    if (screen_height > 768 || screen_width > 1024)
	    {
		for (i = 0; i < maxvesamode; i++)
		   if (vesamodes[i] == 0x107)
		   {
			vesamode = 0x107;	/* 1280x1024x256 */
			svga_width = 1280;
			svga_height = 1024;
			break;
		   }
	    }

gotmode:    inr.x.ax = 0x4F01;	/* VESA BIOS fetch attributes call */
	    inr.x.bx = inr.x.cx = vesamode;	/* get attrs for mode */
	    inr.x.di = FP_OFF(answer);		/* deposit attribs here */
	    segread(&segs);			/* get our DS, etc. */
	    segs.es = FP_SEG(answer);		/* get segment of answer */
	    int86x(0x10, &inr, &outr, &segs);	/* BIOS fetch attrib call */

/*	    tmp_word = *((unsigned int *)(answer + 12)); * adr lo word */
/*	    tmp_word1 = *((unsigned int *)(answer + 14)); * adr hi word */
/*	    vesa_bankswitch = (void (_cdecl *)(void))(((unsigned long)
		  tmp_word1 << 16) | tmp_word); */

	    granule = *((unsigned int*)(answer + 4)); /* "granule" size */
	    if (granule < 1)
		granule = 1;
	    vesa_granularity = (unsigned char)(64 / granule);
	    gran = 65535L / (unsigned long)vesa_granularity;
	    inr.x.ax = 0x4f02;	/* VESA BIOS initialize video mode call */
	    inr.x.bx = vesamode;
	    break;
	case TIGA_16:
	case TIGA_24:
	    svga_width = 512;	/* NTSC standard televison resolution */
	    svga_height = 480;
	    if ((int)(DisplayFormat - '0') == whichvga) /* not autodetected */
		if (tiga_set(CD_OPEN) < 0) /* Attempt to open the TIGA CD */
		    {
		    printf("Error - TIGA CD Not Installed.");
		    exit(1);
		    }
	    if (!set_videomode(TIGA, INIT_GLOBALS | CLR_SCREEN))
		{
		printf("Error Initializing TIGA.");
		exit(1);
		}
	    if (whichvga == TIGA_16)	/* initialize demo globals */
		{
		set_config(2,0);	/* 16 bit */
		Color_Bits = 5; /* Sorry, only 5 bits per each R-G-B pixel */
		}
	    else set_config(4,0);	/* 24 bit */
	    if (install_primitives() < 0) /* load if not already loaded */
		{
		if (!set_videomode(TIGA, INIT))	/*reinit if due to full heap*/
		    {
		    printf("Error Initializing TIGA.");
		    exit(1);
		    }
	        else if (install_primitives() < 0)
		    {
		    printf("Error Installing TIGA Primitives.");
		    exit(1);
		    }
		}
	default:		/* BASIC_VGA */
	    inr.x.ax = 0x0013;	/* setup to VGA 320x200x256 (mode 13H) */
	    svga_width = 320;	/* allow scaling to run at 320x200 */
	    svga_height = 200;
	}

    if (whichvga >= TRUECOLOR)	/* if True-Color then it has no palette */
	return;

    int86(0x10, &inr, &outr);	/* do the BIOS video mode sel. call */

    if (whichvga == MODE13x)	/* Tweak VGA registers to get higher res! */
	{
	outpw(SEQUENCER, 0x0604);   /* disable chain 4 */
	outpw(SEQUENCER, 0x0f02);   /* allow writes to all planes */

	for (u = 0; u < 43200; u++) /* clear the whole screen */
	    {
	    fp = MK_FP(0xA000, u);
	    *fp = 0;		    /* set all bytes to 0 */
	    }

	outpw(SEQUENCER, 0x0100);   /* synchronous reset */
	outp(MISCOUT, 0xe7);	    /* use 28 mhz dot clock */
	outpw(SEQUENCER, 0x0300);   /* restart sequencer */
	outp(CRTC, 0x11);	    /* ctrl register 11, please */
	outp(CRTC+1, inp(CRTC+1) & 0x7f); /* write-prot cr0-7 */

	for (i = 0; i < 17; i++)    /* write CRTC register array */
	    outpw(CRTC, vptbl[i]);
	}

    if (PaletteOption == HICOLOR && (whichvga == TSENG4 || whichvga == VESA))
	{
	if ((int)inr.h.al == 0x78)	/* if it was mode 78 */
	   inr.x.bx = 0x2F;		/* make it 2F... */
        else inr.x.bx = (unsigned int)inr.h.al;
	inr.x.ax = 0x10F0;
	int86(0x10, &inr, &inr); /* Go from 256 color to 32K color mode */
	if (inr.h.al == 0x10)
	   {
	   printf("High color display in mode 0x%02X\n", inr.x.bx);
	   return;
	   }
	}

   palette_init();  /* if we get here it has a normal 256 color palette DAC */

   if (whichvga == CHIPSTECH) {	/* (not sure why this is necessary) */
      outpw(0x46E8, 0x001E);	/* put chip in setup mode */
      outpw(0x103, 0x0080);	/* enable extended registers */
      outpw(0x46E8, 0x000E);	/* take chip out of setup mode */
      outp(0x3D6, 0x10);
      }

   return;
}


int auto_detect_vga()		/* Autodetect (S)VGA Adapter Type */
    {
    unsigned char _far *biosptr;
    unsigned char tmp_byte;
    unsigned int tmp_word;
    int retcode = BASIC_VGA;
    union REGS inr, outr;
    struct SREGS segs;

    if (tiga_set(CD_OPEN) >= 0)	/* Try initializing HGWS/TIGA Adapter */
	{
	vga_512K = TRUE;	/* All True-Color Adapters are > 512K! */
	return(TIGA_24);	/* Assume 24 bit for now! */
	}

    inr.x.ax = 0x4F00;			/* Test for VESA Adapter */
    inr.x.di = FP_OFF(answer);		/* deposit results here */
    segread(&segs);			/* get our DS, etc. */
    segs.es = FP_SEG(answer);		/* get segment of answer */
    int86x(0x10, &inr, &outr, &segs);	/* BIOS adapter identify call */
    if (outr.x.ax == 0x004F)		/* if response successful */
	if (!strncmp(answer, "VESA", 4))
	    {
	    vga_512K = TRUE;		/* assume all VESA's have >= 512K */
	    return (VESA);
	    }
    biosptr = MK_FP(0xC000, 0x0040);	/* Test for ATI Wonder */
    if (*biosptr == '3' && *(biosptr + 1) == '1')
	{
	_disable();			/* Disable system interrupts */
	outp(0x1CE, 0xBB);
	if (inp(0x1CD) & 0x20)
	    vga_512K = TRUE;
	_enable();			/* Re-enable system interrupts */
	return (ATIVGA);
	}
    inr.x.ax = 0x7000;			/* Test for Everex &| Trident */
    inr.x.bx = 0;
    CLD;
    int86(0x10, &inr, &outr);
    if (outr.h.al == 0x70)
	{
	if (outr.h.ch & 0xC0)
	    vga_512K = TRUE;
	outr.x.dx &= 0xFFF0;
	if (outr.x.dx == 0x6780)
	    {
	    printf("\nT6780\n");
	    return (TRIDENT);
	    }
	if (outr.x.dx == 0x2360)
	    {
	    printf("\nT2360\n");
	    return (TRIDENT);
	    }
	if (outr.x.dx == 0x6730)	/* EVGA? (No BIOS Page Fn.) */
	    {
	    printf("\nE6730\n");
	    return (EVEREX);
	    }
	printf("\nE0000\n");
	return (EVEREX);	/* Newer board with fetchable bankswitch */
	}
    outp(0x3C4, 0x0B);			/* Test for Trident */
    tmp_byte = (unsigned char) inp(0x3C5);
    if ((tmp_byte > 1) && (tmp_byte < 0x10))
	{
	vga_512K = TRUE;
	printf("\nT0000\n");
	return (TRIDENT);
	}
    if (cirrus())			/* Test Video7 Vega VGA (Cirrus) */
	return (CIRRUS);
    inr.x.ax = 0x6F00;			/* Test for Video7 SVGA */
    inr.x.bx = 0;			/* note - Vega VGA (Cirrus) will */
    CLD;
    int86(0x10, &inr, &outr);		/* pass this test - test Cirrus 1st */
    if (outr.h.bh == 'V' && outr.h.bl == '7')
	{
	inr.x.ax = 0x6F07;
	CLD;
	int86(0x10, &inr, &outr);
	if ((outr.h.ah & 0x7F) > 1)
	    vga_512K = TRUE;
	return (VIDEO7);
	}
    outp(0x3CE, 9);			/* Test for Paradise */
    if (!inp(0x3CF))
	{
	outpw(0x3CE, 0x050F);		/* Turn off write protect on regs */
	if (chkbank(0,1))		/* if bank 0 and 1 same not para. */
	    {				/* FALSE == banks same... (C) */
	    if (chkbank(0, 64))		/* if bank 0 and 64 same only 256K */
		vga_512K = TRUE;
	    biosptr = MK_FP(0xC000, 0x0039);	/* Test for AT&T VDC600 */
	    if ((*biosptr == '1') && (*biosptr+1 == '6')) /* p/n 003116 */
		return (VDC600);	/* a real Paradise is p/n 003145 */
	    return (PARADISE);
	    }
	}
    inr.x.ax = 0x5F00;			/* Test for Chips & Tech */
    inr.x.bx = 0;
    CLD;
    int86(0x10, &inr, &outr);
    if (outr.h.al == 0x5F)
	{
	if (outr.h.bh >= 1)
	    vga_512K = TRUE;
	return (CHIPSTECH);
	}
    outp(0x3D4, 0x33);			/* Test for Tseng 4000 or 3000 Chip */
    tmp_word = (unsigned int) inp(0x3D5) << 8;
    outpw(0x3D4, 0x0A33);
    outp(0x3D4, 0x33);
    if ((inp(0x3D5) & 0x0F) == 0x0A)
	{
	outpw(0x3D4, 0x0533);
	outp(0x3D4, 0x33);
	if ((inp(0x3D5) & 0x0F) == 0x05)
	    {
	    retcode = TSENG4;
	    outpw(0x3D4, tmp_word | 0x33);
	    outp(0x3D4, 0x37);
	    if ((inp(0x3D5) & 0x0A) == 0x0A)
		vga_512K = TRUE;
	    outp(0x3BF, 0x03);		/* Enable access to extended regs */
	    outp(0x3D8, 0xA0);
	    outp(0x3D8, 0x29);		/* Enable mapping register access */
	    outp(0x3D8, 0xA0);
	    }
	}
    tmp_byte = (unsigned char) inp(0x3CD);	/* save bank switch reg */
    outp(0x3CD, 0xAA);				/* test register w/ 0xAA */
    if (inp(0x3CD) == 0xAA)
	{
	outp(0x3CD, 0x55);			/* test register w/ 0x55 */
	if (inp(0x3CD) == 0x55)
	    {
	    outp(0x3CD, tmp_byte);		/* restore bank switch reg */
	    if (retcode != TSENG4)		/* yep, it's a Tseng... */
		retcode = TSENG3;
	    vga_512K = TRUE;
	    return (retcode);
	    }
	}
    outpw(0x3CE, 0x200F);		/* Test for Ahead A or B chipsets */
    tmp_byte = (unsigned char) inp(0x3CF);
    if (tmp_byte == 0x21)
	{
	vga_512K = TRUE;		/* Assume all Ahead's have 512K... */
	return (AHEADB);
	}
    if (tmp_byte == 0x20)
	{
	vga_512K = TRUE;
	return (AHEADA);
	}
    if ((inp(0x3DE) & 0xE0) == 0x60)	/* Test for Oak Tech OTI-067 */
	{
	outp(0x3DE, 0x0D);
	if (inp(0x3DF) & 0x80)
	    vga_512K = TRUE;
	return(OAKTECH);
	}
    return (BASIC_VGA);			/* Return 1 if Unknown/BASIC_VGA */
    }

int cirrus()			/* Test for presence of Cirrus VGA Chip */
    {
    unsigned char tmp_byte;
    unsigned int crc_word, tmp_word;
    int retcode = FALSE;

    outp(0x3D4, 0x0C);	/* assume 3Dx addressing, scrn A start addr hi */
    crc_word = (unsigned int) inp(0x3D5) << 8;	/* save the crc */
    outp(0x3D5, 0);				/* clear the crc */
    outp(0x3D4, 0x1F);				/* Eagle ID register */
    tmp_byte = (unsigned char) inp(0x3D5);	/* nybble swap "register" */
    tmp_word = (((tmp_byte & 0x0F) << 4) | ((tmp_byte & 0xf0) >> 4)) << 8;
    outpw(0x3C4, tmp_word | 0x06);		/* disable extensions */
    if (!inp(0x3C5))
	{
	tmp_word = (unsigned int) tmp_byte << 8;
	outpw(0x3C4, tmp_word | 0x06);		/* re-enable extensions */
	if (inp(0x3C5) == 1)
	    retcode = TRUE;
	}
    outpw(0x3D5, crc_word | 0x0c);		/* restore the crc */
    return (retcode);
    }

int chkbank(bank0, bank1)	/* Paradise SVGA specific stuff */
   unsigned int bank0, bank1;	/* returns TRUE if banks are different RAM */
   {
   static unsigned int value = 0x1234;
   unsigned int _far *fp;
   unsigned int temp;
   unsigned int oldval0, oldval1;

   fp = MK_FP(0xb800, 0);		/* point out into display RAM */

   outp(0x3CE, 9);
   outp(0x3CF, bank0);	/* save prior video data and write test values */
   oldval0 = *fp;
   *fp ^= value;
   if (*fp != oldval0 ^ value) {
      *fp = oldval0;
      return FALSE;		/* No RAM there at all -- can't be 512K */
      }

   outp(0x3CE, 9);
   outp(0x3CF, bank1);	/* save prior video data and write test values */
   oldval1 = *fp;
   *fp ^= value;
   if (*fp != oldval1 ^ value) {
      *fp = oldval1;
      outp(0x3CE, 9);
      outp(0x3CF, bank0);
      *fp = oldval0;
      return FALSE;		/* No RAM there at all -- can't be 512K */
      }
   if (*fp != oldval0) {
      *fp = oldval1;
      outp(0x3CE, 9);
      outp(0x3CF, bank0);
      *fp = oldval0;
      return TRUE;		/* pages cannot be the same RAM */
      }

   outp(0x3CE, 9);
   outp(0x3CF, bank0);
   temp = *fp;
   outp(0x3CE, 9);
   outp(0x3CF, bank1);
   *fp = oldval1;
   outp(0x3CE, 9);
   outp(0x3CF, bank0);
   *fp = oldval0;

   if (temp == oldval0)
      return FALSE;		/* pages are the same RAM */
   else
      return TRUE;		/* independent values, so not same RAM */
   }

void palette_init()		/* Fill VGA 256 color palette with colors! */
    {
    register unsigned m;
    unsigned r, g, b;
    register DBL hue, sat, val;

    if (PaletteOption == GREY)	/* B/W Video Mod */
    {
	for (m = 1; m < 64; m++)     /* for the 1st 64 colors... */
	    set_palette_register (m, m, m, m); /* set m to rgb value */
	for (m = 64; m < 256; m++)     /* for the remaining, at full value */
	    set_palette_register (m, 63, 63, 63);
	return;
    }
    if (PaletteOption == P_332)	/* 332 Video Mod */
    {
       for (r=0;r<8;r++)
        for (g=0;g<8;g++)
	 for (b=0;b<4;b++) {
	    m = (r * 32 + g * 4 + b);
	    set_palette_register(m, r * 9, g * 9, b * 21);
	    }
	return;
    }
    set_palette_register(0, 0, 0, 0);	/* make palette register 0 black */

    set_palette_register(64, 63, 63, 63); /* make palette register 64 white */

    set_palette_register(128, 31, 31, 31); /* make register 128 dark grey */

    set_palette_register(192, 48, 48, 48); /* make register 192 lite grey */

    for (m = 1; m < 64; m++)     /* for the 1st 64 colors... */
	{
	sat = 0.5;	/* start with the saturation and intensity low */
	val = 0.5;
	hue = 360.0 * ((DBL)(m)) / 64.0;   /* normalize to 360 */
	hsv_to_rgb (hue, sat, val, &r, &g, &b);
	set_palette_register (m, r, g, b); /* set m to rgb value */

	sat = 1.0;	/* high saturation and half intensity (shades) */
	val = 0.50;
	hue = 360.0 * ((DBL)(m)) / 64.0;   /* normalize to 360 */
	hsv_to_rgb (hue, sat, val, &r, &g, &b);
	set_palette_register (m + 64, r, g, b);  /* set m + 64 */

	sat = 0.5;	/* half saturation and high intensity (pastels) */
	val = 1.0;

	hue = 360.0 * ((DBL)(m)) / 64.0;   /* normalize to 360 */
	hsv_to_rgb (hue, sat, val, &r, &g, &b);
	set_palette_register (m + 128, r, g, b); /* set m + 128 */

	sat = 1.0;            /* normal full HSV set at full intensity */
	val = 1.0;
   
	hue = 360.0 * ((DBL)(m)) / 64.0;   /* normalize to 360 */
	hsv_to_rgb (hue, sat, val, &r, &g, &b);
	set_palette_register (m + 192, r, g, b); /* set m + 192 */
	}
    return;
    }


void display_finished ()
    {
    if (Options & PROMPTEXIT)
	{
	printf ("\007\007");	/* long beep */
	while(!kbhit())		/* wait for key hit */
	   ;
	if (!getch())		/* get another if ext. scancode */
	   getch();
	}
    }


void display_close()   /* setup to Text 80x25 (mode 3) */
    {
    union REGS inr, outr;

    if (whichvga < TRUECOLOR)
	{
	inr.x.ax = 0x0003;
	int86(0x10, &inr, &outr);
	}
    else set_videomode(PREVIOUS, INIT);
    return;
    }


void display_plot (x, y, Red, Green, Blue)   /* plot a single RGB pixel */
   int x, y;
   char Red, Green, Blue;
   {
   union REGS inr;
   register unsigned char color, svga_page;
   unsigned char _far *fp;
   unsigned int svga_word;
   unsigned long svga_loc;
   DBL h, s, v, fx, fy;

   if (!x)			/* first pixel on this line? */
	{
	lastx = -1;		/* reset cache, make sure we do the 1st one */
	lasty = lastline;	/* set last line do to prior line */
	}

   if (screen_height > svga_height)	/* auto-scale Y */
	{
	fy = (DBL)y / ((DBL)screen_height / (DBL)svga_height);
	y = (int)fy;		/* scale y to svga_height */
	if (y <= lasty)		/* discard if repeated line */
	    return;
	lastline = y;		/* save current working line */
	}

   if (screen_width > svga_width)		/* auto-scale X */
	{
	fx = (DBL)x / ((DBL)screen_width / (DBL)svga_width);
	x = (int)fx;		/* scale x to svga_width */
	if (x <= lastx)		/* discard if repeated pixel */
	    return;
	lastx = x;		/* save most recent pixel done */
	}

   if (whichvga == TIGA_16)
	{
	set_bcolor(0);
	set_fcolor(((unsigned)Red << 10) + (Green << 5) + Blue);
	draw_point(x, y);
	return;
	}

   if (whichvga == TIGA_24)
        {
	hpt[0] = Green;
	hpt[1] = Red;
	hpt[2] = Blue;
	host2gsp(hptr, 0x4000L * y + ((long) x << 5), 3, 0);
	return;
        }

   if (PaletteOption == HICOLOR)
	{
	svga_loc = (((unsigned long)svga_width) * y + x) << 1;
	svga_page = (unsigned char)(svga_loc/gran);
	svga_loc %= gran;
	if (svga_page != cur_page)
	    {
	    cur_page = svga_page;
	    if (whichvga != VESA)
		outp(0x3CD, (unsigned char)svga_page);
	    else
		{
		inr.x.bx = map_code;		/* map code = 0 or 1 */
		inr.x.ax = 0x4F05;
		inr.x.dx = (unsigned int)svga_page;
		int86(0x10, &inr, &inr);
		}
	    }
	fp = MK_FP(0xA000, (unsigned int)svga_loc);
	*(unsigned int _far *)fp = ((unsigned int) (Red & 0xF8) << 7) |
				((unsigned int)(Green & 0xF8) << 2) |
				(Blue >> 3);  /* stash 16-bit pixel */
	return;
	}

   if (PaletteOption == GREY)		/* RGB are already set the same, so */
      color = (unsigned char)(Green >> 2);  /* really, any color will do... */
   else if (PaletteOption == P_332)
   {
       h = (int)(7.0 * ((DBL)Red) / 255.0);	/* borrow "h" for red */
       s = (int)(7.0 * ((DBL)Green) / 255.0);	/* borrow "s" for green */
       v  = (int)(3.0 * ((DBL)Blue) / 255.0);	/* borrow "v" for blue */
       color  = (unsigned char)(h * 32 + s * 4 + v);
   }
   else  /* Translate RGB value to nearest of 256 palette Colors (by HSV) */
   {
      rgb_to_hsv((unsigned)Red,(unsigned)Green,(unsigned)Blue, &h, &s, &v);
      if (s < 0.20)   /* black or white if no saturation of color... */
      {
         if (v < 0.25)
            color = 0;        /* black */
         else if (v > 0.8)
            color = 64;       /* white */
         else if (v > 0.5)
            color = 192;      /* lite grey */
         else
            color = 128;      /* dark grey */
      }
      else
      {
         color = (unsigned char) (64.0 * ((DBL)(h)) / 360.0);

         if (!color)
            color = 1;        /* avoid black, white or grey */
      
         if (color > 63)
            color = 63;       /* avoid same */

         if (v > 0.50)
            color |= 0x80;    /* colors 128-255 for high inten. */

         if (s > 0.50)        /* more than half saturated? */
            color |= 0x40;    /* color range 64-128 or 192-255 */
      }
   }
   switch (whichvga)	/* decide on (S)VGA bank switching scheme to use */
	{
	   case BASIC_VGA:	/* none */
		fp = MK_FP(0xA000, 320 * y + x);
		break;
	   case MODE13x:	/* faked */
		svga_word = 1 << (x & 3);	/* form bit plane mask */
		svga_word = (svga_word << 8) | 2;
		outpw(SEQUENCER, svga_word);	/* tweak the sequencer */
		fp = MK_FP(0xA000, 90 * y + (x >> 2));
		break;
	   default:		/* actual bank switch for all SVGA cards */
		svga_loc=((unsigned long)svga_width) * y + x;
		svga_page=(unsigned char)(svga_loc/gran);
		svga_loc %= gran;
		if (cur_page != svga_page)	/* if not in correct bank */
		{
		    if (whichvga != VESA)
			_disable();
		    cur_page = svga_page;	/* set new working bank */
		    newbank();
		    if (whichvga != VESA)
			_enable();
		}
		fp = MK_FP(0xA000, (unsigned int)svga_loc);
		break;
	}

   *fp = color;		/* write normalized pixel color val to bitplane */

   return;
   }

void newbank()		/* Perform SVGA bank switch on demand - Voila! */
{
    register unsigned char tmp_byte, tmp_byte1;
    register unsigned int tmp_word;
    union REGS regs;
    static unsigned char xlateT3[] = {0x40, 0x49, 0x52, 0x5B, 0x64, 0x6D,
	    0x76, 0x7F};
    static unsigned char xlateT4[] = {0x00, 0x11, 0x22, 0x33, 0x44, 0x55,
	    0x66, 0x77, 0x88, 0x99, 0xAA, 0xBB, 0xCC, 0xDD, 0xEE, 0xFF};


    switch (whichvga)
        {
	case VDC600:			/* AT&T VDC 600 */
	    tmp_byte = (unsigned char) (cur_page << 4);	/* was >> 12... */
	    outpw(0x03CE,0x050F);
	    outp(0x03CE,0x09);
	    outp(0x03CF, tmp_byte);
	    break;
	case OAKTECH:			/* Oak Technology OTI-067 */
	    tmp_byte = (unsigned char)(cur_page & 0x0F);
	    outp(0x3DF, (tmp_byte << 4) | tmp_byte);
	    break;
	case AHEADA:			/* Ahead Systems Ver A */
	    outpw(0x3CE, 0x200F);	/* enable extended registers */
	    tmp_byte = (unsigned char)(inp(0x3CC) & 0xDF);  /* bit 0 */
	    if (cur_page & 1)
		    tmp_byte |= 0x20;
	    outp(0x3C2, tmp_byte);
	    outp(0x3CF, 0);		   /* bits 1, 2, 3 */
	    tmp_word = (unsigned int)((cur_page >> 1 )|(inp(0x3D0) & 0xf8));
	    outpw(0x3CF, tmp_word << 8);
	    break;
	case AHEADB:			/* Ahead Systems Ver B */
	    outpw(0x3CE, 0x200F);	/* enable extended registers */
	    tmp_word = (unsigned int)((cur_page << 4) | cur_page);
	    outpw(0x3CF, (tmp_word << 8) | 0x000D);
	    break;
	case EVEREX:			/* Everex SVGA's */
	    outp(0x3C4, 8);
	    if (cur_page & 1)
		 tmp_word = (unsigned int)(inp(0x3C5) | 0x80);
	    else tmp_word = (unsigned int)(inp(0x3C5) & 0x7F);
	    outpw(0x3C4, (tmp_word << 8) | 0x0008);
	    tmp_byte = (unsigned char)(inp(0x3CC) & 0xDF);
	    if (!(cur_page & 2))
		 tmp_byte |= 0x20;
	    outp(0x3C2, tmp_byte);
	    break;
	case ATIVGA:			/* ATI VGA Wonder */
	    outp(0x1CE, 0xB2);
	    tmp_word = (unsigned int)((cur_page << 1) | (inp(0x1CF) & 0xE1));
	    outpw(0x1CE, (tmp_word << 8) | 0x00B2);
	    break;
	case TRIDENT:
	    outp(0x3CE, 6);		/* set page size to 64K */
	    tmp_word = (unsigned int)(inp(0x3CF) | 4) << 8;
	    outpw(0x3CE, tmp_word | 0x0006);
	    outp(0x3C4, 0x0b);		/* switch to BPS mode */
	    inp(0x3C5);			/* dummy read?? */
	    tmp_word = (unsigned int)(cur_page ^ 2) << 8;
	    outpw(0x3C4, tmp_word | 0x000E);
	    break;
	case VIDEO7:			/* Video-7 VRAM, FastRAM SVGA cards */
	    tmp_byte1 = tmp_byte = (unsigned char)(cur_page & 0x0F);
	    outpw(0x3C4, 0xEA06);
	    tmp_word = (unsigned int)(tmp_byte & 1) << 8;
	    outpw(0x3C4, tmp_word | 0x00F9);
	    tmp_byte &= 0x0C;
	    tmp_word = (unsigned int)(tmp_byte >> 2 | tmp_byte) << 8;
	    outpw(0x3C4, tmp_word | 0x00F6);
	    tmp_word |= (inp(0x3C5) & 0xF0) << 8;
	    outpw(0x3C4, tmp_word | 0x00F6);
	    tmp_byte = (unsigned char)((tmp_byte1 << 4) & 0x20);
	    outp(0x3C2, (inp(0x3CC) & 0xDF) | tmp_byte);
	    break;
	case CHIPSTECH:			/* Chips & Technology VGA Chip Set */
/*	    outpw(0x46E8, 0x001E);	* put chip in setup mode */
/*	    outpw(0x103, 0x0080);	* enable extended registers */
/*	    outpw(0x46E8, 0x000E);	* take chip out of setup mode */
/*	    tmp_word = (unsigned int)(cur_page << 2) << 8; * 64K -> 16K */
/*	    outpw(0x3D6, tmp_word | 0x0010); */
	    outp(0x3D7, cur_page << 2);	/* this is all that's necessary?? */
	    break;
	case PARADISE:			/* Paradise, Professional, Plus */
	    outpw(0x3CE, 0x050F);	/* turn off VGA reg. write protect */
	    tmp_word = (unsigned int)(cur_page << 4) << 8;
	    outpw(0x3CE, tmp_word | 0x0009);
	    break;
	case TSENG3:			/* Tseng 3000 - Orchid, STB, etc. */
/*	    tmp_byte = (unsigned char)(cur_page & 0x07); */
/*	    tmp_byte1 = (unsigned char)((tmp_byte << 3) | tmp_byte); */
/*	    outp(0x3CD, tmp_byte1 | 0x40);		 */
	    outp(0x3CD, xlateT3[cur_page & 0x07]);
	    break;
	case TSENG4:			/* Tseng 4000 - Orchid PD+, etc. */
/*	    tmp_byte = (unsigned char)(cur_page & 0x0F); */
/*	    tmp_byte1 = (unsigned char)((tmp_byte << 4) | tmp_byte); */
/*	    outp(0x3BF, 3);		* enable access to extended regs */
/*	    outp(0x3D8, 0xA0);		* this may not be necessary! */
/*		outp(0x3CD, tmp_byte1); */
	    outp(0x3CD, xlateT4[cur_page & 0x0F]);
	    break;
	case VESA:			/* VESA standard mode bank switch */
	    regs.x.ax = 0x4F05;		/* VESA BIOS bank switch call */
	    regs.x.dx = cur_page;
	    regs.x.bx = map_code;	/* Map code 0 or 1? */
	    int86(0x10, &regs, &regs);	/* Do the video BIOS interrupt */
/*	    (*vesa_bankswitch)();	 * call the bios bank-switch rtn */
        }
    return;
}

void set_palette_register (Val, Red, Green, Blue)
   unsigned Val;
   unsigned Red, Green, Blue;
   {
   union REGS Regs;

   Regs.x.ax = 0x1010;              /* Set one palette register function */
   Regs.x.bx = Val;                 /* the palette register to set (color#)*/
   Regs.h.dh = (char)(Red & 0x3f);  /* set the gun values (6 bits ea.) */
   Regs.h.ch = (char)(Green & 0x3f);
   Regs.h.cl = (char)(Blue & 0x3f);
   int86(0x10, &Regs, &Regs);       /* Do the video interrupt */
   }

/* Conversion from Hue, Saturation, Value to Red, Green, and Blue and back */
/* From "Computer Graphics", Donald Hearn & M. Pauline Baker, p. 304 */

void hsv_to_rgb(hue, s, v, r, g, b)
   DBL hue, s, v;               /* hue (0.0-360.0) s and v from 0.0-1.0) */
   unsigned *r, *g, *b;         /* values from 0 to 63 */
   {
   register DBL i, f, p1, p2, p3;
   register DBL xh;
   register DBL nr, ng, nb;	/* rgb values of 0.0 - 1.0 */

   if (hue == 360.0)
      hue = 0.0;                /* (THIS LOOKS BACKWARDS BUT OK) */

   xh = hue / 60.0;             /* convert hue to be in 0,6     */
   i = floor(xh);               /* i = greatest integer <= h    */
   f = xh - i;                  /* f = fractional part of h     */
   p1 = v * (1 - s);
   p2 = v * (1 - (s * f));
   p3 = v * (1 - (s * (1 - f)));

   switch ((int) i)
      {
      case 0:
         nr = v;
         ng = p3;
         nb = p1;
         break;
      case 1:
         nr = p2;
         ng = v;
         nb = p1;
         break;
      case 2:
         nr = p1;
         ng = v;
         nb = p3;
         break;
      case 3:
         nr = p1;
         ng = p2;
         nb = v;
         break;
      case 4:
         nr = p3;
         ng = p1;
         nb = v;
         break;
      case 5:
         nr = v;
         ng = p1;
         nb = p2;
         break;
      default:
         nr = ng = nb = 0;
        }

   *r = (unsigned)(nr * 63.0); /* Normalize the values to 63 */
   *g = (unsigned)(ng * 63.0);
   *b = (unsigned)(nb * 63.0);
   
   return;
   }


void rgb_to_hsv(r, g, b, h, s, v)
   unsigned r, g, b;
   DBL *h, *s, *v;
   {
   register DBL m, r1, g1, b1;
   register DBL nr, ng, nb;		/* rgb values of 0.0 - 1.0 */
   register DBL nh = 0.0, ns, nv;	/* hsv local values */

   nr = (DBL) r / 255.0;
   ng = (DBL) g / 255.0;
   nb = (DBL) b / 255.0;

   nv = max (nr, max (ng, nb));
   m = min (nr, min (ng, nb));

   if (nv != 0.0)                /* if no value, it's black! */
      ns = (nv - m) / nv;
   else
      ns = 0.0;                  /* black = no colour saturation */

   if (ns == 0.0)                /* hue undefined if no saturation */
   {
      *h = 0.0;                  /* return black level (?) */
      *s = 0.0;
      *v = nv;
      return;
   }

   r1 = (nv - nr) / (nv - m);    /* distance of color from red   */
   g1 = (nv - ng) / (nv - m);    /* distance of color from green */
   b1 = (nv - nb) / (nv - m);    /* distance of color from blue  */

   if (nv == nr)
   {
      if (m == ng)
         nh = 5. + b1;
      else
         nh = 1. - g1;
   } 

   if (nv == ng)
      {
      if (m == nb)
         nh = 1. + r1;
      else
         nh = 3. - b1;
      }

   if (nv == nb)
      {
      if (m == nr)
         nh = 3. + g1;
      else
         nh = 5. - r1;
      }

   *h = nh * 60.0;      /* return h converted to degrees */
   *s = ns;
   *v = nv;
   return;
   }


#if !__STDC__

/* ANSI Standard psuedo-random number generator */

static unsigned long int next = 1;

int rand()
   {
   next = next * 1103515245L + 12345L;
   return ((int) (next / 0x10000L) & 0x7FFF);
   }

void srand(seed)
   unsigned int seed;
   {
   next = seed;
   }

#endif



/* Math Error exception struct format:
	int type;		- exception type - see below
	char _FAR_ *name;	- name of function where error occured
	long double arg1;	- first argument to function
	long double arg2;	- second argument (if any) to function
	long double retval;	- value to be returned by function
*/

#ifdef __WATCOMC__
#define EDOM	7	/* MSC is 33 */
#define ERANGE	8	/* MSC is 34 */

int matherr(e)
#else
int _cdecl matherr(e)
#endif
   struct exception *e;
   {
   if (Options & DEBUGGING) {
      /* Since we are just making pictures, not keeping nuclear power under
         control - it really isn't important if there is a minor math problem.
         This routine traps and ignores them.  Note: the most common one is
         a DOMAIN error coming out of "acos". */
      switch (e->type) {
	 case DOMAIN   : printf("DOMAIN error in '%s'\n", e->name); break;
	 case SING     : printf("SING   error in '%s'\n", e->name); break;
	 case OVERFLOW : printf("OVERFLOW error in '%s'\n", e->name); break;
	 case UNDERFLOW: printf("UNDERFLOW error in '%s'\n", e->name); break;
	 case TLOSS    : printf("TLOSS error in '%s'\n", e->name); break;
	 case PLOSS    : printf("PLOSS error in '%s'\n", e->name); break;
	 case EDOM     : printf("EDOM error in '%s'\n", e->name); break;
	 case ERANGE   : printf("ERANGE error in '%s'\n", e->name); break;
	 default       : printf("Unknown math error in '%s'\n",e->name);break;
         }
      }
   return (1);	/* Indicate the math error was corrected... */
   }
