/*
 * cylinder.c
 *
 * Copyright (C) 1989, Craig E. Kolb
 *
 * This software may be freely copied, modified, and redistributed,
 * provided that this copyright notice is preserved on all copies.
 *
 * There is no warranty or other guarantee of fitness for this software,
 * it is provided solely .  Bug reports or fixes may be sent
 * to the author, who may or may not act on them as he desires.
 *
 * You may not include this software in a program or other software product
 * without supplying the source, or without informing the end-user that the
 * source is available for no extra charge.
 *
 * If you modify this software, you should include a notice giving the
 * name of the person performing the modification, the date of modification,
 * and the reason for such modification.
 *
 * $Id: cylinder.c,v 3.0.1.2 89/12/06 16:33:41 craig Exp $
 *
 * $Log:	cylinder.c,v $
 * Revision 3.0.1.2  89/12/06  16:33:41  craig
 * patch2: Added calls to new error/warning routines.
 * 
 * Revision 3.0.1.1  89/11/18  14:07:52  craig
 * patch1: Changes to reflect new names of transformation routines.
 * 
 * Revision 3.0  89/10/27  02:05:48  craig
 * Baseline for first official release.
 * 
 */
#include <stdio.h>
#include <math.h>
#include "typedefs.h"
#include "funcdefs.h"
#include "constants.h"

Object *
makcyl(surf, cent, ax, r, trans)
char *surf;
Vector *cent, *ax;
double r;
TransInfo *trans;
{
	Cylinder *cyl;
	Primitive *prim;
	Object *newobj;
	double len;
	Vector axis, dir;

	if (r <= 0.) {
		yywarning("Invalid cylinder radius.\n");
		return (Object *)0;
	}

	prim = mallocprim();
	newobj = new_object(NULL, CYL, (char *)prim, (Trans *)NULL);
	prim->surf = find_surface(surf);
	prim->type = CYL;
	cyl = (Cylinder *)Malloc(sizeof(Cylinder));
	prim->objpnt.p_cylinder = cyl;

	axis.x = ax->x - cent->x;
	axis.y = ax->y - cent->y;
	axis.z = ax->z - cent->z;

	len = normalize(&axis);
	if(len < EPSILON) {
		yywarning("Degenerate cylinder.\n");
		free((char *)cyl);
		free((char *)prim);
		return (Object *)0;
	}

	cyl->rad = r*r;
	cyl->len = len;
	/*
	 * Define matrix to transform from axis-aligned to desired cylinder.
	 */
	dir.x = axis.y;
	dir.y = -axis.x;
	dir.z = 0.;
	RS_rotate(trans, &dir, acos(axis.z));
	RS_translate(trans, cent);

	return newobj;
}

/*
 * Ray-cylinder intersection test.
 */
double
intcyl(pos, ray, obj)
Vector           *pos, *ray;
Primitive       *obj;
{
	double t1, t2, a, b, c, zpos1, zpos2, et1, et2, x, y, disc;
	extern unsigned long primtests[];
	Cylinder *cyl;

	primtests[CYL]++;
	cyl = obj->objpnt.p_cylinder;

	a = ray->x * ray->x + ray->y * ray->y;
	c = pos->x*pos->x + pos->y*pos->y - cyl->rad;

	if (a < EPSILON*EPSILON) {	/* |ray->z| == 1. */
		if(c < EPSILON*EPSILON)	/* Within endcap */
			/* Wrong if origin is inside cylinder. */
			return min(-pos->z / ray->z,
				  (cyl->len - pos->z) / ray->z);
		return 0.;
	}

	b = ray->x * pos->x + ray->y * pos->y;
	disc = b*b - a*c;
	if(disc < 0.)
		return 0.;
	disc = sqrt(disc);
	t1 = (-b + disc) / a;
	t2 = (-b - disc) / a;
	if(t1 < EPSILON && t2 < EPSILON)
		return 0.;
	zpos1 = pos->z + t1 * ray->z;
	zpos2 = pos->z + t2 * ray->z;
	if ((zpos1 > cyl->len && zpos2 > cyl->len) ||
	    (zpos1 < 0. && zpos2 < 0.))
		return 0.;
	if (t1 < EPSILON)
		t1 = FAR_AWAY;
	if (t2 < EPSILON)
		t2 = FAR_AWAY;
	if (t1 == FAR_AWAY && t2 == FAR_AWAY)
		return 0.;
	/*
	 * Don't bother checking endcaps if both intersection points
	 * are on the cylinder.
	 */
	if ((zpos1 > 0. && zpos1 < cyl->len && zpos2 > 0. && zpos2 < cyl->len))
		return min(t1, t2);
	/*
 	 * It's possible to get rid of the ray-disc intersection tests
	 * (by looking at t1, t2 and zpos1, zpos), but the code gets messy.
	 */
	if (zpos1 < 0. || zpos1 > cyl->len)
		t1 = FAR_AWAY;
	if (zpos2 < 0. || zpos2 > cyl->len)
		t2 = FAR_AWAY;
	et1 = -pos->z / ray->z;
	x = pos->x + et1 * ray->x;
	y = pos->y + et1 * ray->y;
	if (x*x + y*y > cyl->rad)
		et1 = FAR_AWAY;
	et2 = (cyl->len - pos->z) / ray->z;
	x = pos->x + et2 * ray->x;
	y = pos->y + et2 * ray->y;
	if (x*x + y*y > cyl->rad)
		et2 = FAR_AWAY;
	t1 = min(t1, min(t2, min(et1, et2)));
	return (t1 == FAR_AWAY ? 0. : t1);
}

nrmcyl(pos, obj, nrm)
Vector *pos, *nrm;
Primitive *obj;
{
	Cylinder *cyl;
	double dist;

	cyl = obj->objpnt.p_cylinder;

	dist = pos->x*pos->x + pos->y*pos->y;
	if (dist+EPSILON < cyl->rad) {
		if (equal(pos->z,0.)) {
			/*
			 * Hit on lower endcap.
			 */
			nrm->x = nrm->y = 0.;
			nrm->z = -1.;
		} else {
			/*
		 	* Hit on upper endcap.
		 	*/
			nrm->x = nrm->y = 0.;
			nrm->z = 1.;
		}
	} else {	/* Hit along cylinder. */
		nrm->x = pos->x;
		nrm->y = pos->y;
		nrm->z = 0.;
		/* Will be normalized by ShadeRay(). */
	}
}

cylextent(o, bounds)
Primitive *o;
double bounds[2][3];
{
	Cylinder *cyl;
	double r;
	cyl = o->objpnt.p_cylinder;

	r = sqrt(cyl->rad);
	bounds[LOW][X] = bounds[LOW][Y] = -r;
	bounds[HIGH][X] = bounds[HIGH][Y] = r;
	bounds[LOW][Z] = 0.;
	bounds[HIGH][Z] = cyl->len;
}
