/*
 * object.c
 *
 * Copyright (C) 1989, Craig E. Kolb
 *
 * This software may be freely copied, modified, and redistributed,
 * provided that this copyright notice is preserved on all copies.
 *
 * There is no warranty or other guarantee of fitness for this software,
 * it is provided solely .  Bug reports or fixes may be sent
 * to the author, who may or may not act on them as he desires.
 *
 * You may not include this software in a program or other software product
 * without supplying the source, or without informing the end-user that the
 * source is available for no extra charge.
 *
 * If you modify this software, you should include a notice giving the
 * name of the person performing the modification, the date of modification,
 * and the reason for such modification.
 *
 * $Id: object.c,v 3.0.1.1 89/12/06 16:33:20 craig Exp $
 *
 * $Log:	object.c,v $
 * Revision 3.0.1.1  89/12/06  16:33:20  craig
 * patch2: Added calls to new error/warning routines.
 * 
 * Revision 3.0  89/10/27  02:05:58  craig
 * Baseline for first official release.
 * 
 */
#include <stdio.h>
#include <math.h>
#include "constants.h"
#include "typedefs.h"
#include "funcdefs.h"
#include "texture.h"

Object *World;			/* World Object */
ObjList *Objects;		/* Linked list of defined objects */
int WorldXSize, WorldYSize, WorldZSize;		/* World grid resolution */

/*
 * Create a new object with the given properties.
 */
Object *
new_object(name, type, data, trans)
char *name, *data;
char type;
Trans *trans;
{
	Object *new;

	new = (Object *)share_malloc(sizeof(Object));
	new->name = strsave(name);
	new->type = type;
	new->data = data;
	new->trans = trans;
#ifdef LINDA
	/*
	 * If the counter is in shared memory, processes will
	 * be modifying it left-and-right.  So, we cheat and
	 * make counter a pointer to a non-shared location and
	 * store the value there.
	 */
	new->counter = (unsigned long *)malloc(sizeof(unsigned long));
	*new->counter = 0;
#else
	new->counter = 0;
#endif
	new->texture = (Texture *)0;
	/*
	 * bounds is left uninitialized.
	 */
	return new;
}

/*
 * Add a copy of the named object to parent object
 */
Object *
add_child_named(name, parent)
char *name;
Object *parent;
{
	Object *child, *newobj;
	int i;

	child = get_object_named(name);
	if (child == (Object *)0)
		yyerror("There is no object named \"%s\".", name);
	/*
	 * Create new object that points to child
	 * and add to 'parent' list.
	 */
	newobj = add_child(child, parent);
	/*
	 * New object's bounding box is initally the same
	 * as the child's.
	 */
	for (i = 0; i < 3; i++) {
		newobj->bounds[0][i] = child->bounds[0][i];
		newobj->bounds[1][i] = child->bounds[1][i];
	}
	return newobj;
}

/*
 * Add primitive object to parent object.
 */
add_prim(child, parent)
Object *child, *parent;
{
	ObjList *newnode;

	newnode = (ObjList *)share_malloc(sizeof(ObjList));
	newnode->data = child;
	if (parent == (Object *)0) {
		newnode->next = (ObjList *)World->data;
		World->data = (char *)newnode;
	} else {
		newnode->next = (ObjList *)parent->data;
		parent->data = (char *)newnode;
	}
}

/*
 * Make a copy of "child" and attach it to parent's linked list
 * of objects.
 */
Object *
add_child(child, parent)
Object *child, *parent;
{
	Object *newobj;
	ObjList *newnode;

	newobj = new_object(NULL, child->type, child->data, child->trans);
	newobj->texture = child->texture;
	newnode = (ObjList *)share_malloc(sizeof(ObjList));
	newnode->data = newobj;
	if (parent == (Object *)0) {
		newnode->next = (ObjList *)World->data;
		World->data = (char *)newnode;
	} else {
		newnode->next = (ObjList *)parent->data;
		parent->data = (char *)newnode;
	}
	return newobj;
}

/*
 * Return pointer to named object, NULL if no such object has been defined.
 */
Object *
get_object_named(name)
char *name;
{
	ObjList *ltmp;
	for (ltmp = Objects; ltmp; ltmp = ltmp->next)
		if (strcmp(name, ltmp->data->name) == 0)
			return ltmp->data;
	return (Object *)0;
}

/*
 * Add object to list of defined objects.
 */
add_to_objects(obj)
Object *obj;
{
	ObjList *ltmp;
	extern int Verbose;
	extern FILE *fstats;

	ltmp = (ObjList *)Malloc(sizeof(ObjList));
	ltmp->data = obj;
	ltmp->next = Objects;
	Objects = ltmp;
	if (Verbose) {
		/*
		 * Report bounding box of named object.
		 */
		fprintf(fstats,"Object \"%s\" extent:\n", obj->name);
		print_bounds(obj->bounds);
	}
}

/*
 * Allocate space for a string, copy string into space.
 */
char *
strsave(s)
char *s;
{
	char *tmp;

	if (s == (char *)0)
		return (char *)0;

	tmp = (char *)Malloc((unsigned)strlen(s) + 1);
	strcpy(tmp, s);
	return tmp;
}

/*
 * Set "bounds" of primitive to be the extent of the primitive.
 */
set_prim_bounds(obj)
Object *obj;
{
	extern int (*objextent[])();

	(*objextent[((Primitive *)obj->data)->type])
		((Primitive *)obj->data, obj->bounds);
	obj->bounds[LOW][X] -= EPSILON;
	obj->bounds[HIGH][X] += EPSILON;
	obj->bounds[LOW][Y] -= EPSILON;
	obj->bounds[HIGH][Y] += EPSILON;
	obj->bounds[LOW][Z] -= EPSILON;
	obj->bounds[HIGH][Z] += EPSILON;
}

