/*
 * texture.c
 *
 * Copyright (C) 1989, Craig E. Kolb
 *
 * This software may be freely copied, modified, and redistributed,
 * provided that this copyright notice is preserved on all copies.
 *
 * There is no warranty or other guarantee of fitness for this software,
 * it is provided solely .  Bug reports or fixes may be sent
 * to the author, who may or may not act on them as he desires.
 *
 * You may not include this software in a program or other software product
 * without supplying the source, or without informing the end-user that the
 * source is available for no extra charge.
 *
 * If you modify this software, you should include a notice giving the
 * name of the person performing the modification, the date of modification,
 * and the reason for such modification.
 *
 * $Id: texture.c,v 3.0.1.2 89/12/06 16:33:50 craig Exp $
 *
 * $Log:	texture.c,v $
 * Revision 3.0.1.2  89/12/06  16:33:50  craig
 * patch2: Added calls to new error/warning routines.
 * 
 * Revision 3.0.1.1  89/11/27  18:36:54  craig
 * patch2: Removed hardcoded constants in calculation of ambient
 * patch2: component in several textures.
 * 
 * Revision 3.0  89/10/27  02:06:05  craig
 * Baseline for first official release.
 * 
 */
#include <stdio.h>
#include <math.h>
#include "constants.h"
#include "typedefs.h"
#include "funcdefs.h"
#include "texture.h"

/*
 * Array of texturing functions indexed by type.
 */
int (*textures[])() =
	{CheckerText, BlotchText, BumpText, MarbleText, fBmText, fBmBumpText,
	 WoodText};
extern double Chaos(), Marble(), fBm(), Noise();
Color	*read_colormap();

/*
 * Return pointer to new texture structure.
 */
Texture *
new_texture(type)
char type;
{
	Texture *new;

	new = (Texture *)Malloc(sizeof(Texture));
	new->type = type;
	new->surf1 = (Surface *)NULL;
	new->next = (Texture *)NULL;
	new->trans = (Trans *)NULL;
	return new;
}

/*
 * Apply appropriate textures to a surface.
 */
apply_textures(hitinfo, list)
HitInfo *hitinfo;
Texture *list;
{
	Texture *ttmp;
	Vector ptmp;

	for (ttmp = list; ttmp; ttmp = ttmp->next) {
		ptmp = hitinfo->pos;
		if (ttmp->trans) {
			/*
			 * Transform position and normal to texture space.
			 */
			transform_point(&ptmp,&ttmp->trans->world2obj);
			TransformNormal(&hitinfo->norm,&ttmp->trans->obj2world);
		}
		/*
		 * Make sure to use a normalized normal.
		 */
		(void)normalize(&hitinfo->norm);
		(*textures[(int)ttmp->type])
			(ttmp,&ptmp,&hitinfo->norm,&hitinfo->surf);
		if (ttmp->trans) {
			/*
			 * Transform the normal back to world-space.
			 */
			TransformNormal(&hitinfo->norm,&ttmp->trans->world2obj);
		}
	}
}

Texture *
NewWoodText()
{
	Texture *text;

	text = new_texture(WOOD);
	return text;
}

WoodText(text, pos, norm, surf)
Texture *text;
Vector *pos, *norm;
Surface *surf;
{
	double red, grn, blu;
	double chaos, midBrown, brownLayer, greenLayer;
	double perturb, brownPerturb, greenPerturb, grnPerturb;
	double t;

	chaos = Chaos(pos, 7);
	t = sin(sin(8.*chaos + 7*pos->x +3.*pos->y));

	greenLayer = brownLayer = abs(t);

	perturb = fabs(sin(40.*chaos + 50*pos->z));

	brownPerturb = .6*perturb + 0.3;
	greenPerturb = .2*perturb + 0.8;
	grnPerturb = .15*perturb + 0.85;
	grn = 0.5 * pow(abs(brownLayer), 0.3);
	brownLayer = pow(0.5 * (brownLayer+1.0), 0.6) * brownPerturb;
	greenLayer = pow(0.5 * (greenLayer+1.0), 0.6) * greenPerturb;

	red = (0.5*brownLayer + 0.35*greenLayer)*2.*grn;
	blu = (0.25*brownLayer + 0.35*greenLayer)*2.0*grn;
	grn *= max(brownLayer, greenLayer) * grnPerturb;

	surf->diff.r *= red;
	surf->diff.g *= grn;
	surf->diff.b *= blu;
	surf->amb.r *= red;
	surf->amb.g *= grn;
	surf->amb.b *= blu;
}

/*
 * Create and return a reference to a "checker" texture.
 */
Texture *
NewCheckText(surf)
char *surf;
{
	Texture *text;

	text = new_texture(CHECKER);
	text->surf1 = find_surface(surf);

	return text;
}

Texture *
NewfBmBumpText(offset, scale, h, lambda, octaves)
double h, lambda, scale, offset;
int octaves;
{
	Texture *text;

	text = NewfBmText(offset, scale, h, lambda, octaves, 0., (char *)0);

	text->type = FBMBUMP;

	return text;
}

Texture *
NewfBmText(offset, scale, h, lambda, octaves, thresh, mapname)
double h, lambda, scale, offset, thresh;
int octaves;
char *mapname;
{
	double beta;
	Texture *text;

	text = new_texture(FBM);

	text->args = (double *)Malloc(5 * sizeof(double));
	beta = 1. + 2 * h;
	text->args[0] = pow(lambda, -0.5 * beta);	/* omega */
	text->args[1] = lambda;
	text->args[2] = scale;
	text->args[3] = offset;
	text->args[4] = thresh;
	text->size = (double)octaves;
	if (mapname != (char *)0)
		text->colormap = read_colormap(mapname);

	return text;
}

/*
 * Create and return a reference to a "bump" texture.
 */
Texture *
NewBumpText(size)
double size;
{
	Texture *text;

	text = new_texture(BUMP);
	text->size = size;

	return text;
}

/*
 * Create and return a reference to a "blotch" texture.
 */
Texture *
NewBlotchText(scale, surf)
double scale;
char *surf;
{
	Texture *text;

	text = new_texture(BLOTCH);
	text->size = scale;
	text->surf1 = find_surface(surf);

	return text;
}

Texture *
NewMarbleText(mapname)
char *mapname;
{
	Texture *text;

	text = new_texture(MARBLE);
	if (mapname)
		text->colormap = read_colormap(mapname);
	return text;
}

/*
 * Apply "blotch" texture.
 */
BlotchText(text, pos, norm, surf)
Texture *text;
Vector *pos, *norm;
Surface *surf;
{
	double val;

	/*
	 * "size" represents the 'average' noise value at a point.
	 */
	val = Noise(pos);
	if (val > text->size) {
		val = (val - text->size) / (1. - text->size);
		blend_surface(surf, text->surf1, 1. - val, val);
	}
}

fBmText(text, pos, norm, surf)
Texture *text;
Vector *pos, *norm;
Surface *surf;
{
	double val;
	int index;

	val = fBm(pos, text->args[0], text->args[1], (int)text->size);
	if (val < text->args[4])
		val = 0.;
	else
		val = text->args[3] + text->args[2] * (val - text->args[4]);
	if (text->colormap) {
		index = 255. * val;
		if (index > 255) index = 255;
		if (index < 0) index = 0;
		surf->diff.r *= text->colormap[index].r;
		surf->diff.g *= text->colormap[index].g;
		surf->diff.b *= text->colormap[index].b;
		surf->amb.r *= text->colormap[index].r;
		surf->amb.g *= text->colormap[index].g;
		surf->amb.b *= text->colormap[index].b;
	} else {
		ScaleColor(val, surf->diff, &surf->diff);
		ScaleColor(val, surf->amb, &surf->amb);
	}
}

/*
 * Apply a "checker" texture.
 */
CheckerText(text, pos, norm, surf)
Texture *text;
Vector *pos, *norm;
Surface *surf;
{
	int xp, yp, zp;

	xp = pos->x > 0. ? pos->x : 1. - pos->x;
	yp = pos->y > 0. ? pos->y : 1. - pos->y;
	zp = pos->z > 0. ? pos->z : 1. - pos->z;

	if ((xp + yp + zp) % 2)
		*surf = *text->surf1;
	/* else surface stays the same. */
}

fBmBumpText(text, pos, norm, surf)
Texture *text;
Vector *pos, *norm;
Surface *surf;
{
	Vector disp;
	double w;

	VfBm(pos, text->args[0], text->args[1], (int)text->size, &disp);
	w = text->args[2];
	norm->x += text->args[3] + disp.x * w;
	norm->y += text->args[3] + disp.y * w;
	norm->z += text->args[3] + disp.z * w;
}

/*
 * Apply a "bump" texture.
 */
BumpText(text, pos, norm, surf)
Texture *text;
Vector *pos, *norm;
Surface *surf;
{
	Vector disp;

	DNoise(pos, &disp);
	norm->x += disp.x * text->size;
	norm->y += disp.y * text->size;
	norm->z += disp.z * text->size;
	(void)normalize(norm);
}

MarbleText(text, pos, norm, surf)
Texture *text;
Vector *pos, *norm;
Surface *surf;
{
	double val;
	int index;

	val = Marble(pos);
	if (text->colormap) {
		index = (int)(255. * val);
		surf->diff.r *= text->colormap[index].r;
		surf->diff.g *= text->colormap[index].g;
		surf->diff.b *= text->colormap[index].b;
		surf->amb.r *= text->colormap[index].r;
		surf->amb.g *= text->colormap[index].g;
		surf->amb.b *= text->colormap[index].b;
	} else {
		ScaleColor(val, surf->amb, &surf->amb);
		ScaleColor(val, surf->diff, &surf->diff);
	}
}

Color *
read_colormap(filename)
char *filename;
{
	FILE *fp;
	Color *map;
	char buf[BUFSIZ];
	int i;

	fp = fopen(filename, "r");
	if (fp == (FILE *)NULL)
		yyerror("Cannot open colormap file \"%s\".\n", filename);

	map = (Color *)Calloc(256, sizeof(Color));

	for (i = 0; fgets(buf,BUFSIZ,fp) != NULL && i < 256; i++) {
		sscanf(buf,"%lf %lf %lf",&map[i].r, &map[i].g, &map[i].b);
		ScaleColor(1. / 255., map[i], &map[i]);
	}
	return map;
}
