/*****************************************************************************
 ****                                                                     ****
 **** bv.c  for Windows                                                   ****
 ****                                                                     ****
 **** atree release 2.7                                                   ****
 **** Copyright (C) A. Dwelly, R. Manderscheid, M. Thomas, W.W. Armstrong ****
 ****               1991, 1992                                            ****
 ****                                                                     ****
 **** License:                                                            ****
 **** A royalty-free license is granted for the use of this software for  ****
 **** NON_COMMERCIAL PURPOSES ONLY. The software may be copied and/or     ****
 **** modified provided this notice appears in its entirety and unchanged ****
 **** in all derived source programs.  Persons modifying the code are     ****
 **** requested to state the date, the changes made and who made them     ****
 **** in the modification history.                                        ****
 ****                                                                     ****
 **** Patent License:                                                     ****
 **** The use of a digital circuit which transmits a signal indicating    ****
 **** heuristic responsibility is protected by U. S. Patent 3,934,231     ****
 **** and others assigned to Dendronic Decisions Limited of Edmonton,     ****
 **** W. W. Armstrong, President.  A royalty-free license is granted      ****
 **** by the company to use this patent for NON_COMMERCIAL PURPOSES ONLY  ****
 **** to adapt logic trees using this program and its modifications.      ****
 ****                                                                     ****
 **** Limited Warranty:                                                   ****
 **** This software is provided "as is" without warranty of any kind,     ****
 **** either expressed or implied, including, but not limited to, the     ****
 **** implied warrantees of merchantability and fitness for a particular  ****
 **** purpose.  The entire risk as to the quality and performance of the  ****
 **** program is with the user.  Neither the authors, nor the             ****
 **** University of Alberta, its officers, agents, servants or employees  ****
 **** shall be liable or responsible in any way for any damage to         ****
 **** property or direct personal or consequential injury of any nature   ****
 **** whatsoever that may be suffered or sustained by any licensee, user  ****
 **** or any other party as a consequence of the use or disposition of    ****
 **** this software.                                                      ****
 ****                                                                     ****
 **** Modification history:                                               ****
 ****                                                                     ****
 **** 90.05.09 Initial implementation, A.Dwelly                           ****
 **** 91.07.15 Release 2, Rolf Manderscheid                               ****
 **** 92.27.02 Release 2.5 Monroe Thomas                                  ****
 **** 92.03.07 Release 2.6, Monroe Thomas                                 ****
 **** 92.01.08 Release 2.7, Monroe Thomas                                 ****
 ****                                                                     ****
 *****************************************************************************/

/*****************************************************************************
 ****                                                                     ****
 **** Include Files                                                       ****
 ****                                                                     ****
 *****************************************************************************/

#ifndef __WINDOWS_H
#include <windows.h>
#endif

#ifndef __STDIO_H
#include <stdio.h>
#endif

#ifndef __STDLIB_H
#include <stdlib.h>
#endif

#ifndef __ASSERT_H
#include <assert.h>
#endif

#ifndef __ALLOC_H
#include <alloc.h>
#endif

#ifndef __BV_H
#include "bv.h"
#endif

/* Public and private procedures */

#define public
#define private static

/* Memory check */

#define MEMCHECK(p) \
		if (p == NULL){ \
						 MessageBox(NULL,"Could not allocate requested memory",\
						 "bv",MB_OK | MB_ICONSTOP);\
						 PostQuitMessage(0);\
						 exit(0);\
						}

#define BYTE	8 /* Byte size in bits.*/

/*
 * Types
 */

typedef int bool_t;      /*
						  * Only TRUE, FALSE, UNEVALUATED or ERROR
						  * are used in these variables
						  */

/*****************************************************************************
 ****                                                                     ****
 **** LPBIT_VEC bv_create(length)                                         ****
 ****                                                                     ****
 **** int length;                                                         ****
 ****                                                                     ****
 **** Synopsis:                                                           ****
 ****                                                                     ****
 **** Produces a vector of _length_ bits, each one of which has been set  ****
 **** to 0                                                                ****
 *****************************************************************************/


public LPBIT_VEC FAR PASCAL
bv_create(length)

int length;

{
	int i;
	LPBIT_VEC out_vec;

	assert(length > 0);

	out_vec = (LPBIT_VEC) farmalloc((unsigned)sizeof(bit_vec));
	MEMCHECK(out_vec);

	out_vec -> len = length;
	out_vec -> bv = farmalloc((unsigned) (length + BYTE - 1) / BYTE);
	MEMCHECK(out_vec -> bv);

	for (i = 0; i < (length + BYTE - 1) / BYTE; i++)
	{
		*((out_vec -> bv) + i) = (char) 0;
	}

	return(out_vec);
}

/*****************************************************************************
 ****                                                                     ****
 **** LPBIT_VEC bv_pack(unpacked,length)                                  ****
 ****                                                                     ****
 **** LPSTR unpacked;                                                     ****
 **** int length;                                                         ****
 ****                                                                     ****
 **** This routine takes an array _arr_ of zero and one characters        ****
 **** and returns a packed bit vector suitable for use with the other     ****
 **** routines in this library.                                           ****
 *****************************************************************************/

public LPBIT_VEC FAR PASCAL
bv_pack(unpacked,length)

LPSTR unpacked;
int length;

{
	LPBIT_VEC out_vec;
	LPSTR out_ptr;
	int i;
	int j;
	int bitptr;

	/* Create the structure */

	out_vec = (LPBIT_VEC) farmalloc((unsigned)sizeof(bit_vec));
	MEMCHECK(out_vec);

	out_vec -> len = length;
	out_vec -> bv = farmalloc((unsigned) (length + BYTE - 1) / BYTE);
	MEMCHECK(out_vec -> bv);

	/* Pack the vector */

	out_ptr = out_vec -> bv;

	bitptr = 0;
	for (i = 0; i < (length + BYTE - 1) / BYTE; i++)
	{
		*out_ptr = 0;

		for (j = 0; j < BYTE; j++)
		{
			if (bitptr < length)
			{
			   *out_ptr |= (unpacked[bitptr] << j);
			   bitptr++;
			}
			else
			{
				break;
			}
		}
		out_ptr++;
	}

	/* Return the vector */

	return(out_vec);
}

/*****************************************************************************
 ****                                                                     ****
 **** int bv_diff(v1,v2)                                                  ****
 ****                                                                     ****
 **** LPBIT_VEC v1;                                                       ****
 **** LPBIT_VEC v2;                                                       ****
 ****                                                                     ****
 **** This function returns the number of bits that are unequal in the    ****
 **** two vectors. They must be the same number of bits in each vector    ****
 *****************************************************************************/

public int FAR PASCAL
bv_diff(v1,v2)

LPBIT_VEC v1;
LPBIT_VEC v2;
{
	int diff = 0;
	int i;

	assert (v1 -> len == v2 -> len);
	for (i = 0; i < v1 -> len; i++)
	{
		if (bv_extract(i,v1) != bv_extract(i,v2))
		{
			diff++;
		}
	}

	return(diff);
}

/*****************************************************************************
 ****                                                                     ****
 **** LPBIT_VEC bv_concat(n,vectors)                                      ****
 ****                                                                     ****
 **** int n;                                                              ****
 **** LPBIT_VEC *vectors;                                                 ****
 ****                                                                     ****
 **** Synopsis:                                                           ****
 ****                                                                     ****
 **** Returns the bit vector which is the string concatenation of each    ****
 **** bit vector in _vector_; _n_ is the number of elements in _vector_.  ****
 *****************************************************************************/

public LPBIT_VEC FAR PASCAL
bv_concat(n,vector)

int n;
LPBIT_VEC FAR *vector;

{
	int size;
	LPBIT_VEC out_vec;
	LPSTR str;
	int i;
	int j;
	int count;

	/* Work out how big the new vector will be */

	size = 0;
	for (i = 0; i < n; i++)
	{
		size += vector[i] -> len;
	}

	/* Unpack the input vectors */

	str = farmalloc((unsigned) size);
	MEMCHECK(str);

	count = 0;
	for (i = 0; i < n; i++)
	{
		for (j = 0; j < vector[i] -> len; j++)
		{
			str[count] = bv_extract(j,vector[i]);
			count++;
		}
	}

	out_vec = bv_pack(str,size);

	farfree(str);

	return(out_vec);
}

/*****************************************************************************
 ****                                                                     ****
 **** LPBIT_VEC bv_copy(vector)                                           ****
 ****                                                                     ****
 **** LPBIT_VEC vector;                                                   ****
 ****                                                                     ****
 **** Synopsis:                                                           ****
 ****                                                                     ****
 **** Returns a copy of the bit vector _vector_.                          ****
 *****************************************************************************/

public LPBIT_VEC FAR PASCAL
bv_copy(vector)

LPBIT_VEC vector;
{
	LPBIT_VEC out_vec;
	int i;

	/* Work out how big the new vector will be */

	out_vec = (LPBIT_VEC) farmalloc((unsigned) sizeof(bit_vec));
	MEMCHECK(out_vec);

	out_vec -> len = vector -> len;
	out_vec -> bv = farmalloc((unsigned) (vector -> len + BYTE - 1) / BYTE);
	MEMCHECK(out_vec -> bv);

	for (i = 0; i < (vector -> len + BYTE - 1) / BYTE; i++)
	{
		out_vec -> bv[i] = vector -> bv[i];
	}

	return(out_vec);
}

/*****************************************************************************
 ****                                                                     ****
 **** void bv_set(n,vec,bit)                                              ****
 ****                                                                     ****
 **** int n;                                                              ****
 **** LPBIT_VEC vec;                                                      ****
 **** BOOL bit;                                                           ****
 ****                                                                     ****
 **** Synopsis:                                                           ****
 ****                                                                     ****
 **** Sets bit _n_ in _vec_ to have the value in _bit_.                   ****
 *****************************************************************************/

public void FAR PASCAL
bv_set(n,vec,bit)

int n;
LPBIT_VEC vec;
BOOL bit;

{
	char mask;
	LPSTR b;
	assert(n < vec -> len);

	mask = 0x1;
	b = (vec -> bv) + ((int) (n / BYTE));

	if (bit)
	{
		*b |= (mask << (n % BYTE));
	}
	else
	{
		*b &= ~(mask << (n % BYTE));
	}
}

/*****************************************************************************
 ****                                                                     ****
 **** BOOL bv_extract(n,vec)                                              ****
 ****                                                                     ****
 **** int n;                                                              ****
 **** LPBIT_VEC vec;                                                      ****
 ****                                                                     ****
 **** Synopsis:                                                           ****
 ****                                                                     ****
 **** Returns the _n_th bit of _vec_.                                     ****
 *****************************************************************************/

public BOOL FAR PASCAL
bv_extract(n,vec)

int n;
LPBIT_VEC vec;

{
	register int mask = 0x1;

	assert(n < vec -> len);
	return(((*((vec -> bv) + ((int) (n / BYTE)))) & (mask << (n % BYTE))) != 0);
}

/*****************************************************************************
 ****                                                                     ****
 **** int bv_print(stream, vector)                                        ****
 ****                                                                     ****
 **** FILE *vector;                                                       ****
 **** LPBIT_VEC vector;                                                   ****
 ****                                                                     ****
 **** Synopsis:                                                           ****
 ****                                                                     ****
 **** Prints out _vector_ in binary, MSB is the rightmost.                ****
 **** Returns 0 for success, 1 on failure.                                ****
 *****************************************************************************/

public int FAR PASCAL
bv_print(stream, vector)

FILE *stream;
LPBIT_VEC vector;

{
	LPSTR ptr; /* Points to the current char */
	char mask;
	int bits; /* Counts the number of bits output */
	int  i;

	bits = 0;
	for (ptr = vector -> bv; bits < vector -> len; ptr++)

	{
		mask = 0x1;
		for (i = 0; i < BYTE; i++)
		{
			if (fprintf(stream, (*ptr & mask) ? "1" : "0") == EOF)
			{
				return(1);
			}
			bits++;
			if (bits == vector -> len)
			{
				break;
			}
			mask <<= 1;
		}
	}
	return(0);
}

/*****************************************************************************
 ****                                                                     ****
 **** void bv_free(vector)                                                ****
 ****                                                                     ****
 **** LPBIT_VEC vector;                                                   ****
 ****                                                                     ****
 **** Synopsis:                                                           ****
 ****                                                                     ****
 **** Frees the memory used by _vector_                                   ****
 *****************************************************************************/

public void FAR PASCAL
bv_free(vector)

LPBIT_VEC vector;

{
	farfree(vector -> bv);
	farfree(vector);
}

/*****************************************************************************
 ****                                                                     ****
 **** BOOL bv_equal(v1,v2)                                                ****
 ****                                                                     ****
 **** LPBIT_VEC v1;                                                       ****
 **** LPBIT_VEC v2;                                                       ****
 ****                                                                     ****
 **** Synopsis:                                                           ****
 ****                                                                     ****
 **** Returns TRUE if each bit in v1 and v2 have the same value in the    ****
 **** same position. It returns FALSE otherwise.                          ****
 *****************************************************************************/

public BOOL FAR PASCAL
bv_equal(v1,v2)

LPBIT_VEC v1;
LPBIT_VEC v2;

{
	bool_t eq;
	int i;

	eq = TRUE;

	if (v1 -> len != v2 -> len)
	{
		eq = FALSE;
	}
	else
	{
		for (i = 0; i < (v1 -> len + BYTE - 1) / BYTE; i++)
		{
			if (v1 -> bv[i] != v2 -> bv[i])
			{
				eq = FALSE;
				break;
			}
		}
	}

	return(eq);
}