
/*****************************************************************************
 ****                                                                     ****
 **** lf.c                                                                ****
 ****                                                                     ****
 **** atree release 2.7 for Windows                                       ****
 **** Adaptive Logic Network (ALN) simulation program.                    ****
 **** Copyright (C) A. Dwelly, R. Manderscheid, M. Thomas, W.W. Armstrong ****
 ****               1991, 1992                                            ****
 ****                                                                     ****
 **** License:                                                            ****
 **** A royalty-free license is granted for the use of this software for  ****
 **** NON_COMMERCIAL PURPOSES ONLY. The software may be copied and/or     ****
 **** modified provided this notice appears in its entirety and unchanged ****
 **** in all derived source programs.  Persons modifying the code are     ****
 **** requested to state the date, the changes made and who made them     ****
 **** in the modification history.                                        ****
 ****                                                                     ****
 **** Patent License:                                                     ****
 **** The use of a digital circuit which transmits a signal indicating    ****
 **** heuristic responsibility is protected by U. S. Patent 3,934,231     ****
 **** and others assigned to Dendronic Decisions Limited of Edmonton,     ****
 **** W. W. Armstrong, President.  A royalty-free license is granted      ****
 **** by the company to use this patent for NON_COMMERCIAL PURPOSES to    ****
 **** adapt logic trees using this program and its modifications.         ****
 ****                                                                     ****
 **** Limited Warranty:                                                   ****
 **** This software is provided "as is" without warranty of any kind,     ****
 **** either expressed or implied, including, but not limited to, the     ****
 **** implied warrantees of merchantability and fitness for a particular  ****
 **** purpose.  The entire risk as to the quality and performance of the  ****
 **** program is with the user.  Neither the authors, nor the             ****
 **** University of Alberta, its officers, agents, servants or employees  ****
 **** shall be liable or responsible in any way for any damage to         ****
 **** property or direct personal or consequential injury of any nature   ****
 **** whatsoever that may be suffered or sustained by any licensee, user  ****
 **** or any other party as a consequence of the use or disposition of    ****
 **** this software.                                                      ****
 **** Modification history:                                               ****
 ****                                                                     ****
 **** 90.09.05 Initial implementation, A.Dwelly                           ****
 **** 91.04.15 Port to PC and minor bug fixes, R. Manderscheid            ****
 **** 91.05.20 Windows Port & Windows Extensions, M. Thomas               ****
 **** 91.07.17 atree v2.0 for Windows, M. Thomas                          ****
 **** 92.04.27 atree v2.5 for Windows, M. Thomas                          ****
 **** 92.03.07 Release 2.6, Monroe Thomas                                 ****
 **** 92.01.08 Release 2.7, Monroe Thomas                                 ****
 ****                                                                     ****
 *****************************************************************************/

#include <windows.h>
#include <stdio.h>
#include <stdlib.h>
#include "atree.h"
#include "lf.h"

#define VERBOSITY 0

extern int errno;
extern char * sys_errlist[];

extern FILE *yyin;
extern int line_no;

/* for use in winsynan.c in Windows version */
float far * far *tmp_table;

BOOL lf_quit = FALSE;

prog_t prog;
BOOL fold_flag; /* true if saving folded trees */
			/* remaining flags true if corresponding statement
					 has been specified. */
BOOL test_size_flag;
BOOL train_size_flag;
BOOL largest_flag;
BOOL smallest_flag;
BOOL code_flag;
BOOL quant_flag;

HWND hlfStatus;

#pragma argsused
long FAR PASCAL lfStatusDlgProc(HWND hwnd, WORD message, WORD wParam, LONG lParam)
{
	if ((message == WM_COMMAND) && (wParam == IDCANCEL))
	{
		lf_quit = TRUE;
		return(TRUE);
	}
	return(FALSE);
}

void PASCAL
prog_init(HANDLE hInstance, HWND hwnd)
{
	atree_init(hInstance, hwnd);

	train_size_flag = FALSE;
	test_size_flag = FALSE;
	largest_flag = FALSE;
	smallest_flag = FALSE;
	code_flag = FALSE;
	quant_flag = FALSE;
	fold_flag = FALSE;

	prog.forest_folded = FALSE;
	prog.testset_sz = 0;
	prog.trainset_sz = 0;
	prog.tree_sz = 0;
	prog.min_correct = 0;
	prog.vote = 1;
	prog.max_epochs = 0;
	prog.test_table = NULL;
	prog.train_table = NULL;
	prog.code = NULL;
	prog.walk_step = NULL;
	prog.save_tree = NULL;
	prog.load_tree = NULL;
	prog.save_code = NULL;
	prog.load_code = NULL;

	tmp_table = NULL;
}

int PASCAL
read_prog(fp)

FILE *fp;

{
	yyin = fp;
	line_no = 1;

	if(yyparse())       /* returns 1 if fatal error */
		return(TRUE);
	else
		return(FALSE);
}

BOOL set_encodings(HWND hwnd)
{
	char szBuffer[80];
	int dim;
	int i;

	if (prog.load_code == NULL && (!code_flag || !quant_flag))
	{
		wsprintf(szBuffer, "either coding+quantization or code load file must be specified.\n");
		MessageBox(hwnd, szBuffer, "lf process_prog()", MB_OK | MB_ICONEXCLAMATION);
		return(0);
	}

	for (dim = 0; dim < prog.total_dimensions; dim++)
	{
		prog.code[dim].vector = NULL;
	}

	if (prog.load_code)
	{
		FILE *fp;

		if ((fp = fopen(prog.load_code, "r")) == NULL)
		{
			wsprintf(szBuffer,"Can't open %s ", (LPSTR)prog.load_code);
			MessageBox(NULL, szBuffer, "LF File Error", MB_OK);
			return(0);
		}
		for (dim = 0; dim < prog.total_dimensions; dim++)
		{

			if (atree_read_code((FILE far *)fp, &prog.code[dim]) == NULL)
			{
				wsprintf(szBuffer, "too few codings in file '%s'\n",(LPSTR)prog.load_code);
				MessageBox(hwnd, szBuffer, "lf process_prog()", MB_OK | MB_ICONEXCLAMATION);
				return(0);
			}
		}
		fclose(fp);
	}
	else
	{
		for (dim = 0; dim < prog.total_dimensions; dim++)
		{
			if (!largest_flag)
			{
				prog.code[dim].high = prog.train_table[dim][0];

				for (i = 1; i < prog.trainset_sz; i++)
				{
					if (prog.train_table[dim][i] > prog.code[dim].high)
					{
						prog.code[dim].high = prog.train_table[dim][i];
					}
				}

				for (i = 0; i < prog.testset_sz; i++)
				{
					if (prog.test_table[dim][i] > prog.code[dim].high)
					{
						prog.code[dim].high = prog.test_table[dim][i];
					}
				}
			}
			if (!smallest_flag)
			{
				prog.code[dim].low = prog.train_table[dim][0];

				for (i = 1; i < prog.trainset_sz; i++)
				{
					if (prog.train_table[dim][i] < prog.code[dim].low)
					{
						prog.code[dim].low = prog.train_table[dim][i];
					}
				}

				for (i = 0; i < prog.testset_sz; i++)
				{
					if (prog.test_table[dim][i] < prog.code[dim].low)
					{
						prog.code[dim].low = prog.test_table[dim][i];
					}
				}
			}

			if (prog.code[dim].high <= prog.code[dim].low)
			{
				wsprintf(szBuffer, "largest value must be greater than smallest value, column %d\n", dim + 1);
				MessageBox(hwnd, szBuffer, "lf set_encodings()", MB_OK | MB_ICONEXCLAMATION);
				return(0);
			}

			/* finish constructing code for this dimension */

			if (atree_set_code(&prog.code[dim], prog.code[dim].low,
												prog.code[dim].high,
												prog.code[dim].vector_count,
												prog.code[dim].width,
												prog.walk_step[dim]))
			{
				wsprintf(szBuffer,"random walk failed for dimension %d\n",dim + 1);
				MessageBox(hwnd, szBuffer, "lf set_encodings", MB_OK | MB_ICONEXCLAMATION);
				return(0);
			}
		} /* for (dim...) */
	}
	return(TRUE);
}

BOOL PASCAL
process_prog(HWND hwnd, HANDLE hInstance)

{
	int dim;
	int voter;
	int i;
	LPATREE tree;
	LPBIT_VEC far *concat;
	LPBIT_VEC domain_set;
	LPBIT_VEC codomain_set;
	FILE *save_tree_fp = NULL;
	FILE *save_code_fp = NULL;
	FILE *load_tree_fp = NULL;
	char szBuffer[80];

	/*
	 * When this function is called, the program has been read in, and
	 * the relevant details have been stored in the global structure
	 * 'prog'. We have some semantic processing to do, then the trees
	 * can be trained as specified.
	 */

	if (prog.tree_sz == 0 && prog.load_tree == NULL)
	{
		wsprintf(szBuffer, "either tree size or tree load file must be specified\n");
		MessageBox(hwnd, szBuffer, "lf process_prog()", MB_OK | MB_ICONEXCLAMATION);
		return(0);
	}

	if ((prog.load_tree != NULL && (prog.save_tree != NULL))
	&& (!strcmpi(prog.load_tree, prog.save_tree)))
	{
		wsprintf(szBuffer, "Save tree and load tree filenames must be different.\n");
		MessageBox(hwnd, szBuffer, "lf process_prog()", MB_OK | MB_ICONEXCLAMATION);
		return(0);
	}

	if (prog.load_tree != NULL
	&& (load_tree_fp = fopen(prog.load_tree, "r")) == NULL)
	{
		wsprintf(szBuffer,"Can't open %s ", (LPSTR)prog.load_tree);
		MessageBox(NULL, szBuffer, "LF File Error", MB_OK);
		return(0);
	}

	if (prog.save_tree != NULL
	&& (save_tree_fp = fopen(prog.save_tree, "w")) == NULL)
	{
		wsprintf(szBuffer,"Can't open %s ", (LPSTR)prog.save_tree);
		MessageBox(NULL, szBuffer, "LF File Error", MB_OK);
		return(0);
	}

	if (save_tree_fp == NULL && (save_tree_fp = fopen("$$$lf$$$.tmp", "w")) == NULL)
	{
		wsprintf(szBuffer,"Can't open LF swap file $$$lf$$$.tmp");
		MessageBox(NULL, szBuffer, "LF File Error", MB_OK);
		return(0);
	}

	if (prog.save_code != NULL
	&& (save_code_fp = fopen(prog.save_code, "w")) == NULL)
	{
		wsprintf(szBuffer,"Can't open %s ", (LPSTR)prog.save_code);
		MessageBox(NULL, szBuffer, "LF File Error", MB_OK);
		return(0);
	}

	hlfStatus = CreateDialog(hInstance, "lfStatus", hwnd,
								 MakeProcInstance((FARPROC)lfStatusDlgProc, hInstance));

	SetDlgItemText(hlfStatus, IDD_TEXT, "Building training data...");
	SetDlgItemText(hlfStatus, IDD_PERCENT, "0 %");
	EnableWindow(GetDlgItem(hlfStatus, IDCANCEL), FALSE);

	if (!set_encodings(hwnd))
	{
		if (IsWindow(hlfStatus)) DestroyWindow(hlfStatus);
		return(0);
	}

	domain_set = (LPBIT_VEC) Malloc((unsigned) sizeof(bit_vec) *
									prog.trainset_sz);
	MEMCHECK(domain_set);
	codomain_set = (LPBIT_VEC) Malloc((unsigned) sizeof(bit_vec) *
									  prog.trainset_sz);
	MEMCHECK(codomain_set);

	prog.domain_width = 0;
	prog.codomain_width = 0;
	for (dim = 0; dim < prog.total_dimensions; dim++)
	{
		if (dim < prog.dimensions)
		{
			prog.domain_width += prog.code[dim].width;
		}
		else
		{
			prog.codomain_width += prog.code[dim].width;
		}
	}

	/*
	 * code[i] covers the dimension i.
	 * we now create the training set of bit vectors.
	 */

	concat = (LPBIT_VEC FAR *) Malloc((unsigned)(prog.total_dimensions) *
								  sizeof(LPBIT_VEC));
	MEMCHECK(concat);

	for (i = 0; i < prog.trainset_sz; i++)
	{
		for (dim = 0; dim < prog.total_dimensions; dim++)
		{
			concat[dim] = prog.code[dim].vector
						  + atree_encode(prog.train_table[dim][i],
										 &prog.code[dim]);
		}
		domain_set[i] = *(bv_concat(prog.dimensions, concat));
		codomain_set[i] = *(bv_concat(prog.codimensions,
									  concat + prog.dimensions));

		sprintf(szBuffer, "%3.1f %%",((float)i / prog.trainset_sz) * 100);
		SetDlgItemText(hlfStatus, IDD_PERCENT, (LPSTR)szBuffer);
	}

	Free(concat);

	if (IsWindow(hlfStatus))	DestroyWindow(hlfStatus);

	/*
	 * Train the trees.
	 */

	if (prog.max_epochs > 0)
	{
		for (i = 0; i < prog.codomain_width; i++)
		{
			for (voter = 0; voter < prog.vote; voter++)
			{
				if (lf_quit) return(0);
				if (load_tree_fp != NULL)
				{
					if ((tree = atree_read((FILE far *)load_tree_fp)) == NULL)
					{
						sprintf(szBuffer, "too few trees in file '%s'\n",(LPSTR)prog.load_tree);
						MessageBox(hwnd, szBuffer, "lf process_prog()", MB_OK | MB_ICONEXCLAMATION);
						return(0);
					}
				}
				else
				{
					tree = atree_create(prog.domain_width,prog.tree_sz);
				}

				if (atree_train(tree, domain_set, codomain_set,
								   i, prog.trainset_sz, prog.min_correct,
								   prog.max_epochs, 1) == -1)
				{
					atree_free(tree);
					lf_quit = TRUE;
					if (load_tree_fp != NULL)
					{
						fclose(load_tree_fp);
					}

					if (save_tree_fp != NULL)
					{
						fclose(save_tree_fp);
					}
					return(1);
				}

				if (prog.save_tree)
				{
					if (fold_flag)
					{
						tree = atree_fold(tree);
					}
				}
				else
				{
					/* fold tree for swap file if not already done */
					tree = atree_fold(tree);
				}
				atree_write((FILE far *)save_tree_fp, tree);
				atree_free(tree);
			}
		}
	}

	if (fold_flag || (!prog.save_tree))
	{
		prog.forest_folded = TRUE;
	}

	if (load_tree_fp != NULL)
	{
		fclose(load_tree_fp);
	}

	if (save_tree_fp != NULL)
	{
		fclose(save_tree_fp);
	}

	/*
	 * Save codings
	 */

	if (save_code_fp != NULL)
	{
		int dim;

		for (dim = 0; dim < prog.total_dimensions; dim++)
		{
			atree_write_code((FILE far *)save_code_fp, &prog.code[dim]);
		}
		fclose(save_code_fp);
	}

	for (i = 0; i < prog.trainset_sz; i++)
	{
		Free(domain_set[i].bv);
		Free(codomain_set[i].bv);
	}
	Free(domain_set);
	Free(codomain_set);
	return(1);
}

BOOL PASCAL
test_prog(HWND hwnd, LPSTR szOutFile, HANDLE hInstance)

{
	char szBuffer[80];
	int i, j;
	int dim;
	int col;
	int nv;
	LPATREE FAR *trees;
	LPFAST_TREE *ftrees;
	LPBIT_VEC FAR *test_vec;
	LPBIT_VEC FAR * FAR *result;
	LPBIT_VEC FAR *concat;
	FILE *load_tree_fp = NULL;
	FILE* out_fp;
	int far * far * histogram;
	int total_samples = 0;
	float total = 0;

	if(prog.trainset_sz == 0)
	{
		// get trees from load file if no training set
		if ((load_tree_fp = fopen(prog.load_tree, "r")) == NULL)
		{
			wsprintf(szBuffer,"Can't open %s ", (LPSTR)prog.load_tree);
			MessageBox(NULL, szBuffer, "LF File Error", MB_OK);
			return(FALSE);
		}
	}

	// else get trees from save file if specified
	if (load_tree_fp == NULL && prog.save_tree != NULL
	&& (load_tree_fp = fopen(prog.save_tree, "r")) == NULL)
	{
		wsprintf(szBuffer,"Can't open %s ", (LPSTR)prog.save_tree);
		MessageBox(NULL, szBuffer, "LF File Error", MB_OK);
		return(FALSE);
	}

	// else get trees from swap file
	if (load_tree_fp == NULL
	&&(load_tree_fp = fopen("$$$lf$$$.tmp", "r")) == NULL)
	{
		wsprintf(szBuffer,"Can't open LF swap file $$$lf$$$.tmp");
		MessageBox(NULL, szBuffer, "LF File Error", MB_OK);
		return(FALSE);
	}

	if ((out_fp = fopen(szOutFile, "w")) == NULL)
	{
	   wsprintf(szBuffer,"Can't open %s ", (LPSTR)szOutFile);
	   MessageBox(hwnd, szBuffer, "LF File Error", MB_OK);
	   return(FALSE);
	}

	hlfStatus = CreateDialog(hInstance, "lfStatus", hwnd,
							 MakeProcInstance((FARPROC)lfStatusDlgProc, hInstance));
	SetDlgItemText(hlfStatus, IDD_TEXT, "Evaluating test data...");
	SetDlgItemText(hlfStatus, IDD_PERCENT, "0 %");

	if (prog.trainset_sz == 0 && !set_encodings(hwnd))
	{
		if (IsWindow(hlfStatus)) DestroyWindow(hlfStatus);
		return(FALSE);
	}

	trees = (LPATREE FAR *) Malloc((unsigned)(prog.vote) * sizeof(LPATREE));
	MEMCHECK(trees);

	ftrees = (LPFAST_TREE FAR *) Malloc((unsigned)(prog.vote) * sizeof(LPFAST_TREE));
	MEMCHECK(ftrees);

	/*
	 * We need a result vector for each codomain dimension
	 * because bv_diff expects vectors of equal length.
	 */
	result = (LPBIT_VEC FAR * FAR *)
			 Malloc((unsigned)(prog.codimensions) * sizeof(LPBIT_VEC FAR *));
	MEMCHECK(result);
	for (dim = prog.dimensions; dim < prog.total_dimensions; dim++)
	{
		result[dim - prog.dimensions] = (LPBIT_VEC FAR *) Malloc((unsigned)(prog.testset_sz) *
										 sizeof(LPBIT_VEC));
		MEMCHECK(result);
		for (i = 0; i < prog.testset_sz; i++)
		{
			result[dim - prog.dimensions][i] = bv_create(prog.code[dim].width);
		}
	}

	/*
	 * Create test vectors and print out test info.
	 * Remember to bv_free test_vec after each iteration.
	 */

	concat = (LPBIT_VEC FAR *) Malloc((unsigned)(prog.dimensions) *
								sizeof(LPBIT_VEC));
	MEMCHECK(concat);

	test_vec = (LPBIT_VEC FAR *) Malloc((unsigned)(prog.testset_sz) *
										 sizeof(LPBIT_VEC));
	MEMCHECK(test_vec);

	for (i = 0; i < prog.testset_sz; i++)
	{
		for (dim = 0; dim < prog.total_dimensions; dim++)
		{
			nv = atree_encode(prog.test_table[dim][i], &prog.code[dim]);
			if (dim < prog.dimensions)
			{
				concat[dim] = prog.code[dim].vector + nv;
			}
		}
		test_vec[i] = bv_concat(prog.dimensions, concat);
	}
	Free(concat);


	for (dim = prog.dimensions; dim < prog.total_dimensions; dim++)
	{
		total_samples += prog.code[dim].width * prog.testset_sz;
	}

	/*
	 * for each codomain dimension ...
	 */
	for (dim = prog.dimensions; dim < prog.total_dimensions; dim++)
	{
		int bit_no;

		for (bit_no = 0; bit_no < prog.code[dim].width; bit_no++)
		{
			for (i = 0; i < prog.vote; i++)
			{
				Windows_Interrupt(200);

				if ((trees[i] = atree_read((FILE far *)load_tree_fp)) == NULL)
				{
					sprintf(szBuffer, "too few trees in file '%s'\n",(LPSTR)prog.load_tree);
					MessageBox(hwnd, szBuffer, "lf process_prog()", MB_OK | MB_ICONEXCLAMATION);
					if (IsWindow(hlfStatus)) DestroyWindow(hlfStatus);
					return(FALSE);
				}
				if (!prog.forest_folded)
				{
					trees[i] = atree_fold(trees[i]);
				}
				ftrees[i] = atree_compress(trees[i]);
				atree_free(trees[i]);
			}

			/*
			 * for each element of the training set ...
			 */

			for (i = 0; i < prog.testset_sz; i++)
			{
				int weight = 0;
				int voter;

				total ++;

				sprintf(szBuffer, "%3.1f %%", (total / total_samples) * 100.0);
				SetDlgItemText(hlfStatus, IDD_PERCENT, (LPSTR)szBuffer);

				/*
				 * Calculate result for this dimension.
				 */

				for (voter = 0; voter < prog.vote; voter++)
				{
					if (atree_fast_eval(ftrees[voter], test_vec[i]))
					{
						weight++;
					}
				}
				bv_set(bit_no, result[dim - prog.dimensions][i], weight > prog.vote / 2);
			}
		} /* for (bit_no...) */

		for (i = 0; i < prog.vote; i++)
		{
			Free(ftrees[i]);
		}

		if (lf_quit)
		{
			break;
		}

	} /* for (dim...) */

	/* set up histogram buckets for each codim */

#define NUM_BUCKETS 10

	histogram = (int far * far *) Malloc((unsigned)(prog.codimensions) * sizeof(int far *));
	MEMCHECK(histogram);

	for (dim = 0; dim < prog.codimensions; dim++)
	{
		histogram[dim] = (int far *) Malloc((unsigned)NUM_BUCKETS * sizeof(int));
		MEMCHECK(histogram[dim]);
		for (i = 0; i < NUM_BUCKETS; i++)
		{
			histogram[dim][i] = 0;
		}
	}

	/* perform output */

	if (!lf_quit)
	{
		fprintf(out_fp,"%d\n", prog.codimensions);
		for (i = 0; i < prog.testset_sz; i++)
		{
			for (dim = 0; dim < prog.total_dimensions; dim++)
			{
				int closest;

				nv = atree_encode(prog.test_table[dim][i], &prog.code[dim]);
				fprintf(out_fp, "%s%f %d", dim ? "\t" : "", prog.test_table[dim][i], nv);

				if (dim >= prog.dimensions)
				{
					closest = atree_decode(result[dim - prog.dimensions][i], &prog.code[dim]);

					fprintf(out_fp, "\t%f %d",
						prog.code[dim].low + prog.code[dim].step * closest, closest);

					closest = abs(closest - nv);
					if (closest < (NUM_BUCKETS - 1))
					{
						histogram[dim - prog.dimensions][closest] += 1;
					}
					else
					{
						histogram[dim - prog.dimensions][NUM_BUCKETS - 1] += 1;
					}
				}
			}
			fprintf(out_fp,"\n");
		} /* for i... */

		fprintf(out_fp,"\nERROR HISTOGRAM\n");

		for (i = 0; i < NUM_BUCKETS; i++)
		{
			if (i < (NUM_BUCKETS - 1))
				fprintf(out_fp, "Out by %d  levels", i);
			else
				fprintf(out_fp, "Out by %d+ levels", i);
			for (dim = 0; dim < prog.codimensions; dim++)
			{
				fprintf(out_fp, "\t%d", histogram[dim][i]);
			}
			fprintf(out_fp,"\n");
		}
	}

	/* clean up */

	Free(trees);
	Free(ftrees);

	for (dim = 0; dim < prog.codimensions; dim++)
	{
		for (i = 0; i < prog.testset_sz; i++)
		{
			bv_free(result[dim][i]);
		}
		Free(histogram[dim]);
		Free(result[dim]);
	}
	Free(histogram);
	Free(result);

	for (i = 0; i < prog.testset_sz; i++)
	{
		bv_free(test_vec[i]);
	}
	Free(test_vec);

	fclose(out_fp);       /* close output file */
	fclose(load_tree_fp);
	if (IsWindow(hlfStatus)) DestroyWindow(hlfStatus);
	return(TRUE);
}

BOOL PASCAL
lf_main(HWND hwnd, LPSTR szInFile, LPSTR szOutFile, HANDLE hInstance)

{
	FILE *fp;
	char szFileName[80];
	int i, j;
	char szBuffer[80];
	HCURSOR hCursor;

	hCursor = SetCursor(LoadCursor(NULL, IDC_WAIT));
	ShowCursor(TRUE);

	/* Initialise the default values for the program */

	prog_init(hInstance, hwnd);

	if (szInFile == NULL)
	{
		MessageBox(hwnd, "No Input File Specified", "LF File Error", MB_OK);
		return(FALSE);
	}
	if (szOutFile == NULL)
	{
		MessageBox(hwnd, "No Output File Specified", "LF File Error", MB_OK);
		return(0);
	}

	/* Read the input file */

	lstrcpy((LPSTR)szFileName, szInFile);   /* fopen needs near pointer */

	if ((fp = fopen(szFileName,"r")) == NULL)
	{
		char szBuffer[80];
		wsprintf(szBuffer,"Can't open %s ", (LPSTR)szInFile);
		MessageBox(NULL, szBuffer, "LF File Error", MB_OK);
		ShowCursor(FALSE);
		SetCursor(hCursor);
		atree_quit();
		return (FALSE);
	}
	else
	{
		int tmp;
		tmp = read_prog(fp);
		(void) fclose(fp);
		if (tmp || prog.error)
		{
			ShowCursor(FALSE);
			SetCursor(hCursor);
			atree_quit();
			prog.error = FALSE;
			lf_quit = FALSE;
			return(FALSE);
		}
	}

	line_no = 1;

	ShowCursor(FALSE);
	SetCursor(hCursor);

	if (!prog.error && (prog.testset_sz == 0) && (prog.trainset_sz == 0))
	{
		wsprintf(szBuffer, "Cannot have both test and training sets empty.\n");
		MessageBox(hwnd, szBuffer, "lf_main()", MB_OK | MB_ICONEXCLAMATION);
		prog.error = TRUE;
	}

	if (!prog.error && (prog.trainset_sz == 0) && (prog.load_tree == NULL))
	{
		wsprintf(szBuffer,
		"Cannot have empty training set without specifying tree load file.\n");
		MessageBox(hwnd, szBuffer, "lf_main()", MB_OK | MB_ICONEXCLAMATION);
		prog.error = TRUE;
	}

	/* Train the trees as specified if there were no syntax errors */

	if (!prog.error)
	{
		if (prog.trainset_sz != 0)
		{
			if(!process_prog(hwnd, hInstance))
			{
				prog.error = TRUE;
			}
		}
	}

	/* Execute the trees as specified */

	if (lf_quit)
	{
		prog.error = TRUE;
	}

	if (!prog.error)
	{
		if (prog.testset_sz != 0)
		{
			if(!test_prog(hwnd, szOutFile, hInstance))
			{
				prog.error = TRUE;
			}
		}
	}

	if (lf_quit)
	{
		prog.error = TRUE;
	}

	if (!prog.error)
	{
		if(prog.testset_sz != 0)
		{
			sprintf(szBuffer,"notepad.exe %s", szOutFile);
			WinExec(szBuffer, SW_SHOWNORMAL);
		}
	}

	/* Finish */

	prog.error = FALSE;
	lf_quit = FALSE;
	atree_quit();

	if (prog.save_tree == NULL)
	{
		remove("$$$lf$$$.tmp");
	}

	for (i = 0; i < prog.total_dimensions; i++)
	{
		for (j = 0; j < prog.code[i].vector_count; j++)
		{
			if (prog.code[i].vector != NULL)
			{
				Free(prog.code[i].vector[j].bv);
			}
		}
		if (prog.code[i].vector != NULL)
		{
			Free(prog.code[i].vector);
		}
		if (prog.trainset_sz && prog.train_table[i] != NULL)
		{
			Free(prog.train_table[i]);
		}
		if (prog.testset_sz && prog.test_table[i] != NULL)
		{
			Free(prog.test_table[i]);
		}
	}
	if(prog.train_table != NULL)
	{
		Free(prog.train_table);
	}
	if(prog.test_table != NULL)
	{
		Free(prog.test_table);
	}
	if(tmp_table != NULL)
	{
		Free(tmp_table);
	}
	if(prog.walk_step != NULL)
	{
		Free(prog.walk_step);
	}
	if(prog.code != NULL)
	{
		Free(prog.code);
	}

	return(TRUE);
}
