/*****************************************************************************
 ****                                                                     ****
 **** bitmap.cpp                                                          ****
 ****                                                                     ****
 **** atree release 2.7 for Windows                                       ****
 **** Adaptive Logic Network (ALN) simulation program.                    ****
 **** Copyright (C) M. Thomas, N. Sanche, W.W. Armstrong 1991, 1992       ****
 **** 		                                                			 			 		****
 **** License:                                                            ****
 **** A royalty-free license is granted for the use of this software for  ****
 **** NON_COMMERCIAL PURPOSES ONLY. The software may be copied and/or     ****
 **** modified provided this notice appears in its entirety and unchanged ****
 **** in all derived source programs.  Persons modifying the code are     ****
 **** requested to state the date, the changes made and who made them     ****
 **** in the modification history.                                        ****
 ****                                                                     ****
 **** Patent License:                                                     ****
 **** The use of a digital circuit which transmits a signal indicating    ****
 **** heuristic responsibility is protected by U. S. Patent 3,934,231     ****
 **** and others assigned to Dendronic Decisions Limited of Edmonton,     ****
 **** W. W. Armstrong, President.  A royalty-free license is granted      ****
 **** by the company to use this patent for NON_COMMERCIAL PURPOSES to    ****
 **** adapt logic trees using this program and its modifications.         ****
 ****                                                                     ****
 **** Limited Warranty:                                                   ****
 **** This software is provided "as is" without warranty of any kind,     ****
 **** either expressed or implied, including, but not limited to, the     ****
 **** implied warrantees of merchantability and fitness for a particular  ****
 **** purpose.  The entire risk as to the quality and performance of the  ****
 **** program is with the user.  Neither the authors, nor the             ****
 **** University of Alberta, its officers, agents, servants or employees  ****
 **** shall be liable or responsible in any way for any damage to         ****
 **** property or direct personal or consequential injury of any nature   ****
 **** whatsoever that may be suffered or sustained by any licensee, user  ****
 **** or any other party as a consequence of the use or disposition of    ****
 **** this software.                                                      ****
 **** Modification history:                                               ****
 ****                                                                     ****
 **** 92.04.27 atree v2.5 for Windows, M. Thomas                          ****
 **** 92.03.07 Release 2.6, Monroe Thomas, Neal Sanche                    ****
 **** 92.01.08 Release 2.7, Monroe Thomas, Neal Sanche                    ****
 ****                                                                     ****
 *****************************************************************************/

// bitmap.cpp

#include "bitmap.h"

const double TBitmap_PI = 3.14159265359;

void
TBitmap::Init(HBITMAP hABitmap)
{
	BITMAP bm;
	HDC hdc1, hdc2;

	if (hBitmap != NULL)
	{
		DeleteObject(hBitmap);
		hBitmap = NULL;
	}

	if (hABitmap == NULL)
	{
		hBitmap = NULL;
		return;
	}

	GetObject(hABitmap, sizeof(BITMAP), (LPSTR)&bm);
	hBitmap = CreateBitmap(bm.bmWidth, bm.bmHeight, bm.bmPlanes,
												 bm.bmBitsPixel, NULL);

	hdc1 = CreateCompatibleDC(NULL);
	hdc2 = CreateCompatibleDC(NULL);

	SelectObject(hdc1, hABitmap);
	SelectObject(hdc2, hBitmap);

	BitBlt(hdc2, 0, 0, bm.bmWidth, bm.bmHeight,
				 hdc1, 0, 0, SRCCOPY);

	DeleteDC(hdc1);
	DeleteDC(hdc2);
}

DWORD
TBitmap::GetBits(LPSTR lpBits)
{
	return GetBitmapBits(hBitmap, GetBytes(), lpBits);
}

LONG
TBitmap::SetBits(LPSTR lpBits, DWORD dwCount)
{
	return SetBitmapBits(hBitmap, dwCount, lpBits);
}

DWORD
TBitmap::GetWidth()
{
	BITMAP bm;
	GetObject(hBitmap, sizeof(BITMAP), (LPSTR)&bm);
	return bm.bmWidth;
}

DWORD
TBitmap::GetHeight()
{
	BITMAP bm;
	GetObject(hBitmap, sizeof(BITMAP), (LPSTR)&bm);
	return bm.bmHeight;
}

DWORD
TBitmap::GetPlanes()
{
	BITMAP bm;
	GetObject(hBitmap, sizeof(BITMAP), (LPSTR)&bm);
	return bm.bmPlanes;
}

DWORD
TBitmap::GetBitsPixel()
{
	BITMAP bm;
	GetObject(hBitmap, sizeof(BITMAP), (LPSTR)&bm);
	return bm.bmBitsPixel;
}

DWORD
TBitmap::GetWidthBytes()
{
	BITMAP bm;
	GetObject(hBitmap, sizeof(BITMAP), (LPSTR)&bm);
	return bm.bmWidthBytes;
}

DWORD
TBitmap::GetBytes()
{
	BITMAP bm;
	GetObject(hBitmap, sizeof(BITMAP), (LPSTR)&bm);
	return (bm.bmWidthBytes * bm.bmPlanes * bm.bmHeight);
}

COLORREF
TBitmap::GetDrawColor()
{
	return crDraw;
}

COLORREF
TBitmap::SetDrawColor(COLORREF crColor)
{
	COLORREF tmp = crDraw;
	crDraw = crColor;
	return tmp;
}

COLORREF
TBitmap::GetEraseColor()
{
	return crErase;
}

COLORREF
TBitmap::SetEraseColor(COLORREF crColor)
{
	COLORREF tmp = crErase;
	crErase = crColor;
	return tmp;
}


DWORD
TBitmap::GetPoint(int x, int y)
{
	HDC hdc = CreateCompatibleDC(NULL);
	DWORD retvalue;
	SelectObject(hdc, hBitmap);
	retvalue = GetPixel(hdc, x, y);
	DeleteDC(hdc);
	return retvalue;
}

DWORD
TBitmap::SetPoint(int x, int y, COLORREF crColor)
{
	HDC hdc = CreateCompatibleDC(NULL);
	DWORD retvalue;
	SetBkColor(hdc, crErase);
	SetTextColor(hdc, crDraw);
	SelectObject(hdc, hBitmap);
	retvalue = SetPixel(hdc, x, y, crColor);
	DeleteDC(hdc);
	return retvalue;
}

void
TBitmap::Rotate(float theta, BOOL isRadians)
{
	HBITMAP hbm;
	BITMAP bm;
	HDC hdc1 = CreateCompatibleDC(NULL);
	HDC hdc2 = CreateCompatibleDC(NULL);
	GetObject(hBitmap, sizeof(BITMAP), (LPSTR)&bm);

//	HCURSOR hcursor = LoadCursor(NULL, IDC_WAIT);
//	hcursor = SetCursor(hcursor);

	if (!isRadians)
	{
		theta = theta * TBitmap_PI / 180;
	}

	int c1x, c1y, c2x, c2y;
	double p1x, p1y;
	int p2x, p2y;
	long double c0, c1, c2, c3; // color info
	double a;	// angle from c2 to p2
	double r;	// radius from c2 to p2

	int n;		// biggest dimension

	c1x = c2x = ceil(bm.bmWidth / 2.0) - 1;
	c1y = c2y = ceil(bm.bmHeight / 2.0) - 1;

	hbm = CreateBitmap(bm.bmWidth, bm.bmHeight, bm.bmPlanes,
										 bm.bmBitsPixel, NULL);

	SelectObject(hdc1, hBitmap);
	SelectObject(hdc2, hbm);

	n = (bm.bmWidth > bm.bmHeight) ? bm.bmWidth : bm.bmHeight;
	n = n / 2;

	for(p2x = 0; p2x <= n; p2x++)
	{
		for(p2y = 0; p2y <= n; p2y++)
		{
			if (p2x == 0)
			{
				a = TBitmap_PI / 2.0;
			}
			else
			{
				a = atan2(p2y, p2x);
			}
			r = sqrt((DWORD)p2x*p2x + (DWORD)p2y*p2y);

			// compute rotated position p1
			p1x = r * cos(a + theta);
			p1y = r * sin(a + theta);

			// copy pixels, 4 quadrants at once

			c0 = GetPixel(hdc1, floor(c1x + p1x + .5), floor(c1y + p1y + .5));
			c1 = GetPixel(hdc1, floor(c1x - p1x + .5), floor(c1y - p1y + .5));
			c2 = GetPixel(hdc1, floor(c1x + p1y + .5), floor(c1y - p1x + .5));
			c3 = GetPixel(hdc1, floor(c1x - p1y + .5), floor(c1y + p1x + .5));

			if (c0 != -1)
			{
				SetPixel(hdc2, floor(c2x + p2x + .5), floor(c2y + p2y + .5), c0);
			}
			if (c1 != -1)
			{
				SetPixel(hdc2, floor(c2x - p2x + .5), floor(c2y - p2y + .5), c1);
			}
			if (c2 != -1)
			{
				SetPixel(hdc2, floor(c2x + p2y + .5), floor(c2y - p2x + .5), c2);
			}
			if (c3 != -1)
			{
				SetPixel(hdc2, floor(c2x - p2y + .5), floor(c2y + p2x + .5), c3);
			}
		}	// for p2y

		MSG msg;
		if (PeekMessage(&msg, NULL, 0, 0, PM_REMOVE))
		{
			TranslateMessage(&msg);
			DispatchMessage(&msg);
		}

	} // for p2x

	BitBlt(hdc1, 0, 0, bm.bmWidth, bm.bmHeight,
				 hdc2, 0, 0, SRCCOPY);

	DeleteDC(hdc2);
	DeleteDC(hdc1);
	DeleteObject(hbm);
//	SetCursor(hcursor);
}

void
TBitmapControl::GetWindowClass(WNDCLASS &AWndclass)
{
	TControl::GetWindowClass(AWndclass);
	AWndclass.hbrBackground = GRAY_BRUSH;
}

void
TBitmapControl::Paint(HDC PaintDC, PAINTSTRUCT _FAR & ps)
{
	HBITMAP hOldbm;
	BITMAP bm;

	RECT rc;

	HDC hMemDC = CreateCompatibleDC(PaintDC);

	SetBkColor(PaintDC, crErase);
	SetTextColor(PaintDC, crDraw);

	GetClientRect(HWindow, &rc);
	hOldbm = SelectObject(hMemDC, hBitmap);
	GetObject(hBitmap, sizeof(BITMAP), (LPSTR)&bm);
	StretchBlt(PaintDC, 0, 0, rc.right,
					 rc.bottom,
					 hMemDC, 0, 0,
					 bm.bmWidth,
					 bm.bmHeight, SRCCOPY);
	SelectObject(hMemDC, hOldbm);
	DeleteDC(hMemDC);
}

DWORD
TBitmapControl::GetPoint(int x, int y)
{
	HDC hdc = GetDC(HWindow);
	DWORD retvalue;
	SelectObject(hdc, hBitmap);
	retvalue = GetPixel(hdc, x, y);
	ReleaseDC(HWindow, hdc);
	return retvalue;
}

DWORD
TBitmapControl::SetPoint(int x, int y, COLORREF crColor)
{
	HDC hdc1 = GetDC(HWindow);
	HDC hdc2 = CreateCompatibleDC(hdc1);
	RECT rc;
	BITMAP bm;
	DWORD retvalue;

	SetBkColor(hdc1, crErase);
	SetTextColor(hdc1, crDraw);
	GetClientRect(HWindow, &rc);
	GetObject(hBitmap, sizeof(BITMAP), (LPSTR)&bm);
	SelectObject(hdc2, hBitmap);
	retvalue = SetPixel(hdc2, x, y, crColor);
	StretchBlt(hdc1, 0, 0, rc.right, rc.bottom,
						 hdc2, 0, 0, bm.bmWidth, bm.bmHeight, SRCCOPY);
	DeleteDC(hdc2);
	ReleaseDC(HWindow, hdc1);
	return retvalue;
}

void
TBitmapControl::Reset(short w, short h)
{
	HDC hdc;

	if(hBitmap != NULL)
	{
		DeleteObject(hBitmap);
	}

	hdc = GetDC(HWindow);
	hBitmap = CreateCompatibleBitmap(hdc, w, h);
	ReleaseDC(HWindow,hdc);
}


void
TBitmapDrawControl::GetWindowClass(WNDCLASS &AWndclass)
{
	TBitmapControl::GetWindowClass(AWndclass);
	AWndclass.style |= CS_DBLCLKS;
	AWndclass.hCursor = LoadCursor(GetApplication()->hInstance, "pencilcur");
}

void
TBitmapDrawControl::setdimensions(HDC wndDC)
{
	BITMAP bm;
	RECT rc;
	GetClientRect(HWindow, &rc);
	SetMapMode(wndDC, MM_ANISOTROPIC);
	GetObject(hBitmap, sizeof(BITMAP), (LPSTR)&bm);
	SetWindowExt(wndDC, bm.bmWidth, bm.bmHeight);
	SetViewportExt(wndDC, rc.right, rc.bottom);
	SetViewportOrg(wndDC, 0, 0);
}

void
TBitmapDrawControl::WMLButtonDown(RTMessage Msg)
{
	TBitmapControl::WMLButtonDown(Msg);
	POINT pt;
	HDC wndDC = GetDC(HWindow);
	pt.x = Msg.LP.Lo;
	pt.y = Msg.LP.Hi;
	setdimensions(wndDC);
	DPtoLP(wndDC, &pt, 1);
	ReleaseDC(HWindow, wndDC);
	drawon = TRUE;
	SetPoint(pt.x, pt.y, crDraw);
	TBDCClear = FALSE;

}

void
TBitmapDrawControl::WMLButtonUp(RTMessage Msg)
{
	POINT pt;
	HDC wndDC = GetDC(HWindow);
	pt.x = Msg.LP.Lo;
	pt.y = Msg.LP.Hi;
	setdimensions(wndDC);
	DPtoLP(wndDC, &pt, 1);
	ReleaseDC(HWindow, wndDC);
	drawon = FALSE;
	if (!TBDCClear)
	{
		SetPoint(pt.x, pt.y, crDraw);
	}
	TBDCClear = FALSE;
}

void
TBitmapDrawControl::WMLButtonDblClk(RTMessage Msg)
{
	RECT rc;
	BITMAP bm;
	HBRUSH hbr;

	hbr = GetStockObject(BLACK_BRUSH);

	HDC hdc1 = GetDC(HWindow);
	HDC hdc2 = CreateCompatibleDC(hdc1);

	SetBkColor(hdc1, crErase);
	SetTextColor(hdc1, crDraw);

	GetObject(hBitmap, sizeof(BITMAP), (LPSTR)&bm);
	rc.left = rc. top = 0;
	rc.right = bm.bmWidth;
	rc.bottom = bm.bmHeight;
	SelectObject(hdc2, hBitmap);
	FillRect(hdc2, &rc, hbr);

	GetClientRect(HWindow, &rc);

	StretchBlt(hdc1, 0, 0, rc.right, rc.bottom,
						 hdc2, 0, 0, bm.bmWidth, bm.bmHeight, SRCCOPY);
	DeleteDC(hdc2);
	ReleaseDC(HWindow, hdc1);
	TBDCClear = TRUE;
}

void
TBitmapDrawControl::WMRButtonDown(RTMessage Msg)
{
	POINT pt;
	HDC wndDC = GetDC(HWindow);
	pt.x = Msg.LP.Lo;
	pt.y = Msg.LP.Hi;
	setdimensions(wndDC);
	DPtoLP(wndDC, &pt, 1);
	ReleaseDC(HWindow, wndDC);
	eraseon = TRUE;
	SetPoint(pt.x, pt.y, crErase);
	TBDCRotate = FALSE;
}

void
TBitmapDrawControl::WMRButtonUp(RTMessage Msg)
{
	POINT pt;
	HDC wndDC = GetDC(HWindow);
	pt.x = Msg.LP.Lo;
	pt.y = Msg.LP.Hi;
	setdimensions(wndDC);
	DPtoLP(wndDC, &pt, 1);
	ReleaseDC(HWindow, wndDC);
	eraseon = FALSE;
	if (!TBDCRotate)
	{
		SetPoint(pt.x, pt.y, crErase);
	}
	else
	{
		Rotate(5);
		TBDCRotate = FALSE;
		Refresh();
	}
}

void
TBitmapDrawControl::WMRButtonDblClk(RTMessage Msg)
{
	RECT rc;
	BITMAP bm;
	HBRUSH hbr;

	hbr = GetStockObject(WHITE_BRUSH);

	HDC hdc1 = GetDC(HWindow);
	HDC hdc2 = CreateCompatibleDC(hdc1);

	SetBkColor(hdc1, crErase);
	SetTextColor(hdc1, crDraw);

	GetObject(hBitmap, sizeof(BITMAP), (LPSTR)&bm);
	rc.left = rc. top = 0;
	rc.right = bm.bmWidth;
	rc.bottom = bm.bmHeight;
	SelectObject(hdc2, hBitmap);
	FillRect(hdc2, &rc, hbr);

	GetClientRect(HWindow, &rc);
	StretchBlt(hdc1, 0, 0, rc.right, rc.bottom,
						 hdc2, 0, 0, bm.bmWidth, bm.bmHeight, SRCCOPY);
	DeleteDC(hdc2);
	ReleaseDC(HWindow, hdc1);
}

void
TBitmapDrawControl::WMMouseMove(RTMessage Msg)
{
	POINT pt;
	HDC wndDC = GetDC(HWindow);
	pt.x = Msg.LP.Lo;
	pt.y = Msg.LP.Hi;
	setdimensions(wndDC);
	DPtoLP(wndDC, &pt, 1);
	SendMessage(Parent->HWindow, TBDC_COORDCHANGE, 0, MAKELONG(pt.x, pt.y));
	ReleaseDC(HWindow, wndDC);
	if (eraseon) SetPoint(pt.x, pt.y, RGB(255, 255, 255));
	else if (drawon) SetPoint(pt.x, pt.y, RGB(0, 0, 0));
}