/*****************************************************************************
 ****                                                                     ****
 **** ocr_run.cpp                                                         ****
 ****                                                                     ****
 **** atree release 2.7 for Windows                                       ****
 **** Adaptive Logic Network (ALN) simulation program.                    ****
 **** Copyright (C) M. Thomas, N. Sanche, W.W. Armstrong 1991, 1992       ****
 ****                                                 			  						****
 **** License:                                                            ****
 **** A royalty-free license is granted for the use of this software for  ****
 **** NON_COMMERCIAL PURPOSES ONLY. The software may be copied and/or     ****
 **** modified provided this notice appears in its entirety and unchanged ****
 **** in all derived source programs.  Persons modifying the code are     ****
 **** requested to state the date, the changes made and who made them     ****
 **** in the modification history.                                        ****
 ****                                                                     ****
 **** Patent License:                                                     ****
 **** The use of a digital circuit which transmits a signal indicating    ****
 **** heuristic responsibility is protected by U. S. Patent 3,934,231     ****
 **** and others assigned to Dendronic Decisions Limited of Edmonton,     ****
 **** W. W. Armstrong, President.  A royalty-free license is granted      ****
 **** by the company to use this patent for NON_COMMERCIAL PURPOSES to    ****
 **** adapt logic trees using this program and its modifications.         ****
 ****                                                                     ****
 **** Limited Warranty:                                                   ****
 **** This software is provided "as is" without warranty of any kind,     ****
 **** either expressed or implied, including, but not limited to, the     ****
 **** implied warrantees of merchantability and fitness for a particular  ****
 **** purpose.  The entire risk as to the quality and performance of the  ****
 **** program is with the user.  Neither the authors, nor the             ****
 **** University of Alberta, its officers, agents, servants or employees  ****
 **** shall be liable or responsible in any way for any damage to         ****
 **** property or direct personal or consequential injury of any nature   ****
 **** whatsoever that may be suffered or sustained by any licensee, user  ****
 **** or any other party as a consequence of the use or disposition of    ****
 **** this software.                                                      ****
 **** Modification history:                                               ****
 ****                                                                     ****
 **** 92.04.27 atree v2.5 for Windows, M. Thomas                          ****
 **** 92.03.07 Release 2.6, Monroe Thomas, Neal Sanche                    ****
 **** 92.01.08 Release 2.7, Monroe Thomas, Neal Sanche                    ****
 ****                                                                     ****
 *****************************************************************************/

// ocr_run.cpp

#include "ocr.h"

BOOL
TOcr::vote(LPFAST_TREE ftree[], LPBIT_VEC vec, int voters)
{
	int result = 0;
	for(WORD i = 0; i < voters; i ++)
	{
		result += atree_fast_eval(ftree[i], vec);
	}
	return(result > (voters / 2));
}

void
TOcr::clear_bv(bit_vec bv)
{
	for(WORD i = 0; i < bv.len; i++)
	{
		bv_set(i, &(bv), 0);
	}
}

void
TOcr::write_default()
{
	HANDLE ResHandle;
	FILE* outstream;
	FILE* instream;
	int treedata;
	int voters;

	ResHandle = FindResource(GetApplication()->hInstance,"def_aln","ALN");
	treedata = AccessResource(GetApplication()->hInstance, ResHandle);

	if ((instream = fdopen(treedata, "r")) == NULL)
	{
		BWCCMessageBox(HWindow, "Can't open internal ALN resource", "OCR write_default()",
					 MB_ICONEXCLAMATION | MB_OK);
		exit(0);
	}

	if ((outstream = fopen("ocr.tre", "w")) == NULL)
	{
		BWCCMessageBox(HWindow, "Can't open 'ocr.tre'", "write_default()",
					 MB_ICONEXCLAMATION | MB_OK);
		exit(0);
	}

	HCURSOR hcursor = LoadCursor(NULL, IDC_WAIT);
	hcursor = SetCursor(hcursor);

	fscanf(instream, "%d", &voters);
	fprintf(outstream, "%d\n", voters);

	voters *= 3;

	for (WORD i = 0; i < voters; i++)
	{
		LPATREE atree = atree_read(instream);
		atree_write(outstream, atree);
	}

	fclose(instream);
	fclose(outstream);

	SetCursor(hcursor);
}

#define NUMBITS 512
#define STEPLEVEL 16

void
TOcr::OcrReverse()
{
	typedef	struct tagMemBit
	{
		WORD bitno;
		BOOL flipto;
	} MemBit;

	LPATREE atree;
	LPFAST_TREE* ftreeA;
	LPFAST_TREE* ftreeL;
	LPFAST_TREE* ftreeN;
	int voters;
	bit_vec test;
	bit_vec flipped;		// keeps track of bits already flipped
	WORD letternumber;
	WORD width = LetterBitmap->GetBytes();
	long correct = 0;
	long incorrect = 0;
	long total = 0;
	FILE* instream;
	char szBuffer[10];
	HBITMAP bitmaps[3];

	MemBit far* MemBits;

	TBitmap TestLetter(hLightOn);

	EnableWindow(Start->HWindow, FALSE);
	EnableWindow(Train->HWindow, FALSE);
	EnableWindow(Stop->HWindow, TRUE);
	EnableWindow(Reverse->HWindow, FALSE);
	EnableWindow(Reset->HWindow, FALSE);
	EnableWindow(Quit->HWindow, FALSE);
	EnableWindow(Forget->HWindow, FALSE);
	EnableWindow(EditA->HWindow, FALSE);
	EnableWindow(EditL->HWindow, FALSE);
	EnableWindow(EditN->HWindow, FALSE);
	EnableMenuItem(hSysMenu, SC_CLOSE, MF_GRAYED);

	MemBits = (MemBit far*) farmalloc(NUMBITS * sizeof(MemBits));

	if ((instream = fopen("ocr.tre", "r")) == NULL)
	{
		write_default();
		if ((instream = fopen("ocr.tre", "r")) == NULL)
		{
			BWCCMessageBox(HWindow, "Can't open 'ocr.tre'", "OCR Reverse",
							 MB_ICONEXCLAMATION | MB_OK);
			exit(0);
		}
	}

	HCURSOR hcursor = LoadCursor(NULL, IDC_WAIT);
	hcursor = SetCursor(hcursor);

	fscanf(instream, "%d", &voters);

	ftreeA = (LPFAST_TREE*) farmalloc(voters * sizeof(LPFAST_TREE));
	for (WORD i = 0; i < voters; i++)
	{
		atree = atree_read(instream);
		ftreeA[i] = atree_compress(atree);
		atree_free(atree);
	}

	ftreeL = (LPFAST_TREE*) farmalloc(voters * sizeof(LPFAST_TREE));
	for (i = 0; i < voters; i++)
	{
		atree = atree_read(instream);
		ftreeL[i] = atree_compress(atree);
		atree_free(atree);
	}

	ftreeN = (LPFAST_TREE*) farmalloc(voters * sizeof(LPFAST_TREE));
	for (i = 0; i < voters; i++)
	{
		atree = atree_read(instream);
		ftreeN[i] = atree_compress(atree);
		atree_free(atree);
	}

	SetCursor(hcursor);

	fclose(instream);

	Correct->Clear();
	Incorrect->Clear();

	LetterBitmap->Init(hLetterClear);
	LetterBitmap->Refresh();
	LightBulbA->Init(hLightOff);
	LightBulbA->Refresh();
	LightBulbL->Init(hLightOff);
	LightBulbL->Refresh();
	LightBulbN->Init(hLightOff);
	LightBulbN->Refresh();

	test.len = width * 8;
	test.bv = (LPSTR)farmalloc(width);
	flipped.len = width * 8;
	flipped.bv = (LPSTR)farmalloc(width);

	stop = FALSE;

	bitmaps[0] = LetterA->GetBitmap();
	bitmaps[1] = LetterL->GetBitmap();
	bitmaps[2] = LetterN->GetBitmap();

	while (!stop)
	{
		letternumber = RANDOM(4);
		if (letternumber == 0) TestLetter.Init(bitmaps[0]);
		else if (letternumber == 1) TestLetter.Init(bitmaps[1]);
		else if (letternumber == 2) TestLetter.Init(bitmaps[2]);
		else TestLetter.Init(hLetterClear);

		TestLetter.Rotate((random(11) - 5));
		translate(TestLetter.GetBitmap(), random(3) - 1, random(3) - 1);

		TestLetter.GetBits(test.bv);

		clear_bv(flipped);

		for (int i = 0; i < NUMBITS; i++)
		{
			WORD tmp = random(width * 8);
			Windows_Interrupt(100);
			while (bv_extract(tmp, &(flipped)))
			{
				tmp = random(width * 8);
			}
			bv_set(tmp, &(flipped), 1);
			MemBits[i].bitno = tmp;
			MemBits[i].flipto = bv_extract(MemBits[i].bitno, &(test));
			if (MemBits[i].flipto)
			{
				bv_set(MemBits[i].bitno, &(test), 0);
			}
			else
			{
				bv_set(MemBits[i].bitno, &(test), 1);
			}
		}

		LetterBitmap->SetBits(test.bv, width);
		LetterBitmap->Refresh();

		for(i = (NUMBITS - 1); i >= 0; i -= STEPLEVEL)
		{
			if (stop) break;
			Windows_Interrupt(100);

			for (WORD j = 0; j < STEPLEVEL; j ++)
			{
				bv_set(MemBits[i-j].bitno, &(test), MemBits[i-j].flipto);
			}

			LetterBitmap->SetBits(test.bv, width);

			if (shutdown) break;

			LetterBitmap->Refresh();

			total++;

			WORD numtrees1 = 0;
			WORD treenumber = 0;
			WORD treemask = 0;

			if (vote(ftreeA, &test, voters) == 1)
			{
				numtrees1++;
				treenumber = 0;
				treemask |= 1;
				LightBulbA->Init(hLightOn);
				LightBulbA->Refresh();
			}

			if (vote(ftreeL, &test, voters) == 1)
			{
				numtrees1++;
				treenumber = 1;
				treemask |= 2;
				LightBulbL->Init(hLightOn);
				LightBulbL->Refresh();
			}

			if (vote(ftreeN, &test, voters) == 1)
			{
				numtrees1++;
				treenumber = 2 ;
				treemask |= 4;
				LightBulbN->Init(hLightOn);
				LightBulbN->Refresh();
			}

			if ((numtrees1 == 0) && (letternumber == 3))
			{
				// blank letter
				correct++;
			}
			else if ((numtrees1 != 1) || (treenumber != letternumber))
			{
				// mistake!
				incorrect++;
			}
			else
			{
				correct++;
			}

			sprintf(szBuffer, "%7.2f", 100.0 * (float)correct/total);
			Correct->SetText(szBuffer);
			sprintf(szBuffer, "%7.2f", 100.0 * (float)incorrect/total);
			Incorrect->SetText(szBuffer);

			if (treemask & 1)
			{
				LightBulbA->Init(hLightOff);
				LightBulbA->Refresh();
			}
			if (treemask & 2)
			{
				LightBulbL->Init(hLightOff);
				LightBulbL->Refresh();
			}
			if (treemask & 4)
			{
				LightBulbN->Init(hLightOff);
				LightBulbN->Refresh();
			}
		}
	}

	farfree(MemBits);
	farfree(test.bv);
	farfree(flipped.bv);
	for (i = 0; i < voters; i++)
	{
		farfree(ftreeA[i]);
		farfree(ftreeL[i]);
		farfree(ftreeN[i]);
	}
	farfree(ftreeA);
	farfree(ftreeL);
	farfree(ftreeN);

	EnableWindow(Start->HWindow, TRUE);
	EnableWindow(Train->HWindow, TRUE);
	EnableWindow(Stop->HWindow, FALSE);
	EnableWindow(Reverse->HWindow, TRUE);
	EnableWindow(Reset->HWindow, TRUE);
	EnableWindow(Quit->HWindow, TRUE);
	EnableWindow(Forget->HWindow, TRUE);
	EnableWindow(EditA->HWindow, TRUE);
	EnableWindow(EditL->HWindow, TRUE);
	EnableWindow(EditN->HWindow, TRUE);
	EnableMenuItem(hSysMenu, SC_CLOSE, MF_ENABLED);
}

void
TOcr::OcrStart()
{
	LPATREE atree;
	LPFAST_TREE* ftreeA;
	LPFAST_TREE* ftreeL;
	LPFAST_TREE* ftreeN;
	int voters;
	bit_vec test;
	bit_vec flipped;
	long correct = 0;
	long incorrect = 0;
	long total;
	short letternumber;
	char szBuffer[10];
	FILE* instream;
	HBITMAP bitmaps[3];
	WORD width = LetterBitmap->GetBytes();

	EnableWindow(Start->HWindow, FALSE);
	EnableWindow(Train->HWindow, FALSE);
	EnableWindow(Stop->HWindow, TRUE);
	EnableWindow(Reverse->HWindow, FALSE);
	EnableWindow(Reset->HWindow, FALSE);
	EnableWindow(Quit->HWindow, FALSE);
	EnableWindow(Forget->HWindow, FALSE);
	EnableWindow(EditA->HWindow, FALSE);
	EnableWindow(EditL->HWindow, FALSE);
	EnableWindow(EditN->HWindow, FALSE);
	EnableMenuItem(hSysMenu, SC_CLOSE, MF_GRAYED);

	stop = FALSE;

	if ((instream = fopen("ocr.tre", "r")) == NULL)
	{
		write_default();
		if ((instream = fopen("ocr.tre", "r")) == NULL)
		{
			BWCCMessageBox(HWindow, "Can't open 'ocr.tre'", "OCR Start",
							 MB_ICONEXCLAMATION | MB_OK);
			exit(0);
		}
	}

	HCURSOR hcursor = LoadCursor(NULL, IDC_WAIT);
	hcursor = SetCursor(hcursor);

	fscanf(instream, "%d", &voters);

	ftreeA = (LPFAST_TREE*) farmalloc(voters * sizeof(LPFAST_TREE));
	for (WORD i = 0; i < voters; i++)
	{
		atree = atree_read(instream);
		ftreeA[i] = atree_compress(atree);
		atree_free(atree);
	}

	ftreeL = (LPFAST_TREE*) farmalloc(voters * sizeof(LPFAST_TREE));
	for (i = 0; i < voters; i++)
	{
		atree = atree_read(instream);
		ftreeL[i] = atree_compress(atree);
		atree_free(atree);
	}

	ftreeN = (LPFAST_TREE*) farmalloc(voters * sizeof(LPFAST_TREE));
	for (i = 0; i < voters; i++)
	{
		atree = atree_read(instream);
		ftreeN[i] = atree_compress(atree);
		atree_free(atree);
	}
	SetCursor(hcursor);

	fclose(instream);

	test.len = width * 8;
	test.bv = (LPSTR)farmalloc(width);
	flipped.len = width * 8;
	flipped.bv = (LPSTR)farmalloc(width);

	correct = incorrect = total = 0;

	LetterBitmap->Init(hLetterClear);
	LetterBitmap->Refresh();

	Correct->Clear();
	Incorrect->Clear();

	LightBulbA->Init(hLightOff);
	LightBulbA->Refresh();
	LightBulbL->Init(hLightOff);
	LightBulbL->Refresh();
	LightBulbN->Init(hLightOff);
	LightBulbN->Refresh();

	bitmaps[0] = LetterA->GetBitmap();
	bitmaps[1] = LetterL->GetBitmap();
	bitmaps[2] = LetterN->GetBitmap();

	while (!stop)
	{
#define FORWARDBITS 288

		clear_bv(flipped);

		letternumber = RANDOM(3);
		LetterBitmap->Init(bitmaps[letternumber]);
		LetterBitmap->Rotate((random(11) - 5));
		translate(LetterBitmap->GetBitmap(), random(3) - 1, random(3) - 1);

		if (stop) break;
		LetterBitmap->GetBits(test.bv);
		for(WORD i = 0; i < (FORWARDBITS/16); i++)
		{
			Windows_Interrupt(100);

			if (stop) break;

			total++;

			WORD numtrees1 = 0;
			WORD treenumber = 0;
			WORD treemask = 0;

			if (vote(ftreeA, &test, voters) == 1)
			{
				numtrees1++;
				treenumber = 0;
				treemask |= 1;
				LightBulbA->Init(hLightOn);
				LightBulbA->Refresh();
			}

			if (vote(ftreeL, &test, voters) == 1)
			{
				numtrees1++;
				treenumber = 1;
				treemask |= 2;
				LightBulbL->Init(hLightOn);
				LightBulbL->Refresh();
			}

			if (vote(ftreeN, &test, voters) == 1)
			{
				numtrees1++;
				treenumber = 2 ;
				treemask |= 4;
				LightBulbN->Init(hLightOn);
				LightBulbN->Refresh();
			}

			if ((numtrees1 != 1) || (treenumber != letternumber))
			{
				// mistake!
				incorrect++;
			}
			else
			{
				correct++;
			}

			sprintf(szBuffer, "%7.2f", 100.0 * (float)correct/total);
			Correct->SetText(szBuffer);
			sprintf(szBuffer, "%7.2f", 100.0 * (float)incorrect/total);
			Incorrect->SetText(szBuffer);

			// Add noise
			if (i < ((FORWARDBITS/16) - 1))
			{
				for (WORD j = i * (FORWARDBITS/16);
						 j < (i * (FORWARDBITS/16) + (FORWARDBITS/16)); j++)
				{
					WORD tmp = random(width * 8);
					while (bv_extract(tmp, &(flipped)))
					{
						tmp = random(width * 8);
					}
					bv_set(tmp, &(flipped), 1);
					if(bv_extract(tmp, &(test)))
					{
						bv_set(tmp, &(test), 0);
					}
					else
					{
						bv_set(tmp, &(test), 1);
					}
				}
			}

			LetterBitmap->SetBits(test.bv, width);
			LetterBitmap->Refresh();
			if (treemask & 1)
			{
				LightBulbA->Init(hLightOff);
				LightBulbA->Refresh();
			}
			if (treemask & 2)
			{
				LightBulbL->Init(hLightOff);
				LightBulbL->Refresh();
			}
			if (treemask & 4)
			{
				LightBulbN->Init(hLightOff);
				LightBulbN->Refresh();
			}
		}
	}

	farfree(test.bv);
	farfree(flipped.bv);
	for (i = 0; i < voters; i++)
	{
		farfree(ftreeA[i]);
		farfree(ftreeL[i]);
		farfree(ftreeN[i]);
	}
	farfree(ftreeA);
	farfree(ftreeL);
	farfree(ftreeN);

	EnableWindow(Start->HWindow, TRUE);
	EnableWindow(Train->HWindow, TRUE);
	EnableWindow(Stop->HWindow, FALSE);
	EnableWindow(Reverse->HWindow, TRUE);
	EnableWindow(Reset->HWindow, TRUE);
	EnableWindow(Quit->HWindow, TRUE);
	EnableWindow(Forget->HWindow, TRUE);
	EnableWindow(EditA->HWindow, TRUE);
	EnableWindow(EditL->HWindow, TRUE);
	EnableWindow(EditN->HWindow, TRUE);
	EnableMenuItem(hSysMenu, SC_CLOSE, MF_ENABLED);
}