/*  ansi-rem - A program to remove ANSI escape sequences from a    */
/*             text file.                                          */

/*  This program uses a very simple-minded method of identifying   */
/*  an escape sequence.  By definition, an escape sequence begins  */
/*  with the ESC character (hex 1B).  The program assumes that     */
/*  the sequence ends with the first alphabetic character (case-   */
/*  insensitive) that follows.                                     */
/*  This identifies all escape sequences, but in fact, some        */
/*  sequences end with a digit and some with just a bracket such   */
/*  as '>'.  As a result, the above method will also chop off the  */
/*  first letter of whatever text follows such an escape sequence  */
/*  You may therefore find it necessary to edit the resulting file */
/*  with a text editor to put some characters back.                */

/*  Compiled using Microsoft QuickC version 2.5 with the following */
/*                                                                 */
/*            qcl /Olt /Za /W4 ansi-rem.c                          */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

/*  Global variable declarations.  */
FILE *infile, *outfile;

int main (int argc, char *argv[])
{
/*  This is implemented as a two-state finite state machine    */
/*  The states are START (found a starting delimiter) and STOP */
/*  These enums are also used as the indices of the            */
/*  array cnt[] which counts how many opening delimiters were  */
/*  found when in the START state, not including the one which */
/*  which put us there, and the number of closing delimiters   */
/*  found in the STOP state, not including the one which put   */
/*  us there.

/*  If either cnt[] is non-zero then a warning message is     */
/*  output to stderr telling how many unbalanced escape       */
/*  characters were found and we return with EXIT_FAILURE.    */
/*  Otherwise we return with EXIT_SUCCESS.                    */

    enum States { START, STOP, STATECHNGE };
    int c, state = STOP, cnt[3] = { 0, 0, 0 };
    char sequence[20], *seqptr, *ptr;

    /*  Announce myself  */

    fprintf(stderr, "\nRemoves escape sequences from text files\n"
                      "(The Cursor Forward sequence is translated to a single space.)\n"
                      "(Cursor Position to column 1 is translated to a newline.)\n"
                      "                        Gordon Torrie  April 1991\n\n");

    /*  Check that the input and output files were specified and
        that they exist!                                         */

    if (argc < 2) {
	fprintf(stderr, "Read from what file?\nWrite to what file?\n"  
                        "Try:  %s <infile.ext> <outfile.ext>\n\n", argv[0]);
        exit(2);
        }
    if (argc < 3) {
	fprintf(stderr, "Write to what file?\n"
                        "Try:  %s %s <outfile.ext>\n\n", argv[0], argv[1]);
        exit(3);
        }

    if ( (infile = fopen(argv[1],"rb")) == NULL) {
        fprintf(stderr, "Unable to open file: %s\nDoes it exist?\n\n", argv[1]);
        exit(4);
        }

    if ( (outfile = fopen(argv[2],"wb")) == NULL) {
        fprintf(stderr, "Unable to open file: %s\nDisk or directory full?\n\n", argv[2]);
        exit(4);
        }

    /*  Main part of program.                                     */
    /*  Report what we are about to do.                           */
    fprintf(stderr, "Removing escape sequences.\n");

    while ( (c = fgetc(infile)) != EOF)
       {

       /* If the character just read is the start delimiter       */
       if (c == '\x1B')
          {
          strcpy( sequence, "" );
          seqptr = sequence;
          if (state == STOP) 
             {
             state = START;
             cnt[STATECHNGE]++;
             }
          else cnt[START]++;
          }

       /* If the character just read is the stop delimiter       */
       else if ( (state == START) &&
                (strchr("ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz", c) != NULL) )
          {
          state = STOP;
          cnt[STATECHNGE]--;

          /*  Here's where we put in some intelligence.            */
          /*  At this point the array sequence[] will contain the  */
          /*  escape sequence string (except for the starting      */
          /*  escape character and the terminating alpha character.*/ 

          switch( c )
             {

             /*  If the sequence was a cursor forward sequence,    */
             /*  then output one space.                            */

             case 'C': fputc(' ', outfile);     break;

             /*  If the sequence was a cursor position sequence    */
             /*  and it positioned to column 1 of whatever line,   */
             /*  then output a newline.                            */

             case 'H':
             case 'f': if ( (ptr = strchr(sequence, ';')) != NULL &&
                          strcmp(++ptr, "1") == 0 )
                          fputc('\n', outfile); break;

             default:                           break;
             }
          }

       /* If the character just read is neither the start nor   */
       /* the stop delimiter                                    */
       else if (state == STOP) fputc(c, outfile);

            /*  If we are reading an escape sequence               */
            /*  (state == START) then save the character in the    */
            /*  array sequence[], increment the pointer and add a  */
            /*  terminating null.                                  */

            else 
               {
               *seqptr++ = (char) c;
               *seqptr = '\0';
               }
       }

   fclose(infile);
   fclose(outfile);
   cnt[START] += cnt[STATECHNGE];
   if (cnt[START] != 0 )
      fprintf(stderr, "\nThere %s %d unbalanced escape character%s\n", (cnt[START] == 1) ? "was" : "were", cnt[START], cnt[START]==1 ? "." : "s.");
   if (cnt[START] != 0 || cnt[STOP] != 0) return EXIT_FAILURE;
   else return EXIT_SUCCESS;
   }
