/*
Buffer manipulation, buffered block and character i/o routines.

(C)copyright	Brian McKeon	1983
*/

#include	"xdefs.h"	/* system-wide definitions */

#define BLKSIZ 512		/* disk block size */
#define NBUF 18			/* number of block buffers in cache */

#define XBFLGS	0		/* flags for a buffer */
#define XBDEV	1		/* device and block of which */
#define XBBLK	2		/* this buffer is a copy */
#define XBAGE	3		/* "time" of last access */
#define	XBFREQ	4		/* frequency-of-access parameter */
#define XBUF	5		/* start of actual buffer */

#define XBFREAD 01		/* buffer header bit flags */
#define XBFWRIT 02

#define XBUFSIZ (XBUF+(BLKSIZ/2)) /* buffer plus header size */

#define MINRDBUF (NBUF/3)	/* cache flush threshold */

extern	int	traceflg,	/* debug tracing flag */
		*sysproc;	/* ptr to system process descriptor */

int	*freebuf,		/* free buffer list pointer */
	bufpool[NBUF*XBUFSIZ],	/* pool of buffers and headers for each */
	devioage;		/* counter of calls to buffered i/o func */

/*
Initialization of buffer management.
'freebuf' is a pointer to the first free buffer on
the free list.
If the free list is empty 'freebuf' will be 0.
*/
bufinit()
{
	int	k, *ip;
	printf("buffer pool at %d\n",bufpool);
	freebuf = 0;	/* freebuf == 0 will indicate end of free list */
	ip = bufpool + XBUF;	 /* point at start of first buffer */
	for(k = 0; k < NBUF ; k++)
		{
		buffree(ip);	/* and free-up all buffers */
		ip = ip + XBUFSIZ;
		}
}

/*
Return a buffer to the pool.
Range checking has been removed
as this function is only called by the system.
*/
buffree(buf)
int *buf;	/* note usage as array of int for correct pointer maths */
{
	buf = buf - XBUF;	/* back-up to address of buffer header */
	buf[XBFLGS] = 0;	/* clear all flags */
	buf[XBFLGS + 1] = freebuf;	/* link onto free list */
	freebuf = buf;
}

/*
Allocate a buffer from the pool.
Look for the buffer with the lowest frequency of usage and,
if more that one with the same value, choose the least
recently used one.
If the number of read-type buffers is below the minimum,
flush any write buffers.
*/
bufalloc()
{
	int *ptr, *ip, k, age, oldest, freq, nrdbuf, nflush;
	if(freebuf)	/* any buffers on free list? */
		{
		ptr = freebuf;	/* point free-list pointer to next on list */
		freebuf = freebuf[XBFLGS + 1];
		fillw(ptr + XBUF, SYSDSEG, 0, BLKSIZ/2); /* fill with null */
		return (ptr + XBUF);
		}
	ptr = bufpool;	/* otherwise look for a 'READ' buffer */
	nrdbuf = 0;	/* counter of number of 'READ' buffers found */
	ip = 0;		/* 'ip' will be non-zero if a 'READ' buffer found */
	freq = 32767;	/* hunt for XBFREQ less than this (a big number) */
	oldest = 0;	/* hunt for a buffer older than this */
	for(k = 0; k < NBUF ; k++)
		{
		if(ptr[XBFLGS] & XBFREAD) /* a READ-type buffer */
			{
			nrdbuf++;
			if(ptr[XBFREQ] <= freq)
				{
				freq = ptr[XBFREQ];
				age = devioage - ptr[XBAGE];
				if(age > oldest)
					{
					oldest = age;
					ip = ptr; /* save it */
					}
				}
			}
		ptr = ptr + XBUFSIZ;
		}
	if(nrdbuf < MINRDBUF)
		nflush = bufsflush();	/* record number flushed */
	if(ip)	/* was a READ buffer found? */
		{
		buffree(ip +XBUF); /* free it up */
		return bufalloc(); /* and try again */
		}
	else if(nflush)	/* were any write buffers were flushed? */
		return bufalloc();
	else	stop("bufalloc/problems");
}

/*
Buffered block i/o function.
Give device, read/write, desired block, address of data and i.d.
of process requesting i/o.
Returns number of bytes transferred.
*/
bufdevio(dev, rwfn, blk, addr, proc)
int *dev, rwfn, blk, addr, *proc;
{
	int	*ip;
	return cbufdevio(dev, rwfn, blk, 0, BLKSIZ, addr, proc);
}

/*
Buffered character i/o function.
Give: device, read/write, desired block, offset in block
at which transfer is to occur, number of bytes to transfer,
address of data and i.d. of process requesting i/o.
Returns number of bytes transferred.
*/
cbufdevio(dev, rwfn, blk, offset, nbytes, addr, proc)
int *dev, rwfn, blk, offset, nbytes, addr, *proc;
{
	int *ip, *buf, k, c_locn;
	if(traceflg > 2)
		printf("cbufdevio(%d, %d, %d, %d, %d, %d, %d)\n",
			dev, rwfn, blk, offset, nbytes, addr, proc);
	if((offset < 0) | (nbytes < 0)) /* check for bad arguments */
		stop("bufdevio negative arg");
	if((offset + nbytes) > BLKSIZ)
		stop("bufdevio request too big");
	if(nbytes == 0)
		return 0;
	devioage++;	/* bump access counter */
	if((devioage & 017) == 0) /* i.e. every 16 times called */
		{ /* 'decay' all XBFREQ values by one half (rotate) */
		ip = bufpool;
		for(k = 0; k < NBUF ; k++)
			if(ip[XBFLGS] & XBFREAD) /* a READ type buffer */
				ip[XBFREQ] = ip[XBFREQ] >> 1; /* halve */
		}
	if(ip = findblk(blk, dev)) /* this device block in a buffer? */
		{
		/*
		increase frequency of access count and
		re-set time last accessed
		*/
		ip[XBFREQ] = ip[XBFREQ] + 128;
		ip[XBAGE] = devioage;
		c_locn = ip + XBUF;	/* get address of start of buffer */
		c_locn = c_locn + offset; /* and transfer address */
		if(rwfn == RDFN)	/* transfer from buffer */
			{
			movsb(c_locn, SYSDSEG, addr, proc[PRDSEG], nbytes);
			return	nbytes;
			}
		if(rwfn == WTFN) /* transfer to buffer and mark as WRITE */
			{
			movsb(addr, proc[PRDSEG], c_locn, SYSDSEG, nbytes);
			ip[XBFLGS] = (ip[XBFLGS] & (~XBFREAD)) | XBFWRIT;
			return nbytes;
			}
		}
	/*
	the device block is not in the pool,
	allocate a buffer and read the block
	and do the requested transfer
	*/
	buf = ip = bufalloc();
	ip = ip - XBUF;	/* backup to start of buffer header */
	ip[XBDEV] = dev; /* set-up header except for flags */
	ip[XBBLK] = blk;
	ip[XBAGE] = devioage;
	ip[XBFREQ] = 0;
	/*
	if a full block is to be written then
	it is not necessary to read the old block
	contents into the buffer
	*/
	if((nbytes == BLKSIZ) & (rwfn == WTFN)) /* write full block? */
		{
		movsw(addr, proc[PRDSEG], buf, SYSDSEG, BLKSIZ/2);
		ip[XBFLGS] = XBFWRIT;	/* mark as WRITE */
		k = BLKSIZ;		/* return value */
		}
	else	/* fetch physical block for reading or up-dating */
		{
		if(devio(dev, RDFN, blk, buf, sysproc) == ERROR)
			{
			buffree(buf);
			k = 0;
			}
		else	{
			ip[XBFLGS] = XBFREAD;
			k = cbufdevio(dev, rwfn, blk, offset,
				nbytes, addr, proc);
			}
		}
	return	k;
}

/*
If a device block is in the buffer pool, return
the address of the buffer, else 0.
*/
findblk(blk, dev)
int	blk, dev;
{
	register int *ip;
	int	k;
	ip = bufpool;
	for(k = 0; k < NBUF ; k++)
		{
		if(ip[XBFLGS]&(XBFREAD | XBFWRIT)) /* a device block image */
			 if(ip[XBBLK] == blk) /* do more rare test first */
				if(ip[XBDEV] == dev)
					return	ip; /* a hit */
		ip = ip + XBUFSIZ;
		}
	return	0;		/* a miss */
}

/*
Flush all pending writes in the buffer pool
and convert write status to read.
Return the number of buffers so changed.
Writing of buffers is performed in order to
minimise head movement across the disk.
Sectors on the same track will also be written
in order.
The order is from highest device i.d. to lowest
and lowest block number to highest.
*/
bufsflush()
{
	int *ip, nwrt;
	nwrt = 0;	/* counter of buffers flushed */
	while(ip = buf_w_next()) /* find next in order */
		{
		devio(ip[XBDEV], WTFN, ip[XBBLK], ip + XBUF, sysproc);
		ip[XBFLGS] = (ip[XBFLGS] & (~XBFWRIT)) | XBFREAD;
		nwrt++;
		}
	return	nwrt;	/* number of buffers written */
}

/*
Scan the buffer pool for buffers marked as WRITE
and return the address of the one with greatest device i.d.
and lowest block number. If none found, return 0.
*/
buf_w_next()
{
	register int	*ip, k;
	int	*devmax, blkmin, rval;
	devmax = 0;		/* a small, impossible, i.d. number */
	blkmin = 32767;		/* the biggest block number */
	rval = 0;		/* default return value */
	ip = bufpool;
	for(k = 0; k < NBUF ; k++)
		{
		if(ip[XBFLGS] & XBFWRIT) /* a WRITE buffer ? */
			{
			if(devmax < ip[XBDEV]) /* larger device i.d.? */
				{
				devmax = ip[XBDEV];
				blkmin = ip[XBBLK];
				rval = ip;	/* save it */
				}
			else  /* same device, smaller block number? */
			      if(devmax == ip[XBDEV])
				if(blkmin > ip[XBBLK])
					{
					blkmin = ip[XBBLK];
					rval = ip;
					}
			}
		ip = ip + XBUFSIZ;
		}
	return	rval;	/* zero or ptr to next block/dev buffer */
}
