/*	Copyright (c) 1989 Citadel	*/
/*	   All Rights Reserved    	*/

/* #ident	"@(#)lsopen.c	1.4 - 90/06/20" */

/* ansi headers */
#include <errno.h>
/*#include <stddef.h>*/
/*#include <string.h>*/

/* library headers */
#include <blkio.h>

/* local headers */
#include "lseq_.h"

/* lseq control structure table definition */
lseq_t lsb[LSOPEN_MAX];

/*man---------------------------------------------------------------------------
NAME
     lsopen - open an lseq

SYNOPSIS
     lseq_t *lsopen(filename, type)
     const char *filename;
     const char *type;

DESCRIPTION
     The lsopen function opens the file named by filename as an lseq.
     A pointer to the lseq_t structure associated with the file is
     returned.

     type is a character string having one of the following values:

          "r"            open for reading
          "r+"           open for update (reading and writing)

     lsopen will fail if one or more of the following is true:

     [EINVAL]       filename is the NULL pointer.
     [EINVAL]       type is not "r" or "r+".
     [ENOENT]       The named lseq file does not exist.
     [LSECORRUPT]   The named file is corrupt.
     [LSEEOF]       No file header.
     [LSEMFILE]     Too many open lseqs.  The maximum
                    is defined as LSOPEN_MAX in lseq.h.

SEE ALSO
     lsclose, lscreate.

DIAGNOSTICS
     lsopen returns a NULL pointer on failure, and errno set to
     indicate the error.

------------------------------------------------------------------------------*/
lseq_t *lsopen(filename, type)
const char *filename;
const char *type;
{
	int terrno = 0;
	lseq_t *lsp = NULL;

	/* validate input parameters */
	if (filename == NULL || type == NULL) {
		errno = EINVAL;
		return NULL;
	}

	/* find free slot in lsb table */
	for (lsp = lsb; lsp < lsb + LSOPEN_MAX; ++lsp) {
		if (!(lsp->flags & LSOPEN)) {
			break;		/* found */
		}
	}
	if (lsp >= lsb + LSOPEN_MAX) {
		errno = LSEMFILE;
		return NULL;		/* no free slots */
	}

	/* open file */
	if (strcmp(type, LS_READ) == 0) {
		lsp->flags = LSREAD;
	} else if (strcmp(type, LS_RDWR) == 0) {
		lsp->flags = LSREAD | LSWRITE;
	} else {
		errno = EINVAL;
		return NULL;
	}
	lsp->bp = bopen(filename, type, sizeof(lshdr_t), (size_t)1, (size_t)0);
	if (lsp->bp == NULL) {
		if ((errno != EACCES) && (errno != ENOENT)) LSEPRINT;
		memset(lsp, 0, sizeof(*lsb));
		lsp->flags = 0;
		return NULL;
	}

	/* load lseq_t structure */
	if (bgeth(lsp->bp, &lsp->lshdr) == -1) {	/* header */
		LSEPRINT;
		terrno = errno;
		bclose(lsp->bp);
		memset(lsp, 0, sizeof(*lsb));
		lsp->flags = 0;
		errno = terrno;
		return NULL;
	}
	if (lsp->lshdr.flags & LSHMOD) {		/* corrupt file */
		bclose(lsp->bp);
		memset(lsp, 0, sizeof(*lsb));
		lsp->flags = 0;
		errno = LSECORRUPT;
		return NULL;
	}
	lsp->clspos = NIL;				/* cursor */
	if (ls_alloc(lsp) == -1) {
		LSEPRINT;
		terrno = errno;
		bclose(lsp->bp);
		memset(lsp, 0, sizeof(*lsb));
		lsp->flags = 0;
		errno = terrno;
		return NULL;
	}

	/* set up buffering */
	if (bsetvbuf(lsp->bp, NULL, ls_blksize(lsp), LSBUFCNT) == -1) {
		LSEPRINT;
		terrno = errno;
		ls_free(lsp);
		bclose(lsp->bp);
		memset(lsp, 0, sizeof(*lsb));
		lsp->flags = 0;
		errno = terrno;
		return NULL;
	}

	errno = 0;
	return lsp;
}
