/*	Copyright (c) 1989 Citadel	*/
/*	   All Rights Reserved    	*/

/* #ident	"@(#)fml.c	1.4 - 90/06/21" */

#include <blkio.h>	/* ansi compatibility macros */

/* ansi headers */
#include <errno.h>
/*#include <stddef.h>*/
#include <stdio.h>
/*#include <stdlib.h>*/
/*#include <string.h>*/

/* macros */
#define min(a,b)	((a) < (b) ? (a) : (b))

/*man---------------------------------------------------------------------------
NAME
     fmltolfm - convert name from fml to lfm format

SYNOPSIS
     int fmltolfm(t, s, n)
     char *t;
     const char *s;
     size_t n;

DESCRIPTION
     The fmltolfm function converts a name from the format "first
     middle last" to "last first middle".  s points to the source
     string in fml format.  t points to the target string to receive
     the converted name.  t and s may point to the same string.  There
     may be more than one middle name, or the middle name may be
     absent.  All leading and trailing spaces must be removed from the
     source string before calling fmltolfm.

     fmltolfm will fail if one or more of the following is true:

     [EINVAL]       t or s is the NULL pointer.

SEE ALSO
     lfmtofml.

DIAGNOSTICS
     Upon successful completion, a value of 0 is returned.  Otherwise,
     a value of -1 is returned, and errno set to indicate the error.

------------------------------------------------------------------------------*/
int fmltolfm(t, s, n)
char *t;
const char *s;
size_t n;
{
	char *p = NULL;
	char *ts = NULL;	/* temporary string */

	/* validate arguments */
	if (t == NULL || s == NULL) {
		errno = EINVAL;
		return -1;
	}
	if (n < 2) {
		errno = 0;
		return 0;
	}

	/* find beginning of last name */
	p = strrchr(s, ' ');
	if (p == NULL) {
		strncpy(t, s, n);
		t[n - 1] = '\0';
		errno = 0;
		return 0;
	}

	/* create temporary string */
	ts = (char *)calloc((size_t)n, (size_t)1);
	if (ts == NULL) {
		errno = ENOMEM;
		return -1;
	}

	/* perform conversion */
	strncpy(ts, p + 1, n);			/* copy last name */
	ts[n - 1] = '\0';
	strncat(ts, " ", n - strlen(ts));	/* add space after last name */
	ts[n - 1] = '\0';
	strncat(ts, s, n - strlen(ts));		/* copy beginning of name */
	ts[min(n - 1, strlen(s))] = '\0';
	strncpy(t, ts, n);			/* copy converted string to t */
	t[n - 1] = '\0';

	/* free temporary string */
	free(ts);

	errno = 0;
	return 0;
}

/*man---------------------------------------------------------------------------
NAME
     lfmtofml - convert name from lfm to fml format

SYNOPSIS
     int lfmtofml(t, s, n)
     char *t;
     const char *s;
     size_t n;

DESCRIPTION
     The lfmtofml function converts a name from the format "last
     first middle" to "first middle last".  s points to the source
     string in lfm format.  t points to the target string to receive
     the converted name.  t and s may point to the same string.  There
     may be more than one middle name, or the middle name may be
     absent.  All leading and trailing spaces must be removed from the
     source string before calling lfmtofml.

     lfmtofml will fail if one or more of the following is true:

     [EINVAL]       t or s is the NULL pointer.

SEE ALSO
     fmltolfm.

DIAGNOSTICS
     Upon successful completion, a value of 0 is returned.  Otherwise,
     a value of -1 is returned, and errno set to indicate the error.

------------------------------------------------------------------------------*/
int lfmtofml(t, s, n)
char *t;
const char *s;
size_t n;
{
	char *p = NULL;
	char *ts = NULL;	/* temporary string */

	/* validate arguments */
	if (t == NULL || s == NULL) {
		errno = EINVAL;
		return -1;
	}
	if (n < 2) {
		errno = 0;
		return 0;
	}

	/* find end of last name */
	p = strchr(s, ' ');
	if (p == NULL) {
		strncpy(t, s, n);
		t[n - 1] = '\0';
		errno = 0;
		return 0;
	}

	/* create temporary string */
	ts = (char *)calloc((size_t)n, (size_t)1);
	if (ts == NULL) {
		errno = ENOMEM;
		return -1;
	}

	/* perform conversion */
	strncpy(ts, p + 1, n);			/* copy beginning of name */
	ts[n - 1] = '\0';
	strncat(ts, " ", n - strlen(ts));	/* add space before last name */
	ts[n - 1] = '\0';
	strncat(ts, s, n - strlen(ts));		/* copy last name */
	ts[min(n - 1, strlen(s))] = '\0';
	strncpy(t, ts, n);			/* copy converted string to t */
	t[n - 1] = '\0';

	/* free temporary string */
	free(ts);

	errno = 0;
	return 0;
}
