/*	Copyright (c) 1989 Citadel	*/
/*	   All Rights Reserved    	*/

/* #ident	"@(#)cbimport.c	1.4 - 90/06/20" */

/* ansi headers */
#include <errno.h>
/*#include <stddef.h>*/
#include <stdio.h>
/*#include <stdlib.h>*/

/* non-ansi headers */
#include <bool.h>

/* local headers */
#include "cbase_.h"

/*man---------------------------------------------------------------------------
NAME
     cbimport - cbase import

SYNOPSIS
     #include <cbase.h>

     int cbimport(cbp, filename)
     cbase_t *cbp;
     const char *filename;

DESCRIPTION
     The cbimport function imports all records from a printable file
     to cbase cbp.  filename points to a character string that
     contains the name of the printable file.  See cbexport for a
     definition of the file format.

     If a record containing an illegal duplicate key is encountered
     during the import, that record is skipped and the import
     continues with the subsequent record.  On successful completion
     of the remainder of the import, a value of -1 is returned with
     errno set to CBEDUP.  Whether or not the calling program should
     treat this as an error condition is application dependent.

     cbimport will fail if one or more of the following is true:

     [EINVAL]       cbp is not a valid cbase pointer.
     [EINVAL]       filename is the NULL pointer.
     [ENOENT]       filename does not exist.
     [CBEDUP]       An illegal duplicate key was encountered.
     [CBEPRFILE]    Error reading from printable file.
     [CBEPRFILE]    Invalid printable file format.

SEE ALSO
     cbexport.

DIAGNOSTICS
     Upon successful completion, a value of 0 is returned.  Otherwise,
     a value of -1 is returned, and errno set to indicate the error.

------------------------------------------------------------------------------*/
int cbimport(cbp, filename)
cbase_t *cbp;
const char *filename;
{
	FILE *	fp	= NULL;		/* import stream */
	void *	buf	= NULL;		/* input buffer */
	bool	dupflag	= FALSE;	/* illegal duplicate key flag */
	int	field	= 0;		/* field number */
	size_t	fldos	= 0;		/* field offset */
	size_t	fldlen	= 0;		/* field length */
	int	fldtype	= 0;		/* field data type */
	int	terrno	= 0;		/* tmp errno */

	/* validate arguments */
	if (!cb_valid(cbp) || filename == NULL) {
		errno = EINVAL;
		return -1;
	}

	/* open file for reading */
	fp = fopen(filename, "r");
	if (fp == NULL) {
		return -1;
	}

	/* import all records from printable file */
	buf = calloc((size_t)1, cbrecsize(cbp));
	if (buf == NULL) {
		CBEPRINT;
		fclose(fp);
		errno = ENOMEM;
		return -1;
	}
	while (!feof(fp)) {
		if (ferror(fp)) {
			CBEPRINT;
			free(buf);
			fclose(fp);
			return -1;
		}
		memset(buf, 0, cbrecsize(cbp));
		/* input each field of new record */
		for (field = 0; field < cbp->fldc; ++field) {
			fldos = cbp->fldv[field].offset;
			fldlen = cbp->fldv[field].len;
			fldtype = cbp->fldv[field].type;
			/* read field from printable file */
			if ((*cbimpv[fldtype])(fp, (char *)buf + fldos, fldlen) == -1) {
				CBEPRINT;
				free(buf);
				fclose(fp);
				errno = CBEPRFILE;
				return -1;
			}
			if (feof(fp)) break;
		}
		if (feof(fp)) {
			break;
		}
		/* add record to database */
		if (cbinsert(cbp, buf) == -1) {
			if (errno == CBEDUP) {
				dupflag = TRUE;
			} else {
				terrno = errno;
				free(buf);
				fclose(fp);
				errno = terrno;
				return -1;
			}
		}
	}
	free(buf);
	buf = NULL;

	/* close file */
	if (fclose(fp) == EOF) {
		CBEPRINT;
		return -1;
	}

	/* notify if illegal duplicate key */
	if (dupflag) {
		errno = CBEDUP;
		return -1;
	}

	errno = 0;
	return 0;
}
