/* joystkmd.c       Middle level joystick functions built on joystklo.c.

                The functions in this module
                    1.  detect and initialize a joystick coordinate system.
                    2.  read changed joystick position
                    3.  provide a rough translation from the top-left origin
                        joystick coordinate system to a balanced 0,0
                        centered cartesian system
                    4.  get/reset joystick center coords
                    5.  get/reset joystick minimum coords
                    6.  get/reset joystick maximum coords
                    7.  get/set an ignore threshold (tolerance) for
                        reading changed joystick positions

                See file joystkmd.se for pseudo code/structured english
                and joystick.doc for an explanation of the functions.

                This module was written for Borland C++ 2.0.
                No C++ extensions were used, no Assembler.
                This is all native C.

                I've done limited testing on an old XT with 11/08/82 BIOS
                and on my PS/2 like 386 AT and haven't found anything yet...

                One thing... the initialization function is intended for
                a single user joystick installation.  Multiple joysticks
                may be initialized seperately by using the Get/SetCenter,
                Get/SetMax, and Get/SetMin functions in a fashion similar
                to my initialization algorithm.

                If you find this module useful, great!!!

                This module has been released to the public domain
                        Tue  01-07-1992   23:30:18
                by Ed Torgerson : CIS 70313,456,  GEnie E.TORGERSON1.
                any constructive correspondence is welcome.

-----------------------------------------------------------------------------*/


#include <stdio.h>
#include <conio.h>
#include "joystick.h"

/*-- Global Typedefs --------*/                
struct StickValues {
        int         jax ;       /* joystick A x-coordinate value */
        int         jay ;       /* joystick A y-coordinate value */
        int         jbx ;       /* joystick B x-coordinate value */
        int         jby ;       /* joystick B y-coordinate value */
} ;

struct Tolerance {
        int         axl ;     /* joystick A x-low breakpoint */
        int         axm ;     /* joystick A x-mid breakpoint */
        int         axh ;     /* joystick A x-hi  breakpoint */
        int         ayl ;     /* joystick A y-low breakpoint */
        int         aym ;     /* joystick A y-mid breakpoint */
        int         ayh ;     /* joystick A y-hi  breakpoint */
        int         bxl ;     /* joystick B x-low breakpoint */
        int         bxm ;     /* joystick B x-mid breakpoint */
        int         bxh ;     /* joystick B x-hi  breakpoint */
        int         byl ;     /* joystick B y-low breakpoint */
        int         bym ;     /* joystick B y-mid breakpoint */
        int         byh ;     /* joystick B y-hi  breakpoint */
        char        t1  ;     /* low graded tolerance */
        char        t2  ;     /* mid graded tolerance */
        char        t3  ;     /* hi  graded tolerance */
} ;


/*-- Global variables -----*/
struct StickValues  G_Stick_Centers ;   /* coords of joystick centers */
struct StickValues  G_Stick_Minimums ;  /* minimum joystick coordinates */
struct StickValues  G_Stick_Maximums ;  /* maximum joystick coordinates */
struct StickValues  G_Stick_Last_Pos ;  /* joystick coordinates last read */
int                 G_Stick_Tolerance ; /* maximum scaled ignore threshold */
struct Tolerance    G_Stick_Breaks ;    /* breakpoints for tolerance scaling */


/*-- Function Prototypes of Internal Subroutines -----*/
int jst_init_sub  ( int msg, int stop_button, struct StickValues *what,
                    void (*NotifyUserFunc)(int)  ) ;
int jst_set_coord_sub ( int stick, int function ) ;


/*-- Function Definitions ---------------------------------------------------*/
int JstInitialize ( void (*NotifyUserFunc)(int) )
{
    register int    retval, Jstk ;

    JstSetActiveAxes( 0 ) ;
    retval = JstDetect () ;
    Jstk = (JAX_AXIS | JAY_AXIS) ;      /* test for presence of Joystick A */
    if ( (retval & Jstk) == (Jstk) )
    {
        JstSetActiveAxes( Jstk ) ;       /* set joystick axes mask active */
    }
    Jstk = (JBX_AXIS | JBY_AXIS) ;      /* test for presence of Joystick B */
    if ( (retval & Jstk) == (Jstk) )
    {
        JstSetActiveAxes( Jstk | JstGetActiveAxes() ) ;   /* add jstk mask */
    }
    if ( JstGetActiveAxes() == 0 )     /* if a 2-axis joystick is not found */
    {
        NotifyUserFunc(0) ;             /* notify user */
        return (0) ;                    /* return error code */
    }
    NotifyUserFunc ( 1 ) ;              /* notify user: press escape to quit */
    if ( jst_init_sub( 2, J_ANY_BUTTON_DOWN, &G_Stick_Centers, NotifyUserFunc )
            == 0 )
        return (0) ;                    /* return error code */
                                        /* get center coordinates from user */
    if ( jst_init_sub( 3, J_BUTTON_1_DOWN, &G_Stick_Minimums, NotifyUserFunc )
            == 0 )
        return (0) ;                    /* return error code */
                                        /* get minimum coordinates from user */
    if ( jst_init_sub (  4, J_BUTTON_2_DOWN,&G_Stick_Maximums, NotifyUserFunc )
            == 0 )
        return (0) ;                    /* return error code */
                                        /* get maximum coordinates from user */
    G_Stick_Last_Pos.jax = 0 ;          /* zero out last position */
    G_Stick_Last_Pos.jay = 0 ;
    G_Stick_Last_Pos.jbx = 0 ;
    G_Stick_Last_Pos.jby = 0 ;                                      
    if ( G_Stick_Maximums.jax != 0 )        /* set tolerance for events */
    {
        Jstk = G_Stick_Maximums.jax + G_Stick_Maximums.jax ;
        G_Stick_Tolerance = Jstk >> 6 ;
    }
    else if ( G_Stick_Maximums.jbx != 0 )
    {
        Jstk = G_Stick_Maximums.jbx + G_Stick_Maximums.jbx ;
        G_Stick_Tolerance = Jstk >> 6 ;
    }
    JstSetTolerance ( G_Stick_Tolerance ) ;
    return ( 1 ) ;                                          /* success */
}  /* end JstInitialize ---*/

            /*------------------------------------------*/
int jst_init_sub ( int msg, int stop_button, struct StickValues *what,
                    void (*NotifyUserFunc)(int) )
                                    /* subroutine to get input from user */
{
    register int    retval ;

    NotifyUserFunc( msg ) ;
    retval = 0 ;
    do
    {
        if ( kbhit() )
        {
            retval = getch() ;
            if (retval == 27)                   /* user pressed escape */
            {
                NotifyUserFunc (-1) ;           /* user terminate */
                NotifyUserFunc (-2) ;           /* "incomplete" warning */
                return(0) ;
            }
        }
        retval = JstGetChangedButton() ;
    } while ( (retval & stop_button) == 0 ) ;
    what->jax = JstGetPosition( JAX_AXIS  ) ;
    what->jay = JstGetPosition( JAY_AXIS  ) ;
    what->jbx = JstGetPosition( JBX_AXIS  ) ;
    what->jby = JstGetPosition( JBY_AXIS  ) ;
    return (1) ;
}  /* end jst_init_sub ---*/

/*---------------------------------------------------------------------*/
int JstGetChangedPosition ( int stick, struct JoystickPosition *stickpos )
                                           /* returns changes in active axes */
{
    int    x_pos, y_pos, xtol, ytol ;

    if ( (stick == JOYSTICK_A) || (stick == JOYSTICK_B) )
    {
        if ( (stick & JstGetActiveAxes() ) == stick )
        {
            xtol = ytol = G_Stick_Tolerance ;
            if (stick == JOYSTICK_A)
            {
                x_pos = JstGetPosition( JAX_AXIS ) ;
                    if ( x_pos < G_Stick_Breaks.axl )
                        xtol = G_Stick_Breaks.t1 ;
                    else if ( x_pos < G_Stick_Breaks.axm )
                        xtol = G_Stick_Breaks.t2 ;
                    else if ( x_pos < G_Stick_Breaks.axh )
                        xtol = G_Stick_Breaks.t3 ;
                y_pos = JstGetPosition( JAY_AXIS ) ;
                    if ( y_pos < G_Stick_Breaks.ayl )
                        ytol = G_Stick_Breaks.t1 ;
                    else if ( y_pos < G_Stick_Breaks.aym )
                        ytol = G_Stick_Breaks.t2 ;
                    else if ( y_pos < G_Stick_Breaks.ayh )
                        ytol = G_Stick_Breaks.t3 ;
                if ( (x_pos < G_Stick_Last_Pos.jax - xtol ) ||
                     (x_pos > G_Stick_Last_Pos.jax + xtol ) ||
                     (y_pos < G_Stick_Last_Pos.jay - ytol ) ||
                     (y_pos > G_Stick_Last_Pos.jay + ytol ) )
                {
                    G_Stick_Last_Pos.jax = x_pos ;
                    G_Stick_Last_Pos.jay = y_pos ;
                    stickpos->jx = x_pos ;
                    stickpos->jy = y_pos ;
                    return ( 1 ) ;                  /* change */
                }
            }
            else if (stick == JOYSTICK_B)
            {
                x_pos = JstGetPosition( JBX_AXIS ) ;
                    if ( x_pos < G_Stick_Breaks.bxl )
                        xtol = G_Stick_Breaks.t1 ;
                    else if ( x_pos < G_Stick_Breaks.bxm )
                        xtol = G_Stick_Breaks.t2 ;
                    else if ( x_pos < G_Stick_Breaks.bxh )
                        xtol = G_Stick_Breaks.t3 ;
                y_pos = JstGetPosition( JBY_AXIS ) ;
                    if ( y_pos < G_Stick_Breaks.byl )
                        ytol = G_Stick_Breaks.t1 ;
                    else if ( y_pos < G_Stick_Breaks.bym )
                        ytol = G_Stick_Breaks.t2 ;
                    else if ( y_pos < G_Stick_Breaks.byh )
                        ytol = G_Stick_Breaks.t3 ;
                if ( (x_pos < G_Stick_Last_Pos.jbx - xtol ) ||
                     (x_pos > G_Stick_Last_Pos.jbx + xtol ) ||
                     (y_pos < G_Stick_Last_Pos.jby - ytol ) ||
                     (y_pos > G_Stick_Last_Pos.jby + ytol ) )
                {
                    G_Stick_Last_Pos.jbx = x_pos ;
                    G_Stick_Last_Pos.jby = y_pos ;
                    stickpos->jx = x_pos ;
                    stickpos->jy = y_pos ;
                    return ( 1 ) ;                  /* change */
                }
            }
        }
    }
    return ( 0 ) ;
}   /*-- end JstGetChangedPosition ----*/
/*---------------------------------------------------------------------*/
void JstXlatToCart ( int stick, struct JoystickPosition *stickpos )
                /* translates axis position relative to the center in
                    rough cartesian coordinates */
{
    if (stick == JOYSTICK_A)
    {
        stickpos->jx = stickpos->jx - G_Stick_Centers.jax ;
        if ( stickpos->jx > 0 )
            stickpos->jx = stickpos->jx >> 1 ;
        stickpos->jy = -(stickpos->jy - G_Stick_Centers.jay) ;
        if ( stickpos->jy < 0 )
            stickpos->jy = stickpos->jy / 2 ;
    }
    else if (stick == JOYSTICK_B)
    {
        stickpos->jx = stickpos->jx - G_Stick_Centers.jbx ;
        if ( stickpos->jx > 0 )
            stickpos->jx = stickpos->jx >> 1 ;
        stickpos->jy = -(stickpos->jy - G_Stick_Centers.jby) ;
        if ( stickpos->jy < 0 )
            stickpos->jy = stickpos->jy / 2 ;
    }
    return ;
}
/*---------------------------------------------------------------------*/
void JstGetCenter( int stick, struct JoystickPosition *jpos )
                            /* gets the set center values for the joystick */
{
    if (stick == JOYSTICK_A)
    {
        jpos->jx = G_Stick_Centers.jax ;
        jpos->jy = G_Stick_Centers.jay ;
    }
    if (stick == JOYSTICK_B)
    {
        jpos->jx = G_Stick_Centers.jbx ;
        jpos->jy = G_Stick_Centers.jby ;
    }
}
/*---------------------------------------------------------------------*/
void JstGetMin( int stick, struct JoystickPosition *jpos )
                            /* gets the set minimum values for the joystick */
{
    if (stick == JOYSTICK_A)
    {
        jpos->jx = G_Stick_Minimums.jax ;
        jpos->jy = G_Stick_Minimums.jay ;
    }
    if (stick == JOYSTICK_B)
    {
        jpos->jx = G_Stick_Minimums.jbx ;
        jpos->jy = G_Stick_Minimums.jby ;
    }
}
/*---------------------------------------------------------------------*/
void JstGetMax( int stick, struct JoystickPosition *jpos)
                            /* gets the set maximum values for the joystick */
{
    if (stick == JOYSTICK_A)
    {
        jpos->jx = G_Stick_Maximums.jax ;
        jpos->jy = G_Stick_Maximums.jay ;
    }
    if (stick == JOYSTICK_B)
    {
        jpos->jx = G_Stick_Maximums.jbx ;
        jpos->jy = G_Stick_Maximums.jby ;
    }
}
/*---------------------------------------------------------------------*/
int JstSetCenter( int stick )   /* grabs current pos as new joystick center */
{
    return (jst_set_coord_sub( stick, 1 )) ;
}
/*---------------------------------------------------------------------*/
int JstSetMin( int stick )      /* grabs current pos as new minimum */
{
    return (jst_set_coord_sub( stick, 2 )) ;
}
/*---------------------------------------------------------------------*/
int JstSetMax( int stick )      /* grabs current pos as new maximum */
{
    return (jst_set_coord_sub( stick, 3 )) ;
}
/*---------------------------------------------------------------------*/
int jst_set_coord_sub ( int stick, int function )
{
    struct StickValues *ptrStickValues ;

    if ( (stick == JOYSTICK_A) || (stick == JOYSTICK_B) )
    {
        if ( (stick & JstGetActiveAxes()) == stick )
        {
            if ( function == 1 )
                ptrStickValues = &G_Stick_Centers ;
            if ( function == 2 )
                ptrStickValues = &G_Stick_Minimums ;
            if ( function == 3 )
                ptrStickValues = &G_Stick_Maximums ;
            if ( stick == JOYSTICK_A )
            {
                ptrStickValues->jax = JstGetPosition( JAX_AXIS  ) ;
                ptrStickValues->jay = JstGetPosition( JAY_AXIS  ) ;
            }
            if ( stick == JOYSTICK_B )
            {
                ptrStickValues->jbx = JstGetPosition( JBX_AXIS ) ;
                ptrStickValues->jby = JstGetPosition( JBY_AXIS ) ;
            }
            JstSetTolerance ( G_Stick_Tolerance ) ;
            return ( 1 ) ;
        }
    }
    return ( 0 ) ;
}  /* end jst_set_coord_sub ---*/
/*---------------------------------------------------------------------*/
int JstGetTolerance ()                  /* get tolerance of change to ignore */
{
    return ( G_Stick_Tolerance ) ;
}
/*---------------------------------------------------------------------*/
void JstSetTolerance (int tolerance)    /* set tolerance of change to ignore */
{
    G_Stick_Tolerance = tolerance ;
    if ( tolerance > 7 )
    {
        G_Stick_Breaks.axl = G_Stick_Centers.jax >> 2 ;
        G_Stick_Breaks.axm = G_Stick_Centers.jax >> 1 ;
        G_Stick_Breaks.axh = G_Stick_Maximums.jax >> 1 ;
        G_Stick_Breaks.ayl = G_Stick_Centers.jay >> 2 ;
        G_Stick_Breaks.aym = G_Stick_Centers.jay >> 1 ;
        G_Stick_Breaks.ayh = G_Stick_Maximums.jay >> 1 ;
        G_Stick_Breaks.bxl = G_Stick_Centers.jbx >> 2 ;
        G_Stick_Breaks.bxm = G_Stick_Centers.jbx >> 1 ;
        G_Stick_Breaks.bxh = G_Stick_Maximums.jbx >> 1 ;
        G_Stick_Breaks.byl = G_Stick_Centers.jby >> 2 ;
        G_Stick_Breaks.bym = G_Stick_Centers.jby >> 1 ;
        G_Stick_Breaks.byh = G_Stick_Maximums.jby >> 1 ;
        G_Stick_Breaks.t1 = (tolerance >> 2) - 1 ;
        G_Stick_Breaks.t2 = tolerance >> 2 ;
        G_Stick_Breaks.t3 = tolerance >> 1 ;
    }
    else
    {
        G_Stick_Breaks.axl = G_Stick_Breaks.axm = G_Stick_Breaks.axh = 0 ;
        G_Stick_Breaks.ayl = G_Stick_Breaks.aym = G_Stick_Breaks.ayh = 0 ;
        G_Stick_Breaks.bxl = G_Stick_Breaks.bxm = G_Stick_Breaks.bxh = 0 ;
        G_Stick_Breaks.byl = G_Stick_Breaks.bym = G_Stick_Breaks.byh = 0 ;
        G_Stick_Breaks.t1 = tolerance ;
        G_Stick_Breaks.t2 = tolerance ;
        G_Stick_Breaks.t3 = tolerance ;
    }
    return ;
}
/*---------------------------------------------------------------------*/
                    
/* end joystkmd.c  */
