/********* Listing 2 *************** PALDEMO.C *******
 *                                                   *
 * Palette Demonstration Program for EGA/VGA colors. *
 * Tested on: Turbo 2.0+, MS 5.0+, Zortech C/C++ 2.0+*
 * by:  Marv Luse, Autumn Hill Software, Inc.        *
 * (c) 1991 Use freely but acknowledge authorship.   *
 *****************************************************/

#include "stdlib.h"
#include "stdio.h"
#include "conio.h"
#include "dos.h"
#include "palette.h"

#define  OFF  0
#define  ON   1

#define INDEX_COL   26
#define COLOR_COL   33
#define R_COL       41
#define G_COL       46
#define B_COL       51

#define HDR_ROW      4
#define CLR_ROW      6

char *hdr = "Index  Color  Red  Grn  Blu";
char *blk = "\xDB\xDB\xDB\xDB\xDB";
char *msg = 
 "\x19 INDEX \x18     \x1B RGB \x1A     - VALUE +";

int   iclr   = 0; /* color index in range 0-15 */
int   irgb   = 0; /* which primary - 0=r, 1=g, 2=b */
int   icol[] = { R_COL, G_COL, B_COL };

/*--------------------------------------------------*/
/* Function to read byte from specified address...  */
/*--------------------------------------------------*/
int get_byte( unsigned short seg, unsigned short ofs )
{
    unsigned char far *p;

    p = (unsigned char far *) 
        ( (((unsigned long) seg)<<16) | ofs );
    return( (int) *p );
}

/*--------------------------------------------------*/
/* Function to write byte to specified address...   */
/*--------------------------------------------------*/
void put_byte( int byte_val, unsigned short seg, 
                              unsigned short ofs )
{
    unsigned char far *p;

    p = (unsigned char far *) 
               ( (((unsigned long) seg)<<16) | ofs );
    *p = (unsigned char) byte_val;
}

/*--------------------------------------------------*/
/* Function to display char at given row and column */
/*--------------------------------------------------*/
void display_ch( int ch, int row, int col, int atr )
{
    unsigned short seg, ofs;

    /* address into video mem */
    seg = 0xB800;
    ofs = (row - 1)*160 + (col - 1)*2;

    /* displayed character */
    put_byte( ch, seg, ofs );

    /* displayed attribute */
    put_byte( atr, seg, ofs+1 );
}

/*--------------------------------------------------*/
/* Display a string at given row and column...      */
/*--------------------------------------------------*/
void display_str(char *str, int row, int col, int atr)
{
    while( *str )
        display_ch( *str++, row, col++, atr );
}

/*--------------------------------------------------*/
/*  Turns hardware cursor on or off...              */
/*--------------------------------------------------*/
void set_cursor ( int state )
{
    union REGS reg;

    if( state ) /* show cursor */
        reg.x.cx = 0x0607;
    else        /* hide cursor */
        reg.x.cx = 0x2000;
    reg.x.ax = 0x0100;
    int86 (0x10, &reg, &reg);
}

/*--------------------------------------------------*/
/*  Clear the screen...                             */
/*--------------------------------------------------*/
void clear_screen( void )
{
    int i, j;

    for( i=1; i<=25; i++ )
        for( j=1; j<=80; j++ )
            display_ch( ' ', i, j, 7 );
}

/*--------------------------------------------------*/
/*  Get RGB values for passed index...              */
/*--------------------------------------------------*/
void get_rgb( int i, unsigned char rgb[] )
{
    int r;

    if( is_ega_color() )
    {
        r = get_ega_pal_register( i );
        rgb[0] = (unsigned char) ((r & 0x04) >> 1) |
                 ((r & 0x20) >> 5);
        rgb[1] = (unsigned char)  (r & 0x02)       |
                 ((r & 0x10) >> 4);
        rgb[2] = (unsigned char) ((r & 0x01) << 1) |
                 ((r & 0x08) >> 3);
    }
    else
    {
        r = get_vga_pal_register( i );
        get_vga_dac_register( r, rgb );
    }
}

/*--------------------------------------------------*/
/*  Set RGB values for passed index...              */
/*--------------------------------------------------*/
void set_rgb( int i, unsigned char rgb[] )
{
    int r;

    if( is_ega_color() )
    {
        r  = 0;
        r |= ((rgb[0] & 0x02) << 1) | 
             ((rgb[0] & 0x01) << 5);
        r |= (rgb[1] & 0x02)        | 
             ((rgb[1] & 0x01) << 4);
        r |= ((rgb[2] & 0x02) >> 1) | 
             ((rgb[2] & 0x01) << 3);
        set_ega_pal_register( i, r );
    }
    else
    {
        r = get_vga_pal_register( i );
        set_vga_dac_register( r, rgb );
    }
}

/*--------------------------------------------------*/
/*  Draw the palette demo display...                */
/*--------------------------------------------------*/
void draw_display( void )
{
    char          line[40];
    unsigned char rgb[3];
    int           i;

    if( is_ega_color() )
        display_str( " EGA Color Palette Demo ", 
                       2, 27, 0x0F );
    else
        display_str( " VGA Color Palette Demo ", 
                       2, 27, 0x0F );

    display_str( hdr, HDR_ROW, INDEX_COL, 0x0F );

    for( i=0; i<16; i++ )
    {
        /* get rgb values for color index i */
        get_rgb( i, rgb );

        /* format display line */
        sprintf( line, 
            "  %2d           %02d   %02d   %02d",
            i, rgb[0], rgb[1], rgb[2] );
        display_str(line,i+CLR_ROW, INDEX_COL, 0x0F);
        display_str(blk, i+CLR_ROW, COLOR_COL, i);
    }

    display_str( msg, 24, 22, 0x0F );
}

/*--------------------------------------------------*/
/*  Mark or Unmark an index (row)...                */
/*--------------------------------------------------*/
void mark_index( int row, int mark )
{
    int ch;

    ch = mark ? 0x10 : 0x20;
    display_ch( ch, CLR_ROW+row, INDEX_COL, 0x0F );
}

/*--------------------------------------------------*/
/*  Mark or Unmark an RGB column...                 */
/*--------------------------------------------------*/
void mark_rgb( int col, int mark )
{
    int ch;

    ch = mark ? 0x1F : 0x20;
    display_ch( ch, HDR_ROW+1, icol[col], 0x0F );
}

/*--------------------------------------------------*/
/*  Change r, g, or b of index by specified amount  */
/*--------------------------------------------------*/
void change_rgb( int index, int pri, int delta )
{
    unsigned char rgb[3];
    char          val[8];
    int           p, pmax;

    /* get rgb values for color index */
    get_rgb( index, rgb );
    pmax = is_ega_color() ? 3 : 63;

    /* modify specified primary */
    p = rgb[pri];
    p += delta;
    if( p < 0 ) p = 0;
    if( p > pmax ) p = pmax;
    rgb[pri] = (unsigned char) p;

    /* set new primary rgb */
    set_rgb( index, rgb );

    /* update display */
    sprintf( val, "%02d", p );
    display_str(val,index+CLR_ROW,icol[pri],0x0F);
}

/*--------------------------------------------------*/
/*  Do the demo...                                  */
/*--------------------------------------------------*/
void do_demo( void )
{
    int done, ch;

    mark_index( iclr, 1 );
    mark_rgb( irgb, 1 );

    done = 0;
    while( ! done )
    {
        ch = getch();

        switch( ch )
        {
            case 13 : /* enter */
            case 27 : /* escape */
                done = 1;
                break;

            case 72 : /* up arrow */
                mark_index( iclr, 0 );
                iclr--;
                if( iclr < 0 ) iclr = 15;
                mark_index( iclr, 1 );
                break;

            case 80 : /* dn arrow */
                mark_index( iclr, 0 );
                iclr++;
                if( iclr > 15 ) iclr = 0;
                mark_index( iclr, 1 );
                break;

            case 75 : /* left arrow */
                mark_rgb( irgb, 0 );
                irgb--;
                if( irgb < 0 ) irgb = 2;
                mark_rgb( irgb, 1 );
                break;

            case 77 : /* right arrow */
                mark_rgb( irgb, 0 );
                irgb++;
                if( irgb > 2 ) irgb = 0;
                mark_rgb( irgb, 1 );
                break;

            case 43 : /* plus */
                change_rgb( iclr, irgb, 1 );
                break;

            case 45 : /* minus */
                change_rgb( iclr, irgb, -1 );
                break;
        }
    }
}

/*--------------------------------------------------*/
/*  Save our palette to disk...                     */
/*--------------------------------------------------*/
void save_palette( void )
{
    int            i;
    unsigned char  rgb[3];
    FILE          *fpal;

    fpal = fopen( "PALDEMO.PAL", "wt" );
    for( i=0; i<16; i++ )
    {
        get_rgb( i, rgb );
        fprintf( fpal, "%d %d %d  ", 
                 rgb[0], rgb[1], rgb[2] );
    }
    fclose( fpal );
}

/*--------------------------------------------------*/
/*  Get our palette from disk...                    */

void read_palette( void )
{
    int            i, r, g, b;
    unsigned char  rgb[3];
    FILE          *fpal;

    fpal = fopen( "PALDEMO.PAL", "rt" );
    if( fpal == NULL ) return;
    for( i=0; i<16; i++ )
    {
        fscanf( fpal, "%d %d %d", &r, &g, &b );
        rgb[0] = (unsigned char) r;
        rgb[1] = (unsigned char) g;
        rgb[2] = (unsigned char) b;
        set_rgb( i, rgb );
    }
    fclose( fpal );
}

/*--------------------------------------------------*/
/*  Palette demo program...                         */
/*--------------------------------------------------*/
void main( void )
{
    int i, r;

    if((is_ega_color()==0) && (is_vga_color() == 0))
    {
        puts(
        "\nThis pgm requires EGA or VGA display!" );
        exit( 0 );
    }

    set_cursor( OFF );
    clear_screen();

    /* on EGA, we must establish our own PSA area */
    if( is_ega_color() )
    {
        init_save_area( 1 );
        /* fill in save area */
        for( i=0; i<16; i++ )
        {
            r = get_ega_pal_register( i );
            set_ega_pal_register( i, r );
        }
    }

    read_palette();
    draw_display();
    do_demo();
    save_palette();

    /* on EGA, restore default PSA area */
    if( is_ega_color() )
        restore_save_area( 1 );

    clear_screen();
    set_cursor( ON );
}