/* texport.c */

/***********************************************************************/
/*                                                                     */
/* Command-line program which will export a CRDE table to either ascii */
/* dBASE III Plus formats.                                             */
/*       							       */
/* Usage: texport [options] <tablename>                                */
/*                                                                     */
/* To compile this program requires the creation of a project file     */
/* which should look something like                                    */
/*                                                                     */
/*   texport                                                           */
/*   crde.lib                                                          */
/*                                                                     */
/***********************************************************************/


#include <dir.h>
#include <stdio.h>
#include <string.h>
#include "crde.h"

/* output formats */

#define DEFAULT 0
#define ASCII	1
#define dBASE  	2

/* main program */

int tbuffers = 128;

int main(int argc, char *argv[])
{
  TABLE *t;
  char path[MAXPATH], drive[MAXDRIVE], dir[MAXDIR], file[MAXFILE],
    ext[MAXEXT];
  int n, flags, type;
  long result;

  /* print help screen if no parameters */
  if (argc < 2) {
    printf("\nusage: texport [options] <CRDE tablename> <destination>\n");
    printf("command-line options:\n");
    printf("\t-a\toutput ascii file (default)\n");
    printf("\t-d\toutput dBASE file (default if '.DBF' extension)\n");
    printf("\n\n");
    exit(1);
  }

  /* get command-line options */
  type = DEFAULT;
  for (n = 1; n < argc && *argv[n] == '-'; n++) {
    switch (*(argv[n]+1)) {
      case 'a' : type = ASCII; break;
      case 'd' : type = dBASE; break;
      default  :
	printf("-%c is not a valid command line option.", *(argv[n]+1));
	exit(1);
    }
  }

  /* no source file */
  if (n == argc) {
    printf("You did not specify a table to export.\n");
    exit(1);
  }

  /* get destination file */
  if (n == argc - 1)
    strcpy(path, argv[n]);
  else
    strcpy(path, argv[n+1]);

  /* determine default type */
  flags = fnsplit(path, drive, dir, file, ext);
  if (type == DEFAULT)
    if ((flags & EXTENSION) && (strcpy(strupr(ext), "DBF")))
      type = dBASE;
    else
      type = ASCII;

  /* default extension */
  if (n == argc - 1)
    switch (type) {
      case ASCII : strcat(path, ".TXT"); break;
      case dBASE : strcat(path, ".DBF"); break;
    }

  /* open table */
  if ((t = topen(argv[n])) == NULL) {
    printf("Unable to open table (error code = %d).\n", terrno);
    exit(1);
  }

  /* export table */
  switch (type) {
    case ASCII : result = texportascii(t, path); break;
    case dBASE : result = texportdBASE(t, path); break;
  }

  /* check results */
  if (result < 0)
    printf("Export failed (error code %d occurred).\n", terrno);
  else {
    printf("Export to ");
    switch (type) {
      case ASCII : printf("ascii"); break;
      case dBASE : printf("dBASE"); break;
    }
    printf(" file %s was successful.\n", path);
    printf("%ld rows exported.\n", result);
  }
  tclose(t);
}


/* end of texport.c */
