;-----------------------------------------------------------------------;
; STRTRIM.ASM                                                           ;
;                                                                       ;
; This module contains a C-callable function to remove leading and      ;
; trailing white space from a string.  The function is passed a         ;
; pointer to the string, and processes the string in place.  It         ;
; returns a pointer to the string that it processed.                    ;
;                                                                       ;
; This module recognizes a _model symbol, and can therefore             ;
; generate object code for any model desired.  If _model is not         ;
; defined, the SMALL model is used by default.                          ;
;                                                                       ;
;-----------------------------------------------------------------------;
; Function prototype:                                                   ;
;                                                                       ;
; char *strtrim (char *string)                                          ;
;-----------------------------------------------------------------------;
; Revision history:                                                     ;
;                                                                       ;
;       1.0     19 APR 92       Original.                               ;
;                                                                       ;
;                               Uses conditional assembly for all       ;
;                               memory models by defining _model.       ;
;                               Defaults to the small model if          ;
;                               _model is not defined.                  ;
;-----------------------------------------------------------------------;
;   Copyright (c) 1992 Ray Waters                                       ;
;   All Rights Reserved                                                 ;
;-----------------------------------------------------------------------;

IFDEF   _model                          ; if a _model was defined,
        .MODEL  _model, C               ;  use it
ELSE                                    ; else, default to
        .MODEL  SMALL, C                ;  SMALL model
ENDIF

IFDEF   ??version                       ; if using TASM,
        LOCALS                          ;  enable local labels
ENDIF

space   EQU     20h                     ; ASCII space or blank
tab     EQU     9h                      ; ASCII tab
lf      EQU     0Ah                     ; ASCII line feed
vtab    EQU     0Bh                     ; ASCII vertical tab
ff      EQU     0Ch                     ; ASCII formfeed
cr      EQU     0Dh                     ; ASCII carriage return

        .CODE                           ; open code segment

        PUBLIC  strtrim                 ; make visible to Linker
;-----------------------------------------------------------------------;
; char *strtrim (char *string)                                          ;
;                                                                       ;
; Returns:  pointer to the string processed                             ;
;-----------------------------------------------------------------------;
strtrim PROC    C USES si di, string:PTR BYTE

IF @DataSize                            ; if large data model,
        push    ds                      ;  save original data segment
        lds     si,[string]             ; load string address in DS:SI
ELSE
        mov     si,[string]             ; load string address in DS:SI
ENDIF
        mov     bx,si                   ; save pointer offset to string
        mov     ax,ds                   ; set ES = DS
        mov     es,ax
        mov     di,si                   ; set DI = SI
        cld                             ; strings forward

@@skip_space:
        lodsb                           ; get a byte
        cmp     al,space                ; is it a space?
        je      @@skip_space            ; if yes, skip it
        cmp     al,tab                  ; is it a tab
        je      @@skip_space            ; if yes, skip it
        cmp     al,lf                   ; is it a line feed?
        je      @@skip_space            ; if yes, skip it
        cmp     al,cr                   ; is is a carriage return?
        je      @@skip_space            ; if yes, skip it
        cmp     al,ff                   ; is it a formfeed?
        je      @@skip_space            ; if yes, skip it
        cmp     al,vtab                 ; is it a vertical tab?
        je      @@skip_space            ; if yes, skip it
        stosb                           ; else, copy it back to buffer
        or      al,al                   ; is it the null byte?
        jz      @@done                  ; if yes, finished

@@copy_chars:                           ; loop to copy rest of characters
        lodsb                           ; get another character
        stosb                           ; store the character
        or      al,al                   ; was it the null character?
        jnz     @@copy_chars            ; if not, loop until finished

        dec     di                      ; back DI up to null character
        dec     di                      ;  and then the last string char
        mov     si,di                   ; set SI = DI
        std                             ; now skip white space in reverse

@@trim_trailing:
        lodsb                           ; get a byte
        cmp     al,space                ; is it a space?
        je      @@trim_trailing         ; yes, keep going
        cmp     al,tab                  ; is it a tab?
        je      @@trim_trailing         ; yes, keep going
        cmp     al,lf                   ; is it a line feed?
        je      @@trim_trailing         ; yes, keep going
        cmp     al,cr                   ; is it a carriage return?
        je      @@trim_trailing         ; yes, keep going
        cmp     al,ff                   ; is it a formfeed?
        je      @@trim_trailing         ; yes, keep going
        cmp     al,vtab                 ; is it a vertical tab?
        je      @@trim_trailing         ; yes, keep going
        mov     BYTE PTR [si+2],0       ; no, finished -- add terminating null

@@done:
IF @DataSize                            ; if large data model,
        mov     dx,ds                   ; return pointer segment in DX
        pop     ds                      ; restore original data segment
ENDIF
        mov     ax,bx                   ; return pointer offset in AX
        ret                             ; return to caller

strtrim ENDP

        END
