/*  FCOPY.C  --  contains fcopy() function                              */
/*  Version 2.2                                                         */
/*                                                                      */
/*  This function copies one file to another, much as the DOS COPY      */
/*  command does.  It does not check to see if the two files are        */
/*  the same, so it is up to the caller to verify that the two file     */
/*  specs do not, in fact, refer to the same file.                      */
/*                                                                      */
/*  fcopy() uses a farmalloc() call to allocate a large file buffer     */
/*  from the far heap, regardless of the memory model being used.  The  */
/*  actual copy operation is performed in the assembler function        */
/*  _copyfile().  The source code for this module is contained in       */
/*  COPYFILE.ASM.  This function is a little smaller and slightly       */
/*  faster than using the separate calls to _farread() and _farwrite()  */
/*  functions (as was done in version 1.1), since the far pointer is    */
/*  only loaded once with _copyfile().                                  */
/*                                                                      */
/*  BUFF_SIZE, indicating the buffer size, can be any int sized         */
/*  value up to (but not including) 65,535.  It is recommended that     */
/*  the buffer be a multiple of the sector size (512 bytes in the       */
/*  current version of DOS), since reads and writes are more efficient  */
/*  if they involve whole sectors.  A good value to use is 65024 (in    */
/*  decimal), since this is 1 sector (512 bytes) short of a complete    */
/*  segment (64k bytes).                                                */
/*                                                                      */
/*  Function prototypes are contained in fcopy.h                        */
/*                                                                      */
/*  Usage:                                                              */
/*                                                                      */
/*      #include "fcopy.h"                                              */
/*      int fcopy (char *sourcename, char *targetname)                  */
/*                                                                      */
/*  The function returns 0 if sucessful, -1 otherwise.  Error codes     */
/*  are returned in the global variables errno and _doserrno.           */
/*  Possible error codes are:                                           */
/*                                                                      */
/*      EINVFNC     Invalid function number  (1)                        */
/*      ENOFILE     File not found  (2)                                 */
/*      ENOPATH     Path not found  (3)                                 */
/*      EMFILE      Too many open files  (4)                            */
/*      EACCESS     Permission denied  (5)                              */
/*      EBADF       Bad file number  (6)                                */
/*      ECONTR      Memory blocks destroyed  (7)                        */
/*      ENOMEM      Not enough core  (8)                                */
/*      EINVMEM     Invalid memory block address  (9)                   */
/*      EINVACC     Invalid access code  (12)                           */
/*           -2     Target disk full  (-2)                              */
/*  ------------------------------------------------------------------- */
/*  Revision history:                                                   */
/*                                                                      */
/*      1.0      5 JAN 92       Original.  Written for Turbo C++.       */
/*                                                                      */
/*      1.1     19 JAN 92       Updated comments to reflect _farread()  */
/*                              and _farwrite() being split into        */
/*                              separate modules.  No code changes to   */
/*                              this module.                            */
/*                                                                      */
/*      2.0     16 MAR 92       Replaced _farread() and _farwrite()     */
/*                              functions with a single _copyfile()     */
/*                              function.  fcopy() modified to match.   */
/*                                                                      */
/*      2.1     12 MAY 92       Dropped the leading underscore from     */
/*                              the function name; it is now fcopy      */
/*                              (rather than _fcopy).                   */
/*                                                                      */
/*      2.2     27 JUN 92       Added static modifier to __cleanup      */
/*                              function to limit its scope to this     */
/*                              module.                                 */
/*                                                                      */
/*                              Now deletes the bad target file if      */
/*                              the file copy fails to avoid leaving    */
/*                              it on disk and forcing the calling      */
/*                              function to deal with it.               */
/*  ------------------------------------------------------------------- */
/*      Copyright (c) 1992 Ray Waters                                   */
/*      All Rights Reserved                                             */
/*  ------------------------------------------------------------------- */

#define BUFF_SIZE   65024U          /* size of file buffer */

#include <io.h>             /* _open, _creat, _close */
#include <stdio.h>          /* remove */
#include <fcntl.h>          /* definition for O_RDONLY */
#include <alloc.h>          /* farmalloc, farfree */
#include "fcopy.h"          /* fcopy, _copyfile */

/* ----- [ function prototype - limits scope to this module ] ----- */
static void __cleanup(void);    /* function to close files, free memory */

char far *buffer;               /* far pointer to the file buffer */
int  SrcHandle, TgtHandle;      /* file handles for source and target */

int fcopy(char *sourcename, char *targetname)
{
    struct ftime ft;                /* structure for file date/time stamp */

    SrcHandle = _open(sourcename, O_RDONLY);    /* open source (read only) */
    if (SrcHandle == -1)                        /* if open failed, */
        return -1;                              /* return error */
    if (getftime(SrcHandle, &ft)) {             /* get date/time stamp */
        _close(SrcHandle);                      /* if error, close file */
        return -1;                              /* return error */
    }

    TgtHandle = _creat(targetname, 0);          /* create/truncate target */
    if (TgtHandle == -1) {                      /* if open failed, */
        _close(SrcHandle);                      /* close source file */
        return -1;                              /* return error */
    }

    if  (!(buffer = farmalloc(BUFF_SIZE))) {    /* allocate a far buffer */
                                                /* if allocation failed, */
        _close(SrcHandle);                      /* close source file */
        _close(TgtHandle);                      /* close target file */
        return -1;                              /* return error */
    }

                                                /* perform the copy */
    if (_copyfile(SrcHandle, TgtHandle, buffer, BUFF_SIZE)) {
        __cleanup();                /* if copy failed, close files, free mem */
        remove(targetname);                     /* delete bad target file */
        return -1;                              /* return error */
    }

    setftime(TgtHandle, &ft);               /* set target date/time stamp */
    __cleanup();                                /* close files, free mem */
    return 0;                                   /* return success */
}

static void __cleanup(void)             /* close files and release memory */
{
    _close(SrcHandle);                          /* close source file */
    _close(TgtHandle);                          /* close target file */
    farfree(buffer);                            /* free memory */
}
