/*[]------------------------------------------------------------[]*/
/*|                                                              |*/
/*|   'Example of how to write an INT 21 handler in 'C' using    |*/
/*|           a "code segment" function pointer'                 |*/
/*|                                                              |*/
/*|     Copyright (c) 1991 by Borland International              |*/
/*|                  All Rights Reserved.                        |*/
/*|                                                              |*/
/*[]------------------------------------------------------------[]*/

/*
   When using ISR's, you will almost always call the old handler for
   your interrupt either before or after you do your processing.  This
   is done by declaring a pointer to an interrupt function and saving
   the old vector into this pointer.  The problem occures when you want
   to hook a system critical int. (like 21h, or 16h or 10h...) which
   needs to have ALL the registers restored so that when it is called
   the temporrary stop in your function is transparent.  If you call DOS
   and it wants a certain value in a particular register then it darn well
   better be in that register when it gets to the real DOS handler,
   regardless of what happened along the way.  To call the old handler
   from 'C' we invoke the function at the address contained by the
   function pointer that we originally saved away.  That function
   pointer is stored in the DATA SEGMENT and accesed via the DS
   register.  So, how is it possible to reset DS to its origional value
   (possibly a parameter you are passing to DOS) and still refer to
   something in our DATA SEGMENT?  What we do is to allocate a function
   pointer which lives in the CODE SEGMENT instead.  Now the compiler
   will never allow you to do this directly, but we can trick it so that
   it's happy and we can still access out data (in the code segment).
   In this example the function 'my_addr' does nothing except to point
   to a chunk of memory in the code segment.  This is where the code of
   this function would normally be executed.  However, we are going to
   be sure never to call this function, and then write over the code
   that's in its place.  An empty function would normally generate this:

   void my_addr (void)   |    PROC _my_addr
                         |                    PUSH BP          ; (1)
                         |                    MOV BP, SP       ; (2)
   {                     |                ;
   }                     |                ;
                         |                    POP BP           ; (1)
                         |                    RET              ; (1)

   The number of bytes each assembly instruction takes up is listed to
   on the right in parens.  You can see that even with no code in the
   function, the compiler still generates 5 bytes.  We are going to use
   four of these five bytes for our function pointer.

   I have provided 3 'C' macros for preforming the inline assembly tasks
   of setting the cs_pointer and then calling the function which it
   points to:

     SET_CS_VECT(my_func, old_vector_pointer) -- set up my_func
     CALL_CS_VECT(my_func)      -- same segment
     CALL_CS_VECT_386(my_func)  -- different segment (requires a 386)

   Because we are putting a variable into the segment which CS points
   to, we need to worry about whether that value will ever change.
   Therefore you MUST put the 'my_addr' function in the same source
   file as the ISR which calls the macro CALL_CS_VECT.  If you have a
   386 machine, you can put it in any source file and call the macro
   CALL_CS_VECT_386 instead.

*/

 /* by Jeff Peters */

 #include <stdio.h>
 #include <dos.h>

 #pragma inline     // Don't use built in assembler in BC++

 #define MAX_ 30000

 #define SET_CS_VECT(addr, oldval)                                  \
       asm   mov dx, seg (addr);                                    \
       asm   mov es, dx;                                            \
       asm   push ds;                                               \
       asm   lds bx, (oldval);                                      \
       asm   mov word ptr es:[(addr)], bx;                          \
       asm   mov word ptr es:[(addr)+2], ds;                        \
       asm   pop ds

 #define CALL_CS_VECT(addr)                                         \
                  asm pushf;                                        \
                  asm call dword ptr cs:[(addr)]

 #define CALL_CS_VECT_386(addr)                                     \
                  asm .386;                                         \
                  asm push ax;                                      \
                  asm mov ax, seg (addr);                           \
                  asm mov fs, ax;                                   \
                  asm pop ax;                                       \
                  asm pushf;                                        \
                  asm call dword ptr fs:[(addr)]


 void my_addr (void);

 void interrupt get_out(); /* interrupt prototype */

 void interrupt (*oldfunc)(); /* interrupt function pointer */
 int looping = 1;

 int main(void)
 {
     puts ("\nThis example will count up to 30000 for every call to DOS\n");
     puts("Press <Esc> to exit\n");

   /* save the old interrupt */
   oldfunc  = getvect(0x21);

   SET_CS_VECT (my_addr, oldfunc);  // set up cs function pointer

    /* install interrupt handler */
   setvect(0x21,get_out);

    /* do nothing */
   while (looping <= MAX_)
   {
     printf ("\rloop = %d", looping);
     if (kbhit())
     {
       char c;
       if ((c = getch()) == 0)
         getch ();
       else
       if (c == 27)
        break;
     }
   }

    /* restore to original interrupt routine */
    setvect(0x21,oldfunc);

   if (looping < MAX_)
     puts ("\nLoop terminated by user.");

   puts("\nSuccess");
   return 0;
}

 void my_addr (void) // DANGER: Don't ever call this function!!!!!!
 {
  /*
     This function is only used only to store data in the code segment.
     If you call it, the code to return (which the compiler generated
     for you) will have been overwritten and you will hang the machine.
  */

  /* NOTE: This function MUST be in the same source file as the
           interrupt function which calls it with 'CALL_CS_VECT', unless
           you are on a 386 machine and you use 'CALL_CS_VECT_386'
           instead. Then you can have it in any source file.
  */
 }

void interrupt get_out (bp,di,si,ds,es,dx,cx,bx,ax,ip,cs,flags)
{
  looping++; /* global counter for the number of calls to this interrupt */

  /* pre-interrupt code goes here */


  _DS = ds;      // reset the ds before the INT21 call
  CALL_CS_VECT (my_addr); // call the old vect -- MUST be in the same seg
//  CALL_CS_VECT_386 (my_addr); // call using 386 reg to access other code seg

  /* post-interrupt code goes here */

  /* if you want to access global var's here be sure to do an
     asm push ds;    // before the _DS = ds...    and
     asm pop ds;     // after the CALL_CS_VECT...
  */


  ax = _AX;
  bx = _BX;
  cx = _CX;
  dx = _DX;
  es = _ES;
  si = _SI;
  di = _DI;
  ds = _DS;
}

