extern "C" {
    #include "memory.h"
};
#include "bigarray.hpp"

// Constructor for BigArray
BigArray::BigArray( long          NElements,
	                unsigned      ElementSize,
	                char         *AllocStrategy) {

    // Work out the size of the Array
    ArraySize =	NElements * ElementSize;
    if (!ArraySize) {status = 0;return;}

    // Decide on a sensible line length. A 2K buffer gives reasonable
    // Performance, but the line must be a multiple of the ElementSize

    if (ElementSize > 2048) Width = ElementSize;
    else Width = (2048/ElementSize) * ElementSize;

    // So how many of those lines will we need?
    Lines = ArraySize/Width + (ArraySize%Width ? 1:0);

    // Now attempt to allocate a buffer
    setXMstrat(AllocStrategy);
    Handle = xalloc(Width,Lines);
    if (!Handle) {
        status = 0;
        return;
    }

    // Ok, we have a buffer. Assign the other stuff, get the first line, etc
    offset = 0;
    Item   = 0;
    Items  = NElements;
    Line   = 0;
    step   = ElementSize;
    status = 1;
    LineBuf= xget(0,Handle);
}

/***    Destructor Simply Frees The Buffer
***/
BigArray::~BigArray() { xfree(Handle);}

/***    Type of memory allocated
***/
char *BigArray::Type(void) {status=1;return mem_type(Handle);}

/***    Error test
***/
int BigArray::OK(void) {return status;}
int BigArray::operator!(void) {return !status;}

/***    '*' (indirection) operator overload
        Return a pointer to the current element as a byte *
***/
byte *BigArray::operator*(void) {
    status=1;
    offset = Item * step;   // Reset the offset pointer always
    return LineBuf + ((long)Item*step - (long)Line*Width);
}

/***    '[]' Punctuator Overload
        Reposition array to given item, set status and return a pointer to it
***/
byte *BigArray::operator[](unsigned long i) {

    offset = Item * step;   // Reset the offset pointer always

    if (i>=Items) {status=0; return NULL;}

    Item   = i;
    offset = i * step;
    Line   = offset/Width;
    LineBuf=xget(Line,Handle);
    status = (LineBuf?1:0);
    return LineBuf + ((long)Item*step - (long)Line*Width);
}

/***    Increment/Decrement operator overloads
***/
int BigArray::operator++(void)
{
    offset = Item * step;   // Reset the offset pointer always

    if (Item+1>=Items) {status=0; return 0;}

    Item++;
    offset = Item * step;
    Line   = offset/Width;
    LineBuf= xget(Line,Handle); // Will come right back if it's the same line
    status = (LineBuf?1:0);

    return status;
}

int BigArray::operator--(void)
{
    offset = Item * step;   // Reset the offset pointer always

    if (!Item) {status=0; return 0;}

    Item--;
    offset = Item * step;
    Line   = offset/Width;
    LineBuf= xget(Line,Handle); // Will come right back if it's the same line
    status = (LineBuf?1:0);

    return status;
}

/***    >> overload:    Extract this byte and increment offset
***/
int BigArray::operator>>(byte &recipient)
{
    if (offset >= ArraySize) {status=0;return -1;}

    recipient = LineBuf[offset - ((long)Line * Width)];

    offset++;
    status=1;

    if (offset >= ArraySize) return 0; // Don't get next line if we're
                                            // At the end

    // Maybe move onto next line
    if (offset/step != Item) {
        Item   = offset/step;
        Line   = offset/Width;
        LineBuf= xget(Line,Handle);
        status = (LineBuf?1:0);
    }

    return 0;
}

/***    << overload:    Store this byte and increment offset
***/
int BigArray::operator<<(byte newval)
{
    if (offset >= ArraySize) {status=0;return -1;}

    LineBuf[offset - ((long)Line * Width)] = newval;

    offset++;
    status=1;

    if (offset >= ArraySize) return status; // Don't get next line if we're
                                            // At the end

    // Maybe move onto next line
    if (offset/step != Item) {
        Item   = offset/step;
        Line   = offset/Width;
        LineBuf= xget(Line,Handle);
        status = (LineBuf?1:0);
    }

    return status;
}
