/*
**                ---  selftest.c ---
**
**  SELFTEST requires two serial ports on the same computer. The
**  program transmits a test string on one port (FirstCOM) and
**  receives on a second port (SecondCOM), where the two ports are
**  connected via a null modem adapter. The received string is tested
**  against the transmit string (they should be idenical).
**
**  Connect the two serial ports (on a single computer) together
**  using a null modem cable. Be sure to modify the configuration
**  section for non-standard PC ports or to setup your multiport
**  board. Note that many multiport boards are either Digiboard or
**  BOCA board compatible.
**
**  IMPORTANT: You may have to modify the port address & IRQ to match
**  your Digiboard or BOCA board installation.
**
**  Compile with /DDPMI for protected mode testing.
*/

#ifdef DPMI
#include "windows.h"
#include "use_dpmi.h"
#endif
#include <stdio.h>
#include <stdlib.h>
#include <malloc.h>
#include <dos.h>
#include <string.h>
#include <conio.h>
#include "pcl4c.h"

#define ESC 0x1b

#define READ_PORT(P) inportb(X)

#define PC 1
#define DB 2
#define BB 3

/*** Global Variables ***/

int BaudCode = Baud9600;  /* Code for 9600 baud  */
char *TestString = "This is a test string";
int TestLength;
int FirstCOM;
int SecondCOM;

int MultiUART = 0;
int MultiIRQ = 0;
int MultiStatus = 0;

void AllDone(void);
void Display(int);
void CheckAlloc(int,char *);
int  ErrorCheck(int);
int  AllocSeg(int);

static int  RxSeg1 = 0;
static int  TxSeg1 = 0;
static int  RxSeg2 = 0;
static int  TxSeg2 = 0;

char *Ptr[4] = {"SMALL","COMPACT","MEDIUM","LARGE"};

/* standard PC port configuration */

static int StdAdrPC[4] = {0x3f8,0x2f8,0x3e8,0x2e8};
static int StdIrqPC[4] = {4,3,4,3};

static int Selection = 0;  /* PC, DB, or BB */

void main(int argc, char *argv[])
{int Port;
 char *P;
 int ComLimit = 0;
 char c;
 int Version;
 int Model;
 int i, rc;
 if(argc!=4)
  {printf("Usage: selftest {pc|db|bb} 1stCom 2ndCom\n");
   exit(1);
  }
 P = argv[1];
 if((strcmp(P,"pc")==0)||(strcmp(P,"PC")==0)) Selection = PC;
 if((strcmp(P,"db")==0)||(strcmp(P,"DB")==0)) Selection = DB;
 if((strcmp(P,"bb")==0)||(strcmp(P,"BB")==0)) Selection = BB;
 if(Selection==0)
   {puts("Must specify 'PC', 'DB' or 'BB' as 1st argument");
    puts("EG:  SELFTEST PC 1 4");
    exit(1);
   }
 if(Selection==PC) ComLimit = COM4;
 if(Selection==DB) ComLimit = COM8;
 if(Selection==BB) ComLimit = COM16;
 FirstCOM = atoi(argv[2]) -1;
 SecondCOM = atoi(argv[3]) -1;
 printf("FirstCOM  = COM%d\n",1+FirstCOM);
 printf("SecondCOM = COM%d\n",1+SecondCOM);
 if(FirstCOM<COM1)
   {puts("1stCom must be >= COM1");
    exit(1);
   }
 if(SecondCOM>ComLimit)
   {printf("2ndCom must be <= COM%d\n",1+ComLimit);
    exit(1);
   }
 if(FirstCOM>=SecondCOM)
   {puts("1stCom must be < 2ndCom");
    exit(1);
   }
 if(Selection==DB)
   {/*** Custom Configuration: DigiBoard PC/8 ***/
    MultiUART = 0x180; MultiIRQ = IRQ5; MultiStatus = 0x1C0;
    printf("[ Configuring DigiBoard as COM1-COM8 (IRQ%d) @ 0x%x ]\n",
       MultiIRQ,MultiUART);
    SioPorts(8,COM1,MultiStatus,DIGIBOARD);
    for(Port=COM1;Port<=COM8;Port++)
      {/* set DigiBoard UART addresses */
       ErrorCheck( SioUART(Port,MultiUART+8*Port) );
       /* set DigiBoard IRQ */
       ErrorCheck( SioIRQ(Port,MultiIRQ) );
      }
   }
 if(Selection==BB)
   {/*** Custom Configuration: BOCA BB2016 ***/
    MultiUART = 0x100; MultiIRQ = IRQ10; MultiStatus = MultiUART + 7;
    printf("[ Configuring BOCA Board as COM1-COM16 (IRQ%d) @ 0x%x ]\n",
       MultiIRQ,MultiUART);
    SioPorts(16,COM1,MultiStatus,BOCABOARD);
    for(Port=COM1;Port<=COM16;Port++)
      {/* set BOCA Board UART addresses */
       ErrorCheck( SioUART(Port,MultiUART+8*Port) );
       /* set BOCA Board IRQ */
       ErrorCheck( SioIRQ(Port,MultiIRQ) );
      }
   }
 if(Selection==PC)
   {/*** Standard Configuration: 4 port card ***/
    puts("[ Configuring for PC ]");
    for(i=COM1;i<=COM4;i++)
      {SioUART(i,StdAdrPC[i]);
       SioIRQ(i,StdIrqPC[i]);
      }
   }
 /* setup transmit & receive buffer */
 CheckAlloc(RxSeg1 = AllocSeg(128), "RX1");
 CheckAlloc(RxSeg2 = AllocSeg(128), "RX2");
 ErrorCheck( SioRxBuf(FirstCOM,RxSeg1,Size128) );
 ErrorCheck( SioRxBuf(SecondCOM,RxSeg2,Size128) );
 if(SioInfo('I'))
   {CheckAlloc(TxSeg1 = AllocSeg(128), "TX1");
    CheckAlloc(TxSeg2 = AllocSeg(128), "TX2");
    ErrorCheck( SioTxBuf(FirstCOM,TxSeg1,Size128) );
    ErrorCheck( SioTxBuf(SecondCOM,TxSeg2,Size128) );
    printf("Transmit buffers created\n");
   }
 /* set port parmameters */
 ErrorCheck( SioParms(FirstCOM,NoParity,OneStopBit,WordLength8) );
 ErrorCheck( SioParms(SecondCOM,NoParity,OneStopBit,WordLength8) );
 /* reset the ports */
 ErrorCheck( SioReset(FirstCOM,BaudCode) );
 ErrorCheck( SioReset(SecondCOM,BaudCode) );
 printf("***\n");
 printf("*** SELFTEST 4.0\n");
 printf("***\n");
 Version = SioInfo('V');
 Model = SioInfo('M');
 printf("*** Lib Ver : %d.%d\n",Version/16,Version%16);
 printf("***   Model : %s \n", Ptr[Model&3] );
 printf("*** TX Intr : ");
 if(SioInfo('I')) puts("Enabled.");
 else puts("Disabled.");
 printf("*** DPMI    : ");
 if(SioInfo('P')) printf("YES\n");
 else printf("NO\n");
 /* display port info */
 Display(FirstCOM);
 Display(SecondCOM);
 printf("***\n");

 printf("Starting @ 9600 baud. COM%d to COM%d\n",FirstCOM+1,SecondCOM+1);

 TestLength = strlen(TestString);
 /* send string */
 printf("  Sending: ");
 for(i=0;i<TestLength;i++)
   {c = TestString[i];
    SioPutc(FirstCOM,c);
    putch(c);
   }
 SioDelay(3);
 /* receive string */
 printf("\nReceiving: ");
 for(i=0;i<TestLength;i++)
   {rc = SioGetc(SecondCOM,18);
    if(rc<0)
      {printf("\nERROR: ");
       SioError(rc);
       SioDone(FirstCOM);
       SioDone(SecondCOM);
       exit(1);
      }
    /* echo just received char */
    putch((char)rc);
    /* compare character */
    if((char)rc!=TestString[i])
      {printf("\nERROR: Expecting '%c'(0x%x), received '%c'(0x%x)\n",
          (char)rc,(char)rc,TestString[i],TestString[i]);
       SioDone(FirstCOM);
       SioDone(SecondCOM);
       exit(1);
      }
   } /* end for */
 puts("\nSUCCESS: Test AOK !");
 AllDone();
 exit(0);
} /* end main */

int ErrorCheck(int Code)
{/* trap PCL error codes */
 if(Code<0)
     {SioError(Code);
      AllDone();
      exit(1);
     }
 return(0);
} /* end ErrorCheck */


void Display(int Port)
{printf("***    COM%d : ",1+Port);
 if(Selection==PC) printf("Adr=%3x IRQ=%d",StdAdrPC[Port],StdIrqPC[Port]);
 else printf("Adr=%3xh IRQ=%d Status=%xh",MultiUART+8*(Port-COM1),MultiIRQ,MultiStatus);
 if( SioFIFO(Port,LEVEL_8) ) printf(" [16550]\n");
 else printf(" [8250/16450]\n");
}


#ifdef DPMI

int AllocSeg(int Size)
{long hMem;
 int  Selector = 0;
 hMem = (long) GlobalDosAlloc( (long)Size );
 if(hMem)
   {/* get selector */
    Selector = LOWORD(hMem);
    GlobalPageLock(Selector);
    return Selector;
   }
 else return 0;
}

#else

int AllocSeg(int Size)
{int Seg;
 char far *Ptr;
 /* allocate far heap */
 Ptr = (char far *) _fmalloc(Size+16);
 if(Ptr==NULL) return 0;
 /* SEG:0 points to buffer */
 Seg = FP_SEG(Ptr) + ((FP_OFF(Ptr)+15)>>4);
 return Seg;
}

#endif

void AllDone(void)
{printf("Shutting down COM%d, ",1+FirstCOM);
 SioDone(FirstCOM);
 printf("COM%d.\n",1+SecondCOM);
 SioDone(SecondCOM);
 /* some more info */
 printf("%d RX Interrupts, ", SioInfo('R') );
 printf("%d TX Interrupts\n", SioInfo('T') );
#ifdef DPMI
 printf("Freeing DOS memory: ");
 /* free DOS memory */
 if(RxSeg1)
    {printf("RX1 ");
     GlobalPageUnlock(RxSeg1);
     GlobalDosFree(RxSeg1);
    }
 if(TxSeg1)
    {printf("TX1 ");
     GlobalPageUnlock(TxSeg1);
     GlobalDosFree(TxSeg1);
    }
 if(RxSeg2)
    {printf("RX2 ");
     GlobalPageUnlock(RxSeg2);
     GlobalDosFree(RxSeg2);
    }
 if(TxSeg2)
    {printf("TX2 ");
     GlobalPageUnlock(TxSeg2);
     GlobalDosFree(TxSeg2);
    }
 #endif
 exit(0);
}

void CheckAlloc(int Seg,char *Msg)
{if(Seg==0)
  {printf("Cannot allocate memory for %s\n",Msg);
   AllDone();
   exit(1);
  }
}