/*
File:    sci
Ext:     c

Title:   Serial Communication Interface,
         Interrupt Driven

Function       Operation

sci_init()     Serial Communication Interface Initilization            (PUBLIC)
sci_isr()      Serial Port Interrupt Service Routine (ISR)

putchar()      Write Character to Serial Port Queue.                   (PUBLIC)
getch()        Read Character from Serial Device w/o Echo.             (PUBLIC)
getchar()      Read Character from Serial Queue w Echo.                (PUBLIC)
_getkey()      Read Character from Serial Device w/o Echo.             (PUBLIC)

Revision Report

Date       By Description

06-SEPT-94 ET Create file.

*/

#define T1M1        0x20        /* tmod timer1 m1 bit */

static d-map int txqc;          /* Tx Queue data count */
static d-map int txqi;          /* Tx Queue data index */
#define TXQLEN 10               /* Tx data Queue Lenght */
static i-map char txq[TXQLEN];  /* Tx data Circular Queue */

static d-map int rxqc;          /* Rx Queue data count */
static d-map int rxqi;          /* Rx Queue data index */
#define RXQLEN 10               /* Rx data Queue Lenght */
static i-map char rxq[RXQLEN];  /* Rx data Circular Queue */


#define TESTING 1               /* COMPILE TESTING CODE IF 1 */

#if TESTING
main()
   {
   sci_init(); /* Initialize Serial Port Interface */
   ea = 1;     /* Enable Interrupts */

   while (1) /* loop forever */
       {
       printf("Hello, world\n");

       if (getrxqs()) /* if Rx Queue NOT EMPTY */
          {
          putchar(getch()); /* echo Rx Character */
          }
       }
   }
#endif


/* Serial Communication Interface Initilization

*/

sci_init()
   {
   /* setup serial port control hardware (19.2K baud @11.059 MHz) */

   sm1 = ti = 1;       /* 8-bit UART, transmitter ready */
   tmod = tmod | T1M1; /* timer1 8-bit auto-reload */
   tr1 = 1;            /* start timer1 */
   tl1 = th1 =  0xfd;  /* baud rate */
   pcon = 0x80;        /* double baud rate (SMOD) */
   ren = 1;            /* enable receiver */
   es = 1;             /* set UART interrupt enable */

   /* INITIALIZE Tx and Rx Circular Queues */
   txqc = 0;           /* reset Tx Queue data count */
   txqi = 0;           /* reset Tx Queue index */
   rxqc = 0;           /* reset Rx Queue data count */
   rxqi = 0;           /* reset Rx Queue index */

   /* Remember to set GLOBAL interrupt enable (i.e. ea = 1;) in main() */
   }


/* Serial Port Interrupt Service Routine (ISR)

*/
static
sci_isr() interrupt 4 using 1
   {
   if (ri)     /* if Serial Port Rx Ready */
      {
      ri = 0;         /* Clear Rx request */

      if (rxqc<RXQLEN) /* if Rx Queue NOT full */
         {
         rxq[(rxqi+rxqc)%RXQLEN] = sbuf;   /* put data at tail of Rx Queue */
         ++rxqc;                           /* inc Rx Queue count */
         }
      else
          {
          /* IGNORE OVERFLOW Rx CHARACTERS ??? */
          }
      }

   if (ti)     /* if Serial Port Tx Ready */
      {
      ti = 0;         /* Clear Tx request */

      if (txqc) /* if Tx Queue NOT empty */
         {
         sbuf = txq[txqi];       /* Tx data at front of Tx Queue */

         --txqc;                 /* dec Tx Queue count */
         ++txqi;                 /* inc Tx Queue index */
         if (txqi >= TXQLEN)     /* if Tx index rollover */
            txqi = 0;                  /* reset index */
         }
      }
   }


/* Write Character to Serial Port Tx Queue

*/

putchar(data)
   char data;  /* character to be transmitted */
   {
   if (data == '\n') /* if char is newline */
      {
      putchar('\r'); /* write a <CR> */
      }

   ea = 0; /* disable interrupts while working on queues */

   if (txqc<RXQLEN) /* if Tx Queue NOT full */
      {
      if (!txqc)       /* if Tx Queue empty */
         ti = 1;          /* make transmitter ready */

      txq[(txqi+txqc)%TXQLEN] = data;   /* put data at tail of Tx Queue */
      ++txqc;                 /* inc Tx Queue count */
      }
   else
       {
       data = -1; /* IGNORE OVERFLOW Tx CHARACTERS ??? */
       }

   ea = 1; /* enable interrupts */

   return(data);     /* return transmit character */
   }


/* Read Character from Serial Port Rx Queue

*/

getch()
   {
   char data;

   ea = 0; /* disable interrupts while working on queues */

   if (rxqc) /* if Rx Queue NOT empty */
      {
      data = rxq[rxqi]; /* get data from front of Rx Queue */

      --rxqc;                 /* dec Rx Queue count */
      ++rxqi;                 /* inc Rx Queue index */
      if (rxqi >= RXQLEN)     /* if Rx index rollover */
         rxqi = 0;                  /* reset index */
      }
   else
       {
       data = -1; /* Rx Queue EMPTY */
       }

   ea = 1; /* enable interrupts */

   return(data);
   }


/* Read Character from Serial Port w/Echo

*/

getchar()
   {
   return(putchar(getch()));
   }


/* Read Character from Serial Port w/o Echo

Note: Added for compatibility with standard MICRO/C-51 Library

*/

_getkey()
   {
   return(getch());
   }


/* Read Serial Port Rx Queue Status

*/

getrxqs()
   {
   return(rxqc); /* Rx Queue EMPTY = 0 */
   }

