/************************************************************************
;									*
;			1988 - Jim Schimpf				*
;									*
;			PC SERIAL DEVICE CODE				*
;									*
;	Module:sercom.c							*
;									*
;     Version	Date	    Person  	Description			*
;-----------------------------------------------------------------------*
;	0.1	 7-Jun-88 J.Schimpf	Initial Version			*
;	0.2	 8-Jun-88 J.Schimpf	Added routines			*
;	0.3	 9-Jun-88 J.Schimpf	Change to handle COM2		*
;	0.4	13-Jun-88 J.Schimpf	Set Vector for COM2		*
;					& Save vector value		*
;	0.5	15-Jun-88 J.Schimpf	Change routine names to be	*
;					like MAC routines		*
;	0.6	16-Jun-88 J.Schimpf	Make sure CX is saved in INT	*
;	0.7	16-Jun-88 J.Schimpf	Save & restore original int	*
;					vectors				*
;	0.8	23-Jun-88 J.Schimpf	Turn on DTR,RTS and OUT2	*
;	0.9	23-Jun-88 J.Schimpf	Change so buffer can be any size*
;					not just a power of 2		*
;	1.0	 7-Apr-89 J.Schimpf	Preserve DS through int set	*
;	1.1	31-May-89 J.Schimpf	Add Flush input buffer call	*
;	1.2	26-Jun-89 J.Schimpf	Add code to clear data before	*
;					setup				*
;	1.3	29-Mar-90 J.Schimpf	Add calls to set/reset RTS & DTR*
;	1.4	 6-Jun-90 J.Schimpf	Make output interrupt driven	*
;	1.5	19-Jan-91 J.Schimpf	Convert to QC2.5 C version	*
;       1.6     22-Jan-91 J.Schimpf     Make PUTSER wait if buffer full *
;	1.7	13-Feb-91 J.Schimpf	Changes to handle buffer full	*
;	1.8	29-May-91 J.Schimpf	Changed to handle two ports at	*
;					once				*
;	1.9	11-Jun-91 J.Schimpf	Fix putsers to wait for free buf*
;					on output			*
;					Also put two STOP bits on this	*
;					to keep from overrunning rec	*
;	2.0	13-Jun-91 J.Schimpf	Keep count of <CR>'s seen	*
;									*
;	DESCRIPTION:							*
;		This module impliments a full duplex interrupt driven	*
;	serial driver for COM1 or COM2 on an IBM PC type machine	*
;	The code is based on DUMBTERM by CJ Dunford, a simple terminal	*
;	emulator (12-Sep-83)						*
;									*
;	Procedures:							*
;									*
;  Written+	Name		Description 				*
;-----------------------------------------------------------------------*
;<--------------OPEN/CLOSE/CNTL ROUTINES------------------------------->*
;	y	ser_init()	Init serial port and baud rate		*
;	y	close_ser()	Close serial port			*
;<--------------STRING I/O FUNCTIONS----------------------------------->*
;	y	getsers()	Get string from serial data port	*
;	y	putsers()	Send string to serial data port		*
;	y	serline()	Check for line input			*
;<--------------SINGLE CHAR I/O FUNCTIONS------------------------------>*
;	y	getser()	Get 1 serial character			*
;	y	putser()	Send 1 serial character			*
;	y	serhit()	Check for serial input			*
;	y	cntl_dtr()	Control DTR				*
;	y	cntl_rts()	Control RTS				*
;	y	clean_ser()	Clean serial data port			*
;===============INTERNAL SUPPORT ROUTINES===============================*
;<--------------OPEN/CLOSE/CNTL ROUTINES (LOW LEVEL)------------------->*
;	y	ser_init0()	Set up serial port for interrupt I/O	*
;	y	ser_close0()	Close serial port			*
;	y	ser_flush()	Flush input buffer			*
;<--------------INTERRUPT CODE( SPECIAL NON-C CALLS )------------------>*
;	y	int_ser1()	Serial int rtn for COM1			*
;	y	int_ser2()	Serial int rtn for COM2			*
;	y	int_ser()	Handle input serial interrupts		*
;									*
;***********************************************************************/

#include <stdio.h>
#include <dos.h>
#include <bios.h>
#include <malloc.h>
#include "base.h"
#include "sercom.h"

/* USEFUL CONSTANTS */

#define CR	0x0d

/************ SERIAL DEVICE CONSTANTS **********/

#define	COM1_BASE		0x03F8
#define	COM1_IRQ_MASK		0xEF /*11101111B IRQ 4 For COM1 */
#define	COM1_IRQ_EOI		0x64		/* IRQ 4 Spec EOI */
#define	COM1_INT_NUM		0x0C		/* Int # for IRQ4 */

#define	COM2_BASE		0x02F8
#define	COM2_IRQ_MASK		0xF7 /*11110111B IRQ 3 For COM2 */
#define COM2_IRQ_EOI            0x63            /* IRQ 3 Spec EOI */
#define COM2_INT_NUM            0x0B            /* Int # for IRQ3 */

/*  8250 ACE register defs   */

#define	THR			0	/* Offset to Xmit hld reg (write) */
#define	RBR			0	/* Receiver holding buffer (read) */
#define	IER			1	/* Interrupt enable register */
#define	IIR			2	/* Interrupt identification reg */
#define	LCR			3	/* Line control register */
#define	MCR			4	/* Modem control register */
#define	LSR			5	/* Line status register */
#define	MSR			6	/* Modem status register */

/*  Write a little MACRO to calculate a register port # from the base and
    and value above
*/

#define SREG( x )       ((unsigned)((unsigned)x + c->com_base))

/*  8259 Int controller registers */

#define INTC_MASK		0x21	/* Interrupt controller MASK reg */
#define INTC_EOI		0x20	/* Interrupt controller EOI reg */

/*	**************  DATA AREA  *************		*/

#define MAX_PORTS	2	/* # I/O ports (DOS limit) */
static int count = 0;
static COM com_list[MAX_PORTS];	/* I/O data structure */

static COM *com1;		/* Pointers for interrupt actions */
static COM *com2;
static COM *com_xfer;           /* Transfer interrupt data structure */

/*	************* SUPPORT ROUTINES *************	*/

COM *ser_init0(int port,char *ibuf,int isize, char *obuf,int osize);
void ser_close0( COM *c );

/*	************* Interrupt routines *************	*/

void (_interrupt _far int_ser1)( void );  /* Int rtn for serial I/O COM 1 */
void (_interrupt _far int_ser2)( void );  /* Int rtn for serial I/O COM 2 */
void (_interrupt _far int_ser_sup)( void );   /* Support int actions */

/************************************************************************
*  LEVEL 1 ROUTINES - Setup & manage rouitines
************************************************************************/
/***********************************************************************
*
*  COM *SER_INIT( port,baud,bit,parity,isize,osize ) 
*
*       INPUT:  port	- Port to be set
*		baud    - Set to this baud rate
*		bit	- # Bits in word
*		parity	- Even/None/Odd parity (1/0/-1)
*		isize	- Size of input buffer
*		osize	- Size of output buffer
*
*       OUTPUT: COM structure or NULL if failure
*
*************************************************************************/

COM *ser_init( int port,int baud,int bit,int parity,int isize,int osize )
{
	unsigned status;
	char ch;
	COM *c;
	char *in_buf,*out_buf;
	
	/*  1) Set the baud rate of port and:
		PARITY = NONE
		BITS   = 8
		STOPS  = 1
	*/

	status = _bios_serialcom(_COM_INIT,port,
				(bit | parity | _COM_STOP2
					| baud ));
				
	/*  2 Create the buffer we will use, if not return FALSE */

	in_buf = malloc( isize );
	if( in_buf == NULL )
	{
		return( NULL );
	}

	out_buf = malloc( osize );
	if( out_buf == NULL )
	{
		return( NULL );
	}

	/*  3) Init the port with this information */
				
	c = ser_init0(port,in_buf,isize,out_buf,osize );
	
	/*  4) Clear out the buffer */

	clean_ser(c);
	
	return( c );
}

/***********************************************************************
*
*  SER_CLOSE(COM *c)	- Close down serial port
*
*	INPUT:	NONE
*
*	OUTPUT:	NONE, Hardware shut down and buffer closed
*
***********************************************************************/

void ser_close(COM *c)
{
	int i;
	
	/*  (0) Check is hardware is running */

	if( !c->ready )
		return;
	
	/*  1) Close down the hardware */

	ser_close0(c);
	
	/*  2) Free up the buffers */

	FREE( c->in_buf );
	FREE( c->out_buf );
	
}

/************************************************************************
*  LEVEL 1 ROUTINES - String input and output
************************************************************************/
/*---------------------------------------------------------------------
;
;  int SERLINE(COM *c)	- Return TRUE or FALSE
;
;	INPUT:	NONE
;
;	OUTPUT:	TRUE if LINE in buffer
;		FALSE if not
;
;---------------------------------------------------------------------*/

BOOLEAN serline( COM *c )
{
	/*  (0) If we weren't ready then skip setup as never done */
	
	if( !c->ready )
		return(FALSE);

	/* Check for the count of CR's in buffer */
	
	if( c->in_crcnt > 0 )
		return( TRUE );
	else
		return( FALSE );
}

/***********************************************************************
*
*  GETSERS( COM *c,len,str )      - Read string from serial port
*
*       INPUT:  c	- Serial data structure
*		str     - String read into here
*               len     - Max length that will fit in string
*
*       OUTPUT: Return # characters seen
*               NOTE: Returned string does NOT have the terminating
*                       \n
*		Return -1 if abort
*
*************************************************************************/

int getsers( COM *c,int len,char *str )
{
	char ch;
	int i;
	
	/*  Get a character, wait till there is something then get it */
	
	i = 0;
	while( i<len )
	{
		while( !serhit(c) )
		{
			/*  CHECK FOR USER KEY, QUIT IF SEEN */
			if( kbhit() )
				return( -1 );
		}
		
		/*  OK we have a character, get it */
		
		ch = 0x7f & getser(c);
		switch( ch ) {
			
		case 0x0d:	str[i++] = '\0';
				return( i );
				
		case 0x00:
		case 0x0a:	break;
				
		default:	str[i++] = ch;
				break;
			}
	}
	
	/*  Very long strings fall out here */
	
	str[i] = '\0';
	return( len );
}

/**************************************************************************
*
*  PUTSERS( char *str, COM *c )	- Output a string to the serial port
*
*	INPUT:	str	- Output this string
*		c	- Serial data structure
*
*	OUTPUT:	 # Characters sent
*
**************************************************************************/
	
int putsers( char *str, COM *c )
{
	int n,i,j;
	
	/* 1) Get the length */
	
	n = strlen( str );
	
	/*  2) Output the string */
	
	for( i=0; i<n; i++ )
	{
		while( !putser( str[i],c ) )
			;
	}
	
	return( n );
}
/************************************************************************
*  LEVEL 0 ROUTINES - Single Character I/O & Control Lines
************************************************************************/
/*-------------------------------------------------------------------
;
;  int PUTSER( char outch, COM *c )	- Output a character to serial out
;
;	INPUT:	outch	- Character to output
;
;	OUTPUT:	TRUE if character output
;		FALSE on buffer full
;		Character sent to serial output (CTS & DTR not checked)
;
;------------------------------------------------------------------*/

BOOLEAN putser( char outch, COM *c )
{
	char val;

	/*  (0) If we weren't ready then skip setup as never done */
	
	if( !c->ready )
		return(FALSE);
	
	while( !c->out_mt && (c->out_head == c->out_tail) )
		;

	/* (1) Store the character in buffer (if we can)
		If full, then quit now
	*/
	
	if( !c->out_full )
	{
		c->out_buf[c->out_head++] = outch;
		if( c->out_head == c->out_size )
			c->out_head = 0;/* Reset buffer circularly */
	}
	
	/*  (2) Check for buffer full, if so then return FALSE and mark
		full flag
	*/
	
	if( c->out_head == c->out_tail )
	{
		c->out_full = TRUE;
		return( FALSE );
	}
	else
		c->out_full = FALSE;
	
	/* (3) Now check to see if the output interrupt is on, if not
		then turn it on now
	*/

	val = inp( SREG(LCR) );	/* Reset DLAB for IER access */
	val &= 0x7F;		/* Clear IER access bit */
	outp(SREG(LCR),val);

	val = inp( SREG(IER) );
	if( !(val & 0x02) )		/* Interrupt ON ? */
	{

		c->out_mt = FALSE;	/* Not MT now */
		_disable();		/* Interrupts OFF NOW */
		outp(SREG(IER),0x03);	/* RX & TX interrupts ON */
		_enable();		/* Interrupts ON again */
	}
		
	/* (6) In all these cases return TRUE */

		return( TRUE );
}
		
/*---------------------------------------------------------------------
;
;  int SERHIT(COM *c)	- Return TRUE or FALSE
;
;	INPUT:	NONE
;
;	OUTPUT:	TRUE if stuff in buffer
;		FALSE if not
;
;---------------------------------------------------------------------*/

BOOLEAN serhit( COM *c )
{
	/*  (0) If we weren't ready then skip setup as never done */
	
	if( !c->ready )
		return(FALSE);

	/*  Buffer has NO data if and only if HEAD == TAIL
	    and flag ! TRUE
	*/
	
	if( !c->in_mt )
		return( TRUE );
	else
		return( FALSE );
}


/*---------------------------------------------------------------------
;
;  int GETSER(COM *c)	- Get a character from the input buffer
;
;	INPUT:	NONE
;
;	OUTPUT:	Return last character from buffer
;	NOTE: Assumes SERHIT() has been run before
;			this routine to assure that there is a
;			character present
;
;--------------------------------------------------------------------*/

int getser( COM *c )
{
	int ch;
	
	/*  (1) If we weren't ready then skip setup as never done */
	
	if( !c->ready )
		return(FALSE);
	
	/*  (1a) Also if no data then just return NUL */
	
	if( !serhit(c) )
		return( 0 );
	
	/* 2) Read the buffer, pull the next character from the
		buffer tail and advance it circularly
	*/

	_disable();
	
	ch = 0xff & c->in_buf[c->in_tail++];
	if( c->in_tail == c->in_size )
		c->in_tail = 0;
	
	if( c->in_tail == c->in_head )
		c->in_mt = TRUE;
	
	if( ch == CR )			/* Keep track of CR's */
		c->in_crcnt--;
	
	_enable();

	/* (3) Return the character */

	return( ch );
}

/*-------------------------------------------------------------------
;
;  void CLEAN_SER( COM *c )	- Flush input buffer
;
;	INPUT:	NONE
;
;	OUTPUT:	Turn off the interrupt and flush input buffer
;
;------------------------------------------------------------------*/

void clean_ser( COM *c )
{

	/* (1) Turn off INTERRUPTS and reset input buffer back to IC */
	
	_disable();

	c->in_head = 0;
	c->in_tail = 0;
	c->in_mt = TRUE;
	c->in_crcnt = 0;
	
	/* (2) Re-enable interrupts and return */
	
	_enable();
}

/*-------------------------------------------------------------------
;
;  void CNTL_DTR( int flag, COM *c )	- Set/Clear DTR
;
;	INPUT:	flag	- Integer TRUE to set DTR, FALSE to clear
;
;	OUTPUT:	NONE
;		DTR set or cleared
;
;------------------------------------------------------------------*/

void cntl_dtr( int flag,COM *c )
{
	char val;
	
	/*  (1) If not set up then quit now */
	
	if( !c->ready )
		return;
	
	/*  (2) Read the current value of the DTR register */
	
	val = inp(SREG(MCR));
	
	/*  (3) If TRUE then set it otherwise CLEAR it */
	
	if( flag )
		val |= 1;
	else
		val &= ~1;
	
	/*  (4) Change the register and return */
	
	outp(SREG(MCR),val);
}

/*-------------------------------------------------------------------
;
;  void CNTL_RTS( int flag, COM *c )	- Set/Clear RTS
;
;	INPUT:	flag	- Integer TRUE to set RTS, FALSE to clear
;
;	OUTPUT:	NONE
;		RTS set or cleared
;
;------------------------------------------------------------------*/

void cntl_rts( int flag, COM *c )
{
	char val;
	
	/*  (1) If not set up then quit now */
	
	if( !c->ready )
		return;
	
	/*  (2) Read the current value of the RTS register */
	
	val = inp(SREG(MCR));
	
	/*  (3) If TRUE then set it otherwise CLEAR it */
	
	if( flag )
		val |= 2;
	else
		val &= ~2;
	
	/*  (4) Change the register and return */
	
	outp(SREG(MCR),val);
}

/*---------------------------------------------------------------------
;	***	LEVEL 0 - SETUP/CLOSE  FUNCTIONS
;----------------------------------------------------------------------*/
/*-----------------------------------------------------------------------
;
;  void SER_INIT0( int port,char *ibuf,int isize,char *obuf,int osize )
;	int port;		Arg 1
;	char *ibuf;		Arg 2
;	int isize;		Arg 3
;	char *obuf;		Arg 4
;	int osize;		Arg 5
;
;	INPUT:	port	- 0 = COM 1
;			- 1 = COM 2
;		ibuf	- Pointer to input buffer
;		isize	- Size of input data buffer
;		obuf	- Pointer to output buffer
;		osize	- Size of output data buffer
;
;	OUTPUT:	Pointer to COM struct for port
;		NULL if failure
;		NOTE: Baud rate & other parameters set by
;		      _bios_serialcom() - See Page 143 of Library manual
;
;-----------------------------------------------------------------------*/

COM *ser_init0(int port,char *ibuf,int isize, char *obuf,int osize)
{
	int i;
	char val;
	COM *c;
	
	/* (0) First select using the port # select the correct struct
		& mark as not ready
	*/
	
	while( port >= MAX_PORTS )	/* Get port # in range */
		port--;
	for( i=0; i<MAX_PORTS; i++ )	/* Select data structure */
	{
		if( !com_list[i].ready )
		{
			c = &(com_list[i]);
			break;
		}
	}
	if( i == MAX_PORTS )		/* Not found */
		return( NULL );

	/*  (1) Get buffers and initialize pointers and flags */
	
	c->in_buf	= ibuf;
	c->in_size	= isize;
	c->in_mt	= TRUE;
	c->in_head	= 0;
	c->in_tail	= 0;
	c->in_crcnt	= 0;

	c->out_buf	= obuf;
	c->out_size	= osize;
	c->out_full    = FALSE;
	c->out_mt	= TRUE;
	c->out_head	= 0;
	c->out_tail	= 0;
	
	/*  (2) Now set up the other stuff in the structure by knowing which
		port we are about to use
	*/
	
	switch( port ) {

		case 0:		/* Here set up for COM1 */
			c->ready = TRUE;
			c->com_base	= COM1_BASE;
			c->irq_mask	= COM1_IRQ_MASK;
			c->irq_eoi	= COM1_IRQ_EOI;
			c->int_number  = COM1_INT_NUM;
			
	/*  (3) Set the interrupt vector for COM 1 */
			
			_disable();
	
	/*  (3a) Read old interrupt and set new one */
	
			com1 = c;
			c->old	= _dos_getvect( c->int_number );
			_dos_setvect(c->int_number,int_ser1);
			break;

		case 1:		/* Here set up for COM1 */
			c->ready = TRUE;
			c->com_base	= COM2_BASE;
			c->irq_mask	= COM2_IRQ_MASK;
			c->irq_eoi	= COM2_IRQ_EOI;
			c->int_number  = COM2_INT_NUM;
	/*  (3) Set the interrupt vector for COM 1 */
			
			_disable();
	
	/*  (3a) Read old interrupt and set new one */

			com2 = c;
			c->old	= _dos_getvect( c->int_number );
			_dos_setvect(c->int_number,int_ser2);
			break;
			
		default:	return(NULL);	/* Bad port SKIP */
	}
	
	/* (3b) Enable IRQ4/COM1 IRQ3/COM2 on the 8259 interrupt controller */
		
	val = inp( INTC_MASK );
	val &= c->irq_mask;
	outp( INTC_MASK, val );

	/* (4) 8250 HARWARE setup */
		
	val = inp( SREG(LSR) );		/* Read and discard STATUS */
	val = inp( SREG(RBR) );		/* Read and discard DATA */
		
	val = inp( SREG(LCR) );		/* Rst DLAB for IER access */
	val &= 0x7F;			/* 01111111B */
	outp( SREG(LCR),val );
		
	outp( SREG(IER),1);		/* Enable Data READY INT */
		
	outp( SREG(MCR),0xB );		/* Enable OUT2,RTS & DTR */
		
	/* (5) All done, restore interrupts to processor */
		
	_enable();
	
	return( c );
	
}

/*-------------------------------------------------------------------
;
;  void SER_CLOSE0( COM *c )	- Close the open serial channel
;
;	INPUT:	NONE
;
;	OUTPUT:	Turn off the interrupt and shutdown the channel
;
;------------------------------------------------------------------*/

void ser_close0( COM *c )
{
	char val;
	
	/*  (1) If we weren't ready then skip setup as never done */
	
	if( !c->ready )
		return;
	
	/*  (2) Turn off the interrupt here & disable all the stuff we enabled
	-------------- INTERUPTS OFF HERE -------------------------------*/
	
	_disable();

	/*  (3) Disable IRQ4 on the 8259 */
	
	val = inp(INTC_MASK);	
	val |= ~c->irq_mask;
	outp(INTC_MASK,val);

	/* (4) Disable 8250 dtr interrupt */
	
	val = inp( SREG(LCR) );	/* Reset DLAB for IER access */
	val &= 0x7F;		/* Clear IER access bit */
	outp(SREG(LCR),val);
	
	val = inp( SREG(RBR) );
	val = inp( SREG(LSR));
	val = inp(SREG(IIR) );
	val = inp(SREG(IER) );
	outp(SREG(IER),0);	/* Disable 8250 Interrupts */
	
	outp(SREG(MCR),0);	/* Disable RTS,DTR and OUT2 */
	
	outp(SREG(MCR),0);	/* Disable OUT2 */
	
	/* (5) Restore original interrupt vector */
	
	_dos_setvect(c->int_number, c->old );
	
/*--------------- INTERRUPTS BACK ON --------------------------------*/
	
	_enable();
	
	c->ready = FALSE;
	
}

/*---------------------------------------------------------------------
;	I N T E R R U P T   C O D E - L o c a l  R o u t i n e s
;----------------------------------------------------------------------*/
/*----------------------------------------------------------------------
;
;  INT_SER1	- Handle a serial interrupt INPUT/OUTPUT COM 1
;
;	INPUT:	Interrupt from serial port COM 1
;
;	OUTPUT:	
;		Store input character in buffer & reset full flag
;		If buffer full, mark full flag & drop character
;
;-----------------------------------------------------------------------*/

void _interrupt _far int_ser1( void )
{
	/* Put the value into TRANSFER & do it */

	com_xfer = com1;
	_chain_intr( int_ser_sup );
}
	
/*----------------------------------------------------------------------
;
;  INT_SER2	- Handle a serial interrupt INPUT/OUTPUT COM 2
;
;	INPUT:	Interrupt from serial port 2
;
;	OUTPUT:	
;		Store input character in buffer & reset full flag
;		If buffer full, mark full flag & drop character
;
;-----------------------------------------------------------------------*/

void _interrupt _far int_ser2( void )
{
	/* Put the value into TRANSFER & do it */

	com_xfer = com2;
	_chain_intr( int_ser_sup );
}

/*----------------------------------------------------------------------
;
;  INT_SER_SUP	- Handle a serial interrupt INPUT/OUTPUT (Handler)
;
;	INPUT:	c	- Pointer to data structure for port
;		Interrupt from serial port
;
;	OUTPUT:	
;		Store input character in buffer & reset full flag
;		If buffer full, mark full flag & drop character
;
;-----------------------------------------------------------------------*/

void _interrupt _far int_ser_sup( void )
{
	char val;
	char ch;
	int ptr;
	COM *c;
	
	/*  (0) Extract vector and begin work */

	c = com_xfer;

	/*  (1) First read the status register and try to determine
		what kind of interrupt we have
	*/
	
	while( TRUE )
	{
		val = inp( SREG(LSR) );	/* Read and discard STATUS */
		val = inp( SREG(IIR) );	/* Get interrupt status register */
		
		/*  (2) Depending on the type of action do it
			a. Transmit interrupt
			b. Receive interrupt
			c. No interrupt, just return
		*/

		if( val & 0x04 )		/* Receive Interrupt */
		{
		/*  (1) Get the character from 8250 */

			ptr = c->in_head;
			ch = inp( SREG(RBR) );
			
		/*  (2) Now if we are not full then store this in the 
			buffer
		*/
			if( c->in_mt || ptr != c->in_tail )
			{
				c->in_buf[ptr++] = ch;
				if( ptr == c->in_size )
					ptr = 0;
				c->in_head = ptr;
				c->in_mt = FALSE;
				
				if( ch == CR )		/* Count lines */
					c->in_crcnt++;
			}
		}
		else
		{
			if( val & 0x02 )        /* Transmit Interrupt */
			{
				/* 1) Read the buffer, pull the next 
				character from the buffer tail If MT 
				then just quit now & turn off int (mark flag)
				There are two cases to check
					If FULL flag set then just do it
					If NOT full check if tail == head, 
						if so MT
				*/
	
				if( (!c->out_full) &&
					(c->out_head == c->out_tail) )
				{

				/*  2) Here buffer is MT, mark it so
				and then turn off the interrupt
				*/
					c->out_mt = TRUE;
					val = inp( SREG(LCR) );
					val &= 0x7F;
					outp(SREG(LCR),val);

					outp(SREG(IER),0x01);
					/* RX  interrupts ON */
				}
				else
				{
		
					/*  (2) Ok, we have to send out
					the data via the 8250 */

					outp(SREG(THR),
					c->out_buf[c->out_tail++]);
					if( c->out_tail == c->out_size )
						c->out_tail = 0;
				}
			}
			 else
				return;         /* No Interrupt */
		}

		/* (3) Clear the interrupt controller and
			try again */
		
		outp(INTC_EOI,c->irq_eoi);
	}
}
