//                                    Copyright John V. Rodley   1994
//
// LX_EXE.C - Module to deal with reading and dumping the contents of
// an LX executable.
//
// Theory of operation:
//
// The exe is made up of three interesting parts:
//    the header
//    the tables
//    the physical pages
//
// The header is read in one shot and printed out by the DumpLXExeHeader
// function
//
// The tables are read by a bunch of Load...Table functions into global
// arrays of structures (described in lx_exe.h).
// All the table entry types are described by a struct somewhere in 
// lx_exe.h.  If a table entry can have multiple "shapes" the 
// struct is a superset of all the possible shapes.  The Load... 
// functions read the tables and deconstruct those entries into the more
// abstract structs contained in the global arrays.  The Dump... 
// functions run through the global arrays printing out the members and
// incorporate most of the intelligence about the meaning of various
// structure members.
//
// The physical pages are read by the LoadPages() function into a 
// global array and printed out in hex and ascii by the DumpLXPages() 
// function.
//
// The Load... functions merely read the individual tables into 
// the global arrays.
// The Dump... functions contain any logic about linking the data
// in tables together (like pinning fixup records to imported functions).


#include <stddef.h>     /* Multi-threads live in mt directory.    */
#include <ctype.h>
#include <malloc.h>
#include <share.h>
#include <stdio.h>
#include <errno.h>
#include <io.h>
#include <fcntl.h>
#include <stdlib.h>
#include <sys\types.h>
#include <sys\stat.h>
#include <string.h>
#include "types.h"
#include "exe.h"
#include "utils.h"


USHORT magic;  // the magic bytes "LX"

// Possible string values for the flags member of the Object Page Table
// entry corresponding 1-1 to int value.  0 = "Legal ...", 1 = "Iterated" ...
char *pgflags[] = {
   "Legal Physical Page",
   "Iterated Data Page",
   "Invalid Page",
   "Zero Filled Page",
   "Range of Pages"
};

// Possible string values for the flags member of the Object Table
// entry corresponding 1-1 to int value.  0 = "READ", 1 = "WRITE" ...
char *obj_flags[] = {
   "READ",
   "WRITE",
   "EXECUTE",
   "RESOURCE",
   "DISCARDABLE",
   "SHARED",
   "PRELOAD",
   "INVALID",
   "ZEROFILLED",
   "RESIDENT",
   "RESLONGLOCK",
   "RESERVED1",
   "16-BIT",
   "32-BIT",
   "CONFORMING",
   "IOPRIV"
};

// String values for resource types from type_id member of resource
// Table entry.  Corresponds 1-1 to type_id, 1 = "POINTER" ...
char *resource_types[] = {
   "BAD TYPE 0",
   "POINTER",
   "BITMAP",
   "MENU",
   "DIALOG",
   "STRING",
   "FONTDIR",
   "FONT",
   "ACCELTABLE",
   "RCDATA",
   "MESSAGE",
   "DLGINCLUDE",
   "VKEYTBL",
   "CHARTBL",
   "DISPLAYINFO",
   "FKASHORT",
   "FKALONG",
   "HELPTABLE",
   "HELPSUBTABLE",
   "FDDIR",
   "FD"
};

LX_EXE lx_exe;   // The global LX exe header.

// This is the count of pages specified by the objects in the 
// exe. Set by LoadObjectTable from the page counts contained in 
// the Object Table entries.  Used by LoadObjectPage, LoadFixupPage ...
USHORT ObjPgCnt = 0;

         // The Object Table
#define MAX_LX_OBJECTS 1000
LX_OBJ *ObjList[MAX_LX_OBJECTS];

         // The Object Page Table
#define MAX_LX_PGS 1000
LX_PG *ObjPgList[MAX_LX_PGS];

         // The Fixup Page Table
ULONG FixupPgList[MAX_LX_PGS];    // per-page fixups are stored here.

         // The list of actual physical pages.
char *ObjPgs[MAX_LX_PGS];     

         // The Non-Resident Name Table
#define MAX_LX_NRESNAMES 1000
LX_NRES *NResNmList[MAX_LX_NRESNAMES];

         // The Resident Name Table
#define MAX_LX_RESNAMES 1000
LX_NRES *ResNmList[MAX_LX_RESNAMES];

         // The Imported Module Table
#define MAX_LX_IMPMODS 1000
LX_NRES *ImpModList[MAX_LX_IMPMODS];

         // The Imported Procedure Table
#define MAX_LX_IMPPROCS 1000
LX_NRES *ImpProcList[MAX_LX_IMPPROCS];

         // The Fixup Record Table
#define MAX_LX_FIXUPS 2000
LX_FIXUP *FixupList[MAX_LX_FIXUPS];

      // The Resource Table
#define MAX_LX_RESOURCES 1000
LX_RSC *RscList[MAX_LX_RESOURCES];
// When we go get the resource, keep a pointer to it here
char *LoadedRscList[MAX_LX_RESOURCES];

// All the intelligence about the structures of the exe tables is
// located in the Load... functions below.
static void LoadFixupPageTable(void);
static void LoadFixupRecordTable(void);
static void LoadImportedModuleTable(void);
static void LoadImportedProcedureTable(void);
static void LoadObjectTable(void);
static void LoadObjectPageTable(void);
static void LoadObjectPageTable(void);
static void LoadResidentNameTable(void);
static void LoadNonResidentNameTable(void);
static void LoadResourceTable(void);

// FreeLXTables - Clear all the global LX lists, freeing all components
// and nulling the array members.
void FreeLXTables()
{
int i;

ClearList((char **)&LoadedRscList[0], sizeof( char *), MAX_LX_RESOURCES );
ClearList((char **)&RscList[0], sizeof( LX_RSC *), MAX_LX_RESOURCES );
ClearList((char **)&ImpProcList[0], sizeof( LX_NRES *), MAX_LX_IMPPROCS );
ClearList((char **)&ImpModList[0], sizeof( LX_NRES *), MAX_LX_IMPMODS );
ClearList((char **)&NResNmList[0], sizeof( LX_NRES *), MAX_LX_NRESNAMES );
ClearList((char **)&ResNmList[0], sizeof( LX_NRES *), MAX_LX_RESNAMES );
ClearList((char **)&ObjList[0], sizeof( LX_RSC *), MAX_LX_OBJECTS );
ClearList((char **)&ObjPgList[0], sizeof( LX_PG *), MAX_LX_PGS );

for( i = 0; FixupList[i] != NULL && i < MAX_LX_FIXUPS; i++ )
   if( FixupList[i]->list )
      free( FixupList[i]->list );
   
ClearList((char **)&FixupList[0], sizeof( LX_FIXUP *), MAX_LX_FIXUPS );
}

// DumpLXExeHdr - Dump all the information contained in the LX exe header
// not including the tables.
void DumpLXExeHdr()
{
char *ptr;

   ptr = (char *)&magic;
	printf( "%40.40s - %c%c (0x%x)\n", "magic", ptr[0], ptr[1], magic );
	printf( "%40.40s - %s %d (0x%x)\n", "Byte Order", 
			(lx_exe.ByteOrder&0x00FF)&BIG_ENDIAN?"BIG ENDIAN":"LITTLE ENDIAN", 
				(lx_exe.ByteOrder&0x00FF), (lx_exe.ByteOrder&0x00ff) );
	printf( "%40.40s - %s %d (0x%x)\n", "Word Order", 
		(lx_exe.WordOrder&0x00FF)&BIG_ENDIAN?"BIG ENDIAN":"LITTLE ENDIAN",
			(lx_exe.WordOrder&0x00FF), (lx_exe.WordOrder&0x00FF));
	printf( "%40.40s - %d (0x%x)\n", "Exe Format Level", lx_exe.FormatLevel, lx_exe.FormatLevel );
	switch( lx_exe.CpuType )
		{
		case PROC_286_UP:	ptr = "80286+"; break;
		case PROC_386_UP:	ptr = "80386+"; break;
		case PROC_486_UP:	ptr = "80486+"; break;
		case PROC_PENTIUM_UP:	ptr = "PENTIUM+"; break;
		default:	ptr = "OTHER"; break;
		}
	printf( "%40.40s - %s %d (0x%x)\n", "Cpu Type", ptr, lx_exe.CpuType, lx_exe.CpuType );
	switch( lx_exe.OSType )
		{
		case OST_UNKNOWN:	ptr = "Unknown"; break;
		case OST_OS2:	ptr = "OS/2"; break;
		case OST_WINREF:	ptr = "Windows Ref"; break;
		case OST_DOS4:	ptr = "DOS 4.X"; break;
		case OST_WIN386REF:	ptr = "Win386 Ref"; break;
		default:	ptr = "OTHER"; break;
		}
	printf( "%40.40s - %s %d (0x%x)\n", "OSType", ptr, lx_exe.OSType, lx_exe.OSType );
	printf( "%40.40s - %lu (0x%lx)\n", "ModuleVersion", lx_exe.ModVersion, lx_exe.ModVersion );
	printf( "%40.40s - %lu (0x%lx)\n", "ModFlags", lx_exe.ModFlags, lx_exe.ModFlags );
	printf( "%40.40s - %lu (0x%lx)\n", "ModNumPgs", lx_exe.ModNumPgs, lx_exe.ModNumPgs );
	printf( "%40.40s - %lu (0x%lx)\n", "EIPObjNum", lx_exe.EIPObjNum, lx_exe.EIPObjNum );
	printf( "%40.40s - %lu (0x%lx)\n", "EIP", lx_exe.EIP, lx_exe.EIP );
	printf( "%40.40s - %lu (0x%lx)\n", "ESPObjNum", lx_exe.ESPObjNum, lx_exe.ESPObjNum );
	printf( "%40.40s - %lu (0x%lx)\n", "Esp", lx_exe.Esp, lx_exe.Esp );
	printf( "%40.40s - %lu (0x%lx)\n", "PgSize", lx_exe.PgSize, lx_exe.PgSize );
	printf( "%40.40s - %lu (0x%lx)\n", "PgOfsShift", lx_exe.PgOfsShift, lx_exe.PgOfsShift );
	printf( "%40.40s - %lu (0x%lx)\n", "FixupSectionSize", lx_exe.FixupSectionSize, lx_exe.FixupSectionSize );
	printf( "%40.40s - %lu (0x%lx)\n", "FixupCksum", lx_exe.FixupCksum, lx_exe.FixupCksum );
	printf( "%40.40s - %lu (0x%lx)\n", "LdrSecSize", lx_exe.LdrSecSize, lx_exe.LdrSecSize );
	printf( "%40.40s - %lu (0x%lx)\n", "LdrSecCksum", lx_exe.LdrSecCksum, lx_exe.LdrSecCksum );
	printf( "%40.40s - %lu (0x%lx)\n", "ObjTblOfs", lx_exe.ObjTblOfs, lx_exe.ObjTblOfs );
	printf( "%40.40s - %lu (0x%lx)\n", "NumObjects", lx_exe.NumObjects, lx_exe.NumObjects );
	printf( "%40.40s - %lu (0x%lx)\n", "ObjPgTblOfs", lx_exe.ObjPgTblOfs, lx_exe.ObjPgTblOfs );
	printf( "%40.40s - %lu (0x%lx)\n", "ObjIterPgsOfs", lx_exe.ObjIterPgsOfs, lx_exe.ObjIterPgsOfs );
	printf( "%40.40s - %lu (0x%lx)\n", "RscTblOfs", lx_exe.RscTblOfs, lx_exe.RscTblOfs );
	printf( "%40.40s - %lu (0x%lx)\n", "NumRscTblEnt", lx_exe.NumRscTblEnt, lx_exe.NumRscTblEnt );
	printf( "%40.40s - %lu (0x%lx)\n", "ResNameTblOfs", lx_exe.ResNameTblOfs, lx_exe.ResNameTblOfs );
	printf( "%40.40s - %lu (0x%lx)\n", "EntryTblOfs", lx_exe.EntryTblOfs, lx_exe.EntryTblOfs );
	printf( "%40.40s - %lu (0x%lx)\n", "ModDirOfs", lx_exe.ModDirOfs, lx_exe.ModDirOfs );
	printf( "%40.40s - %lu (0x%lx)\n", "NumModDirs", lx_exe.NumModDirs, lx_exe.NumModDirs );
	printf( "%40.40s - %lu (0x%lx)\n", "FixupPgTblOfs", lx_exe.FixupPgTblOfs, lx_exe.FixupPgTblOfs );
	printf( "%40.40s - %lu (0x%lx)\n", "FixupRecTblOfs", lx_exe.FixupRecTblOfs, lx_exe.FixupRecTblOfs );
	printf( "%40.40s - %lu (0x%lx)\n", "ImpModTblOfs", lx_exe.ImpModTblOfs, lx_exe.ImpModTblOfs );
	printf( "%40.40s - %lu (0x%lx)\n", "NumImpModEnt", lx_exe.NumImpModEnt, lx_exe.NumImpModEnt );
	printf( "%40.40s - %lu (0x%lx)\n", "ImpProcTblOfs", lx_exe.ImpProcTblOfs, lx_exe.ImpProcTblOfs );
	printf( "%40.40s - %lu (0x%lx)\n", "PerPgCksumOfs", lx_exe.PerPgCksumOfs, lx_exe.PerPgCksumOfs );
	printf( "%40.40s - %lu (0x%lx)\n", "DataPgOfs", lx_exe.DataPgOfs, lx_exe.DataPgOfs );
	printf( "%40.40s - %lu (0x%lx)\n", "NumPreloadPg", lx_exe.NumPreloadPg, lx_exe.NumPreloadPg );
	printf( "%40.40s - %lu (0x%lx)\n", "NResNameTblOfs", lx_exe.NResNameTblOfs, lx_exe.NResNameTblOfs );
	printf( "%40.40s - %lu (0x%lx)\n", "NResNameTblLen", lx_exe.NResNameTblLen, lx_exe.NResNameTblLen );
	printf( "%40.40s - %lu (0x%lx)\n", "NResNameTblCksum", lx_exe.NResNameTblCksum, lx_exe.NResNameTblCksum );
	printf( "%40.40s - %lu (0x%lx)\n", "AutoDSObj", lx_exe.AutoDSObj, lx_exe.AutoDSObj );
	printf( "%40.40s - %lu (0x%lx)\n", "DebugInfoOfs", lx_exe.DebugInfoOfs, lx_exe.DebugInfoOfs );
	printf( "%40.40s - %lu (0x%lx)\n", "DebugInfoLen", lx_exe.DebugInfoLen, lx_exe.DebugInfoLen );
	printf( "%40.40s - %lu (0x%lx)\n", "NumInstPreload", lx_exe.NumInstPreload, lx_exe.NumInstPreload );
	printf( "%40.40s - %lu (0x%lx)\n", "NumInstDemand", lx_exe.NumInstDemand, lx_exe.NumInstDemand );
	printf( "%40.40s - %lu (0x%lx)\n", "HeapSize", lx_exe.HeapSize, lx_exe.HeapSize );
	printf( "%40.40s - %lu (0x%lx)\n", "StackSize", lx_exe.StackSize, lx_exe.StackSize );

if( lx_exe.ModFlags & MODF_RESERVED1 )
	printf( "Module flag MODF_RESERVED1 set\n" );
if( lx_exe.ModFlags & MODF_RESERVED2 )
	printf( "Module flag MODF_RESERVED2 set\n" );
if( lx_exe.ModFlags & MODF_PERPROCLIB )
	printf( "Module flag MODF_PERPROCLIB set\n" );
if( lx_exe.ModFlags & MODF_RESERVED3 )
	printf( "Module flag MODF_RESERVED3 set\n" );
if( lx_exe.ModFlags & MODF_INTERNALFIX )
	printf( "Module flag MODF_INTERNALFIX set\n" );
if( lx_exe.ModFlags & MODF_EXTERNALFIX )
	printf( "Module flag MODF_EXTERNALFIX set\n" );
if( lx_exe.ModFlags & MODF_RESERVED4 )
	printf( "Module flag MODF_RESERVED4 set\n" );
if( lx_exe.ModFlags & MODF_RESERVED5 )
	printf( "Module flag MODF_RESERVED5 set\n" );
if( lx_exe.ModFlags & MODF_NOWINCOMPAT )
	printf( "Module flag MODF_NOWINCOMPAT set\n" );
if( lx_exe.ModFlags & MODF_WINCOMPAT )
	printf( "Module flag MODF_WINCOMPAT set\n" );
if( lx_exe.ModFlags & MODF_USESWINAPIU )
	printf( "Module flag MODF_USESWINAPIU set\n" );
if( lx_exe.ModFlags & MODF_RESERVED6 )
	printf( "Module flag MODF_RESERVED6 set\n" );
if( lx_exe.ModFlags & MODF_RESERVED7 )
	printf( "Module flag MODF_RESERVED7 set\n" );
if( lx_exe.ModFlags & MODF_RESERVED8 )
	printf( "Module flag MODF_RESERVED8 set\n" );
if( lx_exe.ModFlags & MODF_NOTLOADABLE )
	printf( "Module flag MODF_NOTLOADABLE set\n" );
if( lx_exe.ModFlags & MODF_RESERVED9 )
	printf( "Module flag MODF_RESERVED9 set\n" );
if( lx_exe.ModFlags & MODF_PERPROCLIBTERM )
	printf( "Module flag MODF_PERPROCLIBTERM is set.\n" );
if( lx_exe.ModFlags & MODF_TYPEMASK == MODF_TYPPROGRAM )
	printf( "EXE is a PROGRAM\n" );
if( lx_exe.ModFlags & MODF_TYPEMASK == MODF_TYPLIBRARY )
	printf( "EXE is a LIBRARY\n" );
if( lx_exe.ModFlags & MODF_TYPEMASK == MODF_TYPPROTMEMLIB )
	printf( "EXE is a PROTMEMLIB\n" );
if( lx_exe.ModFlags & MODF_TYPEMASK == MODF_TYPPHYSDEV )
	printf( "EXE is a PHYSDEV\n" );
if( lx_exe.ModFlags & MODF_TYPEMASK == MODF_TYPVIRTDEV )
	printf( "EXE is a VIRTDEV\n" );
}

// DumpLXTables - Print out all the info loaded from the EXE
// and link together any interesting bits of info, like fixups
// and imported names ...
void DumpLXTables()
{
int y, pg;
USHORT i, j, upper_limit, k;
   printf( "\nObject and Object Page Tables\n" );
   for( i = 0; i < (USHORT )lx_exe.NumObjects; i++ )   
      {
      printf( "\tObject %d", i+1 );
      printf( ", Size %lu", ObjList[i]->size );     
      printf( ", Addr %u", ObjList[i]->reloc_base_addr );  
      printf( ", Flags %lx", ObjList[i]->obj_flags ); 
      printf( ", PgTableInd %lu", ObjList[i]->pg_tbl_index ); 
      printf( ", NumPgs %lu", ObjList[i]->num_pg_tbl_entries ); 
      printf( ", Rsv%lx", ObjList[i]->reserved ); 
      printf( "\n\t\t\t" );
      for( j = 0, y = 0x0001; j < 16; j++ )
         {
         if( ObjList[i]->obj_flags & y )
            printf( "%s, ", obj_flags[j] );         
         y *= 2;
         }
      printf( "\n" );
      // Are there any pages attached to this object
      if( ObjList[i]->num_pg_tbl_entries )
         {
         upper_limit = (USHORT )(ObjList[i]->pg_tbl_index-1+ObjList[i]->num_pg_tbl_entries);
         // List those pages
         printf( "\tPAGES:\tNumber\tOffset\t\tSize\tFlags\n" );
         for( k = (USHORT )(ObjList[i]->pg_tbl_index-1); k < upper_limit && k < ObjPgCnt; k++ )
            {
            printf( "\t\t %u\t%lu (0x%lx)\t%u\t%s - %4.4x\n", k+1,
               ObjPgList[k]->offset, ObjPgList[k]->offset,
                  ObjPgList[k]->size, pgflags[ObjPgList[k]->flags], 
                     ObjPgList[k]->flags ); 
            }
         }
      printf( "\n" );
      }
   printf( "\nResident Name Table\n" );
   if( !ResNmList[0] )
      printf( "\tEMPTY\n" );
   else
      printf( "\tOrdinal\tName\n" );
   // TblLen includes the null terminator at end which we don't store.
   for( i = 0; ResNmList[i] != NULL; i++ )
      {
         printf( "\t%u", ResNmList[i]->ordinal );
         printf( "\t%s\n", ResNmList[i]->string );
      }
   printf( "\nNon Resident Name Table\n" );
   if( !NResNmList[0] )
      printf( "\tEMPTY\n" );
   else
      printf( "\tOrdinal\t\tName\n" );
   // TblLen includes the null terminator at end which we don't store.
   for( i = 0; i < (USHORT )lx_exe.NResNameTblLen && NResNmList[i]; i++ )
      {
         printf( "\t%u", NResNmList[i]->ordinal );
         printf( "\t%s\n", NResNmList[i]->string );
      }
   printf( "\nImported Modules\n" );
   if( !ImpModList[0] )
      printf( "\tEMPTY\n" );
   for( i = 0; ImpModList[i]; i++ )
         printf( "\t%s\n", ImpModList[i]->string );

   printf( "\nImported Procedures\n" );
   if( !ImpProcList[0] )
      printf( "\tEMPTY\n" );
   for( i = 0; ImpProcList[i]; i++ )
         printf( "\t%s\n", ImpProcList[i]->string );

   printf( "\nResources\n" );
   if( !RscList[0] )
      printf( "\tEMPTY\n" );
   else
      printf( "\tName_id\tSize\tObject\tOffset\t\tType_id\n" );
   for( i = 0; i < (USHORT )lx_exe.NumRscTblEnt; i++ )   
      {
      printf( "\t%u", RscList[i]->name_id );
      printf( "\t%lu", RscList[i]->size );     
      printf( "\t%u", RscList[i]->object );  
      printf( "\t%lu (0x%lx)", RscList[i]->offset, RscList[i]->offset ); 
      if( RscList[i]->type_id < 1 || RscList[i]->type_id > RT_FDDIR )
         printf( "\t(%u) BAD TYPE", RscList[i]->type_id );
      else
         printf( "\t(%u) %s", RscList[i]->type_id, resource_types[RscList[i]->type_id] );
      printf( "\n" );
      }

   printf( "\nFixups\n" );
   pg = 0;      
   printf( "Type             \tTarget\tSource\n" );
   for( i = 0; FixupList[i] != NULL && i < MAX_LX_FIXUPS; i++ )
      {
      for( j = 0; j < ObjPgCnt; j++ )
         if( FixupList[i]->offset == FixupPgList[i] )
            {
            pg = j+1;
            break;
            }
      switch( FixupList[i]->flags & LX_FUF_TARGETMASK )
         {
         case LX_FUF_INTERNAL:
            printf( "Internal        " );
            // Since fixups are done on a per-page basis, 
            // target offset is always < 4096 bytes, so %4.4x will do it.
            printf( "\t%u.%4.4x\t%u", pg, FixupList[i]->sc.ofs, FixupList[i]->targ.i.object );
            if(( FixupList[i]->source & LX_SRC_MASK ) != LX_FU_16BITSELECTOR )
               {
               if( FixupList[i]->flags & LX_FUF_32BITTARGOFS )
                  printf( ".%8.8lx", FixupList[i]->targ.i.target_ofs );
               else
                  printf( ".%4.4x", FixupList[i]->targ.i.target_ofs );
               }
            printf( "\n" );
            break;
         case LX_FUF_IMPORTBYORDINAL:
            printf( "Import by ordinal" );
            for( k = 0; ImpModList[k] != NULL && k < MAX_LX_IMPMODS; k++ )
               {
               if(( k + 1 ) == FixupList[i]->targ.o.module )
                  {
                  printf( "\t%u.%x\t%s.%u", pg, FixupList[i]->sc.ofs, ImpModList[k]->string, FixupList[i]->targ.o.ordinal,
                        FixupList[i]->targ.o.additive );
                  if( FixupList[i]->targ.o.additive )
                     printf( " Additive = %lu", FixupList[i]->targ.o.additive );
                  break;
                  }
               }
            printf( "\n" );
            break;
         case LX_FUF_IMPORTBYNAME:
            printf( "Import by name" );
            printf( "\t%u.%x\tmod %u.", pg, FixupList[i]->sc.ofs, FixupList[i]->targ.n.module );
            printf( "proc_nm_ofs %lu", FixupList[i]->targ.n.proc_nm_ofs );
            if( FixupList[i]->targ.n.additive )
               printf( "\tadditive %lu", FixupList[i]->targ.n.additive );
            printf( "\n" );
            break;
         case LX_FUF_IMPORTVIAENTRY:
            printf( "Import by entry" );
            printf( "\t%u.%x\tord %u", pg, FixupList[i]->sc.ofs, FixupList[i]->targ.e.ordinal );
            if( FixupList[i]->targ.e.additive )
               printf( "\tadditive %lu", FixupList[i]->targ.e.additive );
            printf( "\n" );
            break;
         }
      if( FixupList[i]->source & LX_FU_SOURCELIST )
         for( j = 0; j < FixupList[i]->sc.cnt; j++ )
            printf( "\t\t\toffset list[%d] = 0x%4.4x\n", j, FixupList[i]->list[j] );
      }
}

// DumpLXPages - Dump in hex and ASCII all the pages that were loaded
// during the Read.
void DumpLXPages()
{
int j, y;
USHORT i, relative_page,k, upper_limit;

   // Dump all the pages for all the objects
   for( i = 0; i < (USHORT )lx_exe.NumObjects; i++ )   
      {
      printf( "\tObject %d", i+1 );
      printf( ", Size %lu", ObjList[i]->size );     
      printf( ", Addr %u", ObjList[i]->reloc_base_addr );  
      printf( ", Flags %lx", ObjList[i]->obj_flags ); 
      printf( ", PgTableInd %lu", ObjList[i]->pg_tbl_index ); 
      printf( ", NumPgs %lu", ObjList[i]->num_pg_tbl_entries ); 
      printf( ", Rsv%lx", ObjList[i]->reserved ); 
      printf( "\n\t\t\t" );
      for( j = 0, y = 0x0001; j < 16; j++ )
         {
         if( ObjList[i]->obj_flags & y )
            printf( "%s, ", obj_flags[j] );         
         y *= 2;
         }
      printf( "\n" );
      // Are there any pages attached to this object
      if( ObjList[i]->num_pg_tbl_entries )
         {
         upper_limit = (USHORT )(ObjList[i]->pg_tbl_index-1+ObjList[i]->num_pg_tbl_entries);
         // List those pages
         for( k = (USHORT )(ObjList[i]->pg_tbl_index-1); k < upper_limit && k < ObjPgCnt; k++ )
            {
            relative_page = (USHORT )(k - (ObjList[i]->pg_tbl_index-1) + 1);
            printf( "\tObject %d, relative page %d\n", i+1, relative_page );
            printf( "\tPAGE:\tNumber\tOffset\t\tSize\tFlags\n" );
            printf( "\t\t %u\t%lu (0x%lx)\t%u\t%s - %4.4x\n", k+1,
               ObjPgList[k]->offset, ObjPgList[k]->offset,
                  ObjPgList[k]->size, pgflags[ObjPgList[k]->flags], 
                     ObjPgList[k]->flags ); 
            if( ObjPgs[k] )
               dump( ObjPgs[k], ObjPgList[k]->size, (ULONG )(lx_exe.PgSize*(relative_page-1)), 2 );
            }
         }
      printf( "\n" );
      }
}

// FreeLXPages - Free the actual physical pages.
void FreeLXPages()
{
USHORT i;

   // Free all the in-memory pages
   for( i = 0; i < ObjPgCnt; i++ )   
      if( ObjPgs[i] )
         {
         free( ObjPgs[i] );
         ObjPgs[i] = NULL;
         }
}

// LoadFixupPageTable - Locate the Fixup Page table and load it's contents
// into FixupPgList[] array using ObjPgCnt (set by LoadObjectTable) as the 
// count.
static void LoadFixupPageTable()
{
USHORT i;

    if( fseek( fp, mz_exe.lfanew + lx_exe.FixupPgTblOfs, 0 ))
      {
      printf(  "Can't seek to Fixup Page Table 0x%lx\n", mz_exe.lfanew+lx_exe.FixupPgTblOfs );
      goto error;
      }

   for( i = 0; i < ObjPgCnt; i++ )
      {
       if( fread( (char *)&FixupPgList[i], sizeof( ULONG ), 1, fp ) != 1 )
         goto error;
      }
return;
error:
   printf( "LoadFixupPageTable error!\n" );
}

// LoadFixupRecordTable - Load the Fixup Record Table.  This is the
// craziest LX table to read because of the varying entry structure
// Use the diagram from the article to follow this.
static void LoadFixupRecordTable()
{
USHORT i;
ULONG currloc;
USHORT targ_size, srcsize;
LX_FIXUP fixup;

    if( fseek( fp, mz_exe.lfanew + lx_exe.FixupRecTblOfs, 0 ))
      {
      printf( "Can't seek to Fixup Record Table 0x%lx\n", mz_exe.lfanew+lx_exe.FixupRecTblOfs );
      goto error;
      }

   // read until we hit the Imported Module Name table.
   currloc = mz_exe.lfanew + lx_exe.FixupRecTblOfs;
   for( i = 0;currloc <= mz_exe.lfanew+lx_exe.ImpModTblOfs; i++ )
      {
      memset( &fixup, '\0', sizeof( fixup ));
      fixup.offset = currloc - ( mz_exe.lfanew+lx_exe.FixupRecTblOfs );      

      // This is the only legal way to break out of this algorithm
      // ... by hitting the next section of the file right on the
      // nail.  Gives me the willies but hasn't failed me yet.
      if( currloc == mz_exe.lfanew+lx_exe.ImpModTblOfs )
         break;
       if( fread( (char *)&fixup, (2*sizeof(UCHAR)), 1, fp ) != 1 )
         {
         printf( "Error reading fixup source!\n" );
         goto error;
         }
      currloc += (2*sizeof( UCHAR ));
      if( fixup.source & LX_FU_SOURCELIST )
         srcsize = 1;
      else
         srcsize = 2;

      if( fread( (char *)&fixup.sc, srcsize * sizeof(UCHAR), 1, fp ) != 1 )
         {
         printf( "Error reading fixup target!\n" );
         break;
         }
      currloc += (srcsize*sizeof( UCHAR ));
      switch( fixup.flags & LX_FUF_TARGETMASK )
         {
         case LX_FUF_INTERNAL:
            if( fixup.flags & LX_FUF_16BITOBJORDINAL )
               targ_size = sizeof( USHORT );
            else
               targ_size = sizeof( UCHAR );

            if( fread( (char *)&fixup.targ.i.object, targ_size, 1, fp ) != 1 )
               {
               printf( "Error reading fixup target object (internal)!\n" );
               goto error;
               }
            currloc += targ_size;

            if(( fixup.source & LX_SRC_MASK ) != LX_FU_16BITSELECTOR )
               {
               if( fixup.flags & LX_FUF_32BITTARGOFS )
                  targ_size = sizeof( ULONG );
               else
                  targ_size = sizeof( USHORT );
               if( fread( (char *)&fixup.targ.i.target_ofs, targ_size, 1, fp ) != 1 )
                  {
                  printf( "Error reading fixup target offset (internal)!\n" );
                  goto error;
                  }
               currloc += targ_size;
               }
            break;
         case LX_FUF_IMPORTBYORDINAL:
            if( fixup.flags & LX_FUF_16BITOBJORDINAL )
               targ_size = sizeof( USHORT );
            else
               targ_size = sizeof( UCHAR );

            if( fread( (char *)&fixup.targ.o.module, targ_size, 1, fp ) != 1 )
               {
               printf( "Error reading fixup target module (imp-by-ord)!\n" );
               goto error;
               }
            currloc += targ_size;

            if( fixup.flags & LX_FUF_8BITORDINAL )
               targ_size = sizeof( UCHAR );
            else            
               {
               if( fixup.flags & LX_FUF_32BITTARGOFS )
                  targ_size = sizeof( ULONG );
               else
                  targ_size = sizeof( USHORT );
               }
            if( fread( (char *)&fixup.targ.o.ordinal, targ_size, 1, fp ) != 1 )
               {
               printf( "Error reading fixup target ordinal (imp-by-ord)!\n" );
               goto error;
               }
            currloc += targ_size;
            if( fixup.flags & LX_FUF_ADDITIVEFIXUP )
               {
               if( fixup.flags & LX_FUF_32BITADDITIVE )
                     targ_size = sizeof( ULONG );
               else
                     targ_size = sizeof( USHORT );
               if( fread( (char *)&fixup.targ.o.additive, targ_size, 1, fp ) != 1 )
                  {
                  printf( "Error reading fixup target additive (imp-by-ord)!\n" );
                  goto error;
                  }
               currloc += targ_size;
               }
            break;
         case LX_FUF_IMPORTBYNAME:
            if( fixup.flags & LX_FUF_16BITOBJORDINAL )
               targ_size = sizeof( USHORT );
            else
               targ_size = sizeof( UCHAR );

            if( fread( (char *)&fixup.targ.n.module, targ_size, 1, fp ) != 1 )
               {
               printf( "Error reading fixup target module (imp-by-name)!\n" );
               goto error;
               }
            currloc += targ_size;

            if( fixup.flags & LX_FUF_32BITTARGOFS )
               targ_size = sizeof( ULONG );
            else
               targ_size = sizeof( USHORT );
            if( fread( (char *)&fixup.targ.n.proc_nm_ofs, targ_size, 1, fp ) != 1 )
               {
               printf( "Error reading fixup target proc name offset (imp-by-name)!\n" );
               goto error;
               }
            currloc += targ_size;

            if( fixup.flags & LX_FUF_ADDITIVEFIXUP )
               {
               if( fixup.flags & LX_FUF_32BITADDITIVE )
                     targ_size = sizeof( ULONG );
               else
                     targ_size = sizeof( USHORT );
               if( fread( (char *)&fixup.targ.n.additive, targ_size, 1, fp ) != 1 )
                  {
                  printf( "Error reading fixup target additive (imp-by-name)!\n" );
                  goto error;
                  }
               currloc += targ_size;
               }
            break;
         case LX_FUF_IMPORTVIAENTRY:
            if( fixup.flags & LX_FUF_16BITOBJORDINAL )
               targ_size = sizeof( USHORT );
            else
               targ_size = sizeof( UCHAR );

            if( fread( (char *)&fixup.targ.e.ordinal, targ_size, 1, fp ) != 1 )
                  {
                  printf( "Error reading fixup target ordinal (entry)!\n" );
                  goto error;
                  }
            currloc += targ_size;

            if( fixup.flags & LX_FUF_ADDITIVEFIXUP )
               {
               if( fixup.flags & LX_FUF_32BITADDITIVE )
                     targ_size = sizeof( ULONG );
               else
                     targ_size = sizeof( USHORT );
               if( fread( (char *)&fixup.targ.e.additive, targ_size, 1, fp ) != 1 )
                  {
                  printf( "Error reading fixup target additive (entry)!\n" );
                  goto error;
                  }
               currloc += targ_size;
               }
            break;
         }

         // Now read the source list for this fixup
         if( fixup.source & LX_FU_SOURCELIST )
            {
            if(( fixup.list = malloc( fixup.sc.cnt * sizeof( USHORT ))) == NULL )
               {
               printf( "Error allocing source list (entry)!\n" );
               goto error;
               }
            if( fread(( char *)fixup.list, fixup.sc.cnt * sizeof( USHORT ), 1, fp ) != 1 )
               {
               printf( "Error reading fixup source list!\n" );
               goto error;
               }
            currloc += fixup.sc.cnt * sizeof( USHORT );
            }
         if( AddStructToList((char *)&fixup, (char **)FixupList, sizeof( LX_FIXUP ), MAX_LX_FIXUPS ))
            goto error;
         }
return;
error:
   printf( "LoadFixupRecordTable error!\n" );
}

// LoadImportedModuleTable - Locate the Imported Module Table and load 
// it's contents into ImpModList until entry count is exhausted.
static void LoadImportedModuleTable()
{
USHORT i;
LX_NRES nres;

    if( fseek( fp, (long)mz_exe.lfanew+lx_exe.ImpModTblOfs, 0 ))
      {
      printf( "Can't seek to Imported Module Table 0x%lx\n", mz_exe.lfanew+lx_exe.ImpModTblOfs );
      goto error;
      }

   for( i = 0; i < (USHORT )lx_exe.NumImpModEnt; i++ )   
      {
       if( fread( (char *)&nres.len, sizeof( nres.len ), 1, fp ) == 1 )
         {
         if( !nres.len )   // This is the only legal way to exit algorithm
            break;
         if( fread( (char *)&nres.string, (USHORT )nres.len, 1, fp ) != 1 )
            goto error;
         nres.string[nres.len] = '\0';
         if( AddStructToList((char *)&nres, (char **)ImpModList, sizeof( LX_NRES ), MAX_LX_IMPMODS ))
            goto error;
         }
      }
return;
error:
   printf( "LoadImportedModuleTable error!\n" );
}

// LoadImportedProcedureTable - Locate the Imported Procdedure Table and load 
// it's contents into ImpProcList until we hit the preload data pages 
// section of the file.
static void LoadImportedProcedureTable()
{
USHORT i;
LX_NRES nres;
ULONG currloc;

    if( fseek( fp, (long)mz_exe.lfanew+lx_exe.ImpProcTblOfs, 0 ))
      {
      printf( "Can't seek to Imported Procedure Table 0x%lx\n", mz_exe.lfanew+lx_exe.ImpProcTblOfs );
      goto error;
      }
   currloc = (long)mz_exe.lfanew+lx_exe.ImpProcTblOfs;
   for( i = 0; currloc < lx_exe.DataPgOfs; i++ )   
      {
       if( fread( (char *)&nres.len, sizeof( nres.len ), 1, fp ) == 1 )
         {
         currloc += sizeof( nres.len );
         if( !nres.len )
            break;
         if( fread( (char *)&nres.string, (USHORT )nres.len, 1, fp ) != 1 )
            goto error;
         nres.string[nres.len] = '\0';
         currloc += nres.len;
         if( AddStructToList((char *)&nres, (char **)ImpProcList, sizeof( LX_NRES ), MAX_LX_IMPPROCS ))
            goto error;
         }
      }
return;
error:
   printf( "LoadImportedProcedureTable error!\n" );
}

// LoadObjectTable - Locate the object table and load it's contents
// into ObjList[] array.  Set ObjPgCnt to the number of pages that
// will be contained in the Object Page Table so that we can run
// through that table more easily.
static void LoadObjectTable()
{
LX_OBJ obj;
USHORT i;

    if( fseek( fp, (long)mz_exe.lfanew+lx_exe.ObjTblOfs, 0 ))
      {
      printf( "Can't seek to Object Table 0x%lx\n", mz_exe.lfanew+lx_exe.ObjTblOfs );
      goto error;
      }

   ObjPgCnt = 0;
   for( i = 0; i < (USHORT )lx_exe.NumObjects; i++ )   
      {
      if( fread( (char *)&obj, sizeof( LX_OBJ ), 1, fp ) != 1 )
         goto error;
      if( AddStructToList((char *)&obj, (char **)ObjList, sizeof( LX_OBJ ), MAX_LX_OBJECTS ))
         goto error;

      // Keep track of how many possible pages there are going to be
      if( obj.pg_tbl_index + obj.num_pg_tbl_entries - 1 > ObjPgCnt )
         ObjPgCnt = (USHORT )(obj.pg_tbl_index + obj.num_pg_tbl_entries - 1);
      }
return;
error:
   printf( "LoadObjectTable error!\n" );
}

// LoadObjectPageTable - Locate the object page table and load it's contents
// into ObjPgList[] array.  Use ObjPgCnt as the number of pages that
// will be contained in the Object Page Table (set by LoadObjectTable()).
static void LoadObjectPageTable()
{
LX_PG pg;
USHORT i;

    if( fseek( fp, (long)mz_exe.lfanew+(USHORT )lx_exe.ObjPgTblOfs, 0 ))
      {
      printf( "Can't seek to Object Page Table 0x%lx\n", mz_exe.lfanew+lx_exe.ObjPgTblOfs );
      goto error;
      }
   for( i = 0; i < ObjPgCnt; i++ )
      {
      if( fread( (char *)&pg, sizeof( LX_PG ), 1, fp ) != 1 )
         goto error;
      if( AddStructToList((char *)&pg, (char **)ObjPgList, sizeof( LX_PG ), MAX_LX_PGS ))
         goto error;
      }
return;
error:
   printf( "LoadObjectPageTable error!\n" );
}

// LoadPages - Load the actual physical pages into the ObjPgs list.
// using ObjPgCnt (set by LoadObjectTable) as the count.
static void LoadPages()
{
USHORT i;
ULONG currloc;

   for( i = 0; i < ObjPgCnt; i++ )
      {
      if( !ObjPgList[i]->size )  // Skip 0-length pages
         continue;

      // if ITERATED DATA, offset from start of iterated data
      switch( ObjPgList[i]->flags )
         {
         case LX_DATA_ITERATED:
            currloc = lx_exe.ObjIterPgsOfs + ( ObjPgList[i]->offset << lx_exe.PgOfsShift );
            break;
         default:
            currloc = lx_exe.DataPgOfs + ( ObjPgList[i]->offset << lx_exe.PgOfsShift );
            break;
         }
      fseek( fp, currloc, 0 );
      if(( ObjPgs[i] = malloc( ObjPgList[i]->size )) == NULL )
         {
         printf( "Couldn't malloc %u byte page\n", ObjPgList[i]->size );
         goto error;
         }
      if( fread( ObjPgs[i], ObjPgList[i]->size, 1, fp ) != 1 )
         {
         printf( "Couldn't read page[%d] at %ld, flags %x, size %d!!\n", i, currloc, ObjPgList[i]->flags, ObjPgList[i]->size );
         goto error;
         }
      }
return;
error:
   printf( "LoadPages error!\n" );
}

// LoadResidentNameTable - Locate and load resident name table into
// ResNmList array reading until 0-length name is encountered.
static void LoadResidentNameTable()
{
USHORT i;
LX_NRES nres;     // All name tables use the NRES struct for entries.

    if( fseek( fp, (long)mz_exe.lfanew+lx_exe.ResNameTblOfs, 0 ))
      {
      printf( "Can't seek to Resident Name Table 0x%lx\n", mz_exe.lfanew+lx_exe.ResNameTblOfs );
      goto error;
      }
   i = 0;
   while( 1 )
      {
      if( fread( (char *)&nres.len, sizeof( nres.len ), 1, fp ) == 1 )
         {
         if( nres.len == 0 )  // the only legal way to leave algorithm
            break;
         if( fread( (char *)&nres.string, (USHORT )nres.len, 1, fp ) != 1 )
            goto error;
         nres.string[nres.len] = '\0';
         if( fread( (char *)&nres.ordinal, sizeof( nres.ordinal ), 1, fp ) != 1 )
            goto error;
         if( AddStructToList((char *)&nres, (char **)ResNmList, sizeof( LX_NRES ), MAX_LX_RESNAMES ))
            goto error;
         }
      else
         goto error;
      i++;
      }
return;
error:
   printf( "LoadResidentNameTable error!\n" );
}

// LoadNonResidentNameTable - Locate and load non-resident name table into
// NResNmList array reading until NonResNameTable byte count is 
// exhausted.
static void LoadNonResidentNameTable()
{
USHORT i;
LX_NRES nres;     // All name tables use the NRES struct for entries.

   // Let's read the non-resident name table
   // non resident name table offset is relative to BOF!!!
    if( fseek( fp, lx_exe.NResNameTblOfs, 0 ))
      {
      printf( "Can't seek to Non-Resident Name Table 0x%lx\n", lx_exe.NResNameTblOfs );
      goto error;
      }

   for( i = 0; i < (USHORT )lx_exe.NResNameTblLen; i++ )
      {
       if( fread( (char *)&nres.len, sizeof( nres.len ), 1, fp ) == 1 )
         {
         if( !nres.len )   // Only legal way to leave algorithm
            break;
         if( fread( (char *)nres.string, (USHORT )nres.len, 1, fp ) != 1 )
            goto error;
         nres.string[nres.len] = '\0';
         if( fread( (char *)&nres.ordinal, sizeof( nres.ordinal ), 1, fp ) != 1 )
            goto error;
         if( AddStructToList((char *)&nres, (char **)NResNmList, sizeof( LX_NRES ), MAX_LX_NRESNAMES ))
            goto error;
         }
      }
return;
error:
   printf( "LoadNonResidentNameTable error!\n" );
}

// LoadResourceTable - Locate and load resource table into
// ResourceList array reading until resource count is 
// exhausted.
static void LoadResourceTable()
{
USHORT i;
LX_RSC rsc;

   // Let's read the resource table
    if( fseek( fp, (long)mz_exe.lfanew+lx_exe.RscTblOfs, 0 ))
      {
      printf( "Can't seek to Resource Table 0x%lx\n", mz_exe.lfanew+lx_exe.RscTblOfs );
      goto error;
      }

   for( i = 0; i < (USHORT )lx_exe.NumRscTblEnt; i++ )   
      {
      if( fread( (char *)&rsc, sizeof( LX_RSC ), 1, fp ) != 1 )
         goto error;
      if( AddStructToList((char *)&rsc, (char **)RscList, sizeof( LX_RSC ), MAX_LX_RESOURCES ))
         goto error;
      }
return;
error:
   printf( "LoadResourceTable error!\n" );
}

// Read_LXExe - Read the magic bytes, and the LX header, then
// call the individual Load... functions to load the tables and 
// the physical pages.  Tables and pages are loaded in the order
// in which they appear in the file.
int Read_LXExe()
{
   if( bVerbose )
       printf( "xxx   Reading LX executable header.\n" );

   ObjPgCnt = 0;

   // Seek to the magic bytes
    if( fseek( fp, (long)mz_exe.lfanew, 0 ))
      {
      printf(  "Can't seek to %04xh\n", mz_exe.lfarlc );
      goto error;
      }

   // Read the two magic bytes
    if( fread( (char *)&magic, sizeof( magic ), 1, fp ) != 1 )
      {
      printf(  "Can't read magic.\n" );
      goto error;
      }
   // Read the LX header.
    if( fread( (char *)&lx_exe, sizeof( LX_EXE ), 1, fp ) != 1 )
      goto error;

   LoadObjectTable();
   LoadObjectPageTable();
   LoadResourceTable();
   LoadResidentNameTable();
   LoadFixupPageTable();
   LoadFixupRecordTable();
   LoadImportedModuleTable();
   LoadImportedProcedureTable();
   LoadPages();
   LoadNonResidentNameTable();

return( 0 );
error:
return( 1 );
}

