/*
**	$id: ssvcid bitmaps.c 1.1 11/12/91  8:38 am$
**		Main program for Bitmap and Icon File demo program.
**
**	(C) 1991 Larry Widing
*/

#include	<windows.h>
#include	"bitmaps.h"
#include	"bmfile.h"
#include	"bmmanip.h"
#include	"bminfo.h"
#include	"icnfile.h"

/*
**	Global Variables
*/
HWND		MainWindow;			/* Handle to the application's main window */
HANDLE	AppInstance;		/* Handle to application's instance */
char		FileName[128];		/* Name of file returned by open file dialog */
HBITMAP	BitmapHandle;		/* Handle of currently loaded bitmap */
HICON		IconHandle;			/* Handle of currently loaded icon */
HANDLE	DIBitmapHandle;	/* Handle of packed DI Bitmap */
int		FileSaveMode = 0;	/* 0 to save as a normal DIB, 1 for RLE, 2 for OS2 */

const char	AppName[] = "Bitmaps";

/*
**	Local function prototypes
*/
BOOL					RegisterWindows(HANDLE, HANDLE);
BOOL					InitializeApplication(HANDLE, HANDLE *, int);
#if	defined(__TSC__)
	#pragma	save
	#pragma	call(windows=>on)
#endif
LONG FAR PASCAL	MainWndProc(HWND, WORD, WORD, DWORD);
BOOL FAR PASCAL	OpenFileDialog(HWND, WORD, WORD, DWORD);
BOOL FAR PASCAL	AboutBoxDialog(HWND, WORD, WORD, DWORD);
#if	defined(__TSC__)
	#pragma	restore
#endif

/*
** int PASCAL
** WinMain(
**   HANDLE hInstance,
**   HANDLE hPrevInstance,
**   LPSTR lpCmdLine,
**   int nCmdShow);
**
**    Entry point from windows for this application.
**
** Modification History:
** 08/20/91  LCW  Created
*/
int PASCAL
WinMain(HANDLE hInstance, HANDLE hPrevInstance, LPSTR lpCmdLine, int nCmdShow)
{
	MSG		msg;
	HANDLE	accelTable;

	/*
	**	Register main window class
	*/
	if (!RegisterWindows(hInstance, hPrevInstance))
		return 1;

	/*
	**	Initialize main window, and application
	*/
	if (!InitializeApplication(hInstance, &accelTable, nCmdShow))
		return 1;

	while (GetMessage((LPMSG)&msg, NULL, 0, 0))
	{
		if (!TranslateAccelerator(MainWindow, accelTable, (LPMSG)&msg))
		{
			TranslateMessage((LPMSG)&msg);
			DispatchMessage((LPMSG)&msg);
		}
	}

	return msg.wParam;
}

/*
** BOOL								// TRUE if successful, FALSE if an error occurred
** RegisterWindows(
**   HANDLE hInstance,			// Application instance handle
**   HANDLE hPrevInstance);	// handle of previous instance of this application
**
**    Register window classes used in this application.
**
** Modification History:
** 08/20/91  LCW  Created
*/
BOOL
RegisterWindows(HANDLE hInstance, HANDLE hPrevInstance)
{
	WNDCLASS	class;

	if (hPrevInstance)
		return TRUE;

	class.style = 0;
	class.lpfnWndProc = MainWndProc;
	class.cbClsExtra = 0;
	class.hInstance = hInstance;
	class.hIcon = LoadIcon(hInstance, (LPSTR)AppName);
	class.hCursor = LoadCursor((HANDLE)NULL, IDC_ARROW);
	class.hbrBackground = COLOR_APPWORKSPACE + 1;
	class.lpszMenuName = (LPSTR)AppName;
	class.lpszClassName = (LPSTR)AppName;

	if (!RegisterClass((LPWNDCLASS)&class))
	{
		MessageBox((HWND)NULL, (LPSTR)"Cannot register main window class",
			(LPSTR)"Bitmaps Error", MB_APPLMODAL | MB_OK | MB_ICONSTOP);
		return FALSE;
	}

	return TRUE;
}

/*
** BOOL
** InitializeApplication(
**   HANDLE hInstance,
**   HANDLE *accelTable,
**   int nCmdShow);
**
**    Initialize this application.  This involves loading the menu for the
**	main window, creating the main window, and loading the accelerator table
**	from the resource file.
**
** Modification History:
** 08/20/91  LCW  Created
*/
BOOL
InitializeApplication(HANDLE hInstance, HANDLE *accelTable, int nCmdShow)
{
	HMENU		menu;

	menu = LoadMenu(hInstance, (LPSTR)AppName);

	MainWindow = CreateWindow((LPSTR)AppName,
		(LPSTR)"Bitmap and Icon File Demo",
		WS_OVERLAPPEDWINDOW, CW_USEDEFAULT, CW_USEDEFAULT, CW_USEDEFAULT,
		CW_USEDEFAULT, (HWND)NULL, menu, hInstance, (LPSTR)NULL);

	if (MainWindow == (HWND)NULL)
	{
		MessageBox((HWND)NULL, (LPSTR)"Cannot open main window", (LPSTR)"Bitmaps Error",
			MB_APPLMODAL | MB_OK | MB_ICONSTOP);
		return FALSE;
	}

	ShowWindow(MainWindow, nCmdShow);
	*accelTable = LoadAccelerators(hInstance, (LPSTR)AppName);

	AppInstance = hInstance;
	BitmapHandle = (HBITMAP)NULL;
	IconHandle = (HICON)NULL;

	return TRUE;
}

static void
ClearHandles(void)
{
	if (DIBitmapHandle != (HANDLE)NULL)
	{
		GlobalFree(DIBitmapHandle);
		DIBitmapHandle = (HANDLE)NULL;
	}

	if (BitmapHandle != (HBITMAP)NULL)
	{
		DeleteObject(BitmapHandle);
		BitmapHandle = (HBITMAP)NULL;
	}

	if (IconHandle != (HICON)NULL)
	{
		DestroyIcon(IconHandle);
		IconHandle = (HICON)NULL;
	}
}

/*
** LONG FAR PASCAL
** MainWndProc(
**   HWND wnd,				Handle of window
**   WORD message,		Message received from Windows
**   WORD wParam,			word parameter
**   DWORD lParam);		long parameter
**
**    Handle message from the main window.
**
** Modification History:
** 09/06/91  LCW  Created
*/
LONG FAR PASCAL
MainWndProc(HWND wnd, WORD message, WORD wParam, DWORD lParam)
{
	FARPROC	pfn;
	int		rc;
	HANDLE	handle;

	switch (message)
	{
		case WM_COMMAND:
			switch (wParam)
			{
				case IDM_FILE_OPENBITMAP:
					pfn = MakeProcInstance((FARPROC)OpenFileDialog, AppInstance);
					if (pfn != (FARPROC)NULL)
					{
						lstrcpy((LPSTR)FileName, (LPSTR)"*.bmp");
						rc = DialogBox(AppInstance, (LPSTR)"OpenFile", wnd, pfn);
						FreeProcInstance(pfn);
						if (rc == 1)
						{
							/* Attempt to load the bitmap */
							handle = ReadBitmapFile(FileName);
							if (handle != (HBITMAP)NULL)
							{
								ClearHandles();
								DIBitmapHandle = handle;
								InvalidateRect(wnd, NULL, TRUE);
							}
						}
					}
					break;

				case IDM_FILE_OPENICON:
					pfn = MakeProcInstance((FARPROC)OpenFileDialog, AppInstance);
					if (pfn != (FARPROC)NULL)
					{
						lstrcpy((LPSTR)FileName, (LPSTR)"*.ico");
						rc = DialogBox(AppInstance, (LPSTR)"OpenFile", wnd, pfn);
						FreeProcInstance(pfn);

						if (rc == 1)
						{
							/* Attempt to load the icon */
							handle = ReadIconFile(FileName);
							if (handle != (HICON)NULL)
							{
								ClearHandles();
								IconHandle = handle;
								InvalidateRect(wnd, NULL, TRUE);
							}
						}
					}
					break;

				case IDM_FILE_FMT_DIB:
					FileSaveMode = 0;
					break;

				case IDM_FILE_FMT_RLE:
					FileSaveMode = 1;
					break;

				case IDM_FILE_FMT_OS2:
					FileSaveMode = 2;
					break;

				case IDM_FILE_SAVEBITMAP:
					pfn = MakeProcInstance((FARPROC)OpenFileDialog, AppInstance);
					if (pfn != (FARPROC)NULL)
					{
						lstrcpy((LPSTR)FileName, (LPSTR)"*.bmp");
						rc = DialogBox(AppInstance, (LPSTR)"SaveFile", wnd, pfn);
						FreeProcInstance(pfn);

						if (rc == 1)
						{
							/* Attempt to save the bitmap */
							if (DIBitmapHandle == (HANDLE)NULL)
							{
								/* First, convert to a DIB */
								handle = BitmapToDIB(BitmapHandle, FileSaveMode);
								if (handle != (HANDLE)NULL)
								{
									WriteBitmapFile(FileName, handle);
									GlobalFree(handle);
								}
							}
							else
							{
								WriteBitmapFile(FileName, DIBitmapHandle);
							}
						}
					}
					break;

				case IDM_FILE_EXIT:
					DestroyWindow(wnd);
					break;

				case IDM_FILE_ABOUT:
					pfn = MakeProcInstance((FARPROC)AboutBoxDialog, AppInstance);
					if (pfn != (FARPROC)NULL)
					{
						DialogBox(AppInstance, (LPSTR)"AboutBox", wnd, pfn);
						FreeProcInstance(pfn);
					}
					break;

				case IDM_EDIT_COPY:
					if (OpenClipboard(wnd))
					{
						int	fmt = 0;

						handle = (HANDLE)NULL;

						if (DIBitmapHandle != (HANDLE)NULL)
						{
							handle = CopyDIBitmap(DIBitmapHandle);
							fmt = CF_DIB;
						}
						else if (BitmapHandle != (HBITMAP)NULL)
						{
							handle = CopyBitmap(BitmapHandle);
							fmt = CF_BITMAP;
						}

						if (handle != (HANDLE)NULL)
						{
							EmptyClipboard();
							SetClipboardData(fmt, handle);
						}
						CloseClipboard();
					}
					break;

				case IDM_EDIT_PASTE:
					if (OpenClipboard(wnd))
					{
						HANDLE	handle;
						int		fmt = 0, fbmp = FALSE, fdib = FALSE;

						while ((fmt = EnumClipboardFormats(fmt)) != 0)
						{
							if (fmt == CF_BITMAP)
							{
								fbmp = TRUE;
							}
							if (fmt == CF_DIB)
							{
								fdib = TRUE;
							}
						}
						if (fdib)
						{
							fmt = CF_DIB;
						}
						else if (fbmp)
						{
							fmt = CF_BITMAP;
						}

						if (fmt != 0)
						{
							handle = GetClipboardData(fmt);
							if (fmt == CF_BITMAP)
							{
								HBITMAP	hbm;

								hbm = CopyBitmap(handle);
								if (hbm != (HBITMAP)NULL)
								{
									ClearHandles();
									BitmapHandle = hbm;
									InvalidateRect(wnd, NULL, TRUE);
								}
							}
							else if (fmt == CF_DIB)
							{
								HANDLE	hbm;

								hbm = CopyDIBitmap(handle);
								if (hbm != (HANDLE)NULL)
								{
									ClearHandles();
									DIBitmapHandle = hbm;
									InvalidateRect(wnd, NULL, TRUE);
								}
							}
						}

						CloseClipboard();
					}
					break;

				case IDM_INFO_ABOUT:
					InfoDisplay();
					break;

				case IDM_FILE_PRINT:
					break;

				case IDM_FILE_PRSETUP:
					break;

				case IDM_CONVERT_LOGICAL:
					handle = DibToBitmap(DIBitmapHandle);
					if (handle)
					{
						ClearHandles();
						BitmapHandle = handle;
						InvalidateRect(wnd, NULL, TRUE);
					}
					break;

				case IDM_CONVERT_DIB:
					if (BitmapHandle)
					{
						handle = BitmapToDIB(BitmapHandle, 0);
					}
					else
					{
						HBITMAP	hbm = DibToBitmap(DIBitmapHandle);

						if (hbm)
						{
							handle = BitmapToDIB(hbm, 0);
							DeleteObject(hbm);
						}
					}
					if (handle)
					{
						ClearHandles();
						DIBitmapHandle = handle;
						InvalidateRect(wnd, NULL, TRUE);
					}
					break;

				case IDM_CONVERT_RLE:
					if (BitmapHandle)
					{
						handle = BitmapToDIB(BitmapHandle, 1);
					}
					else
					{
						HBITMAP	hbm = DibToBitmap(DIBitmapHandle);

						if (hbm)
						{
							handle = BitmapToDIB(hbm, 1);
							DeleteObject(hbm);
						}
					}
					if (handle)
					{
						ClearHandles();
						DIBitmapHandle = handle;
						InvalidateRect(wnd, NULL, TRUE);
					}
					break;

				case IDM_CONVERT_OS2:
					if (BitmapHandle)
					{
						handle = BitmapToDIB(BitmapHandle, 2);
					}
					else
					{
						HBITMAP	hbm = DibToBitmap(DIBitmapHandle);

						if (hbm)
						{
							handle = BitmapToDIB(hbm, 2);
							DeleteObject(hbm);
						}
					}
					if (handle)
					{
						ClearHandles();
						DIBitmapHandle = handle;
						InvalidateRect(wnd, NULL, TRUE);
					}
					break;
			}
			break;

		case WM_PAINT:
		{
			HDC			hdc;
			PAINTSTRUCT	ps;

			hdc = BeginPaint(wnd, (LPPAINTSTRUCT)&ps);

			/*
			**	Draw the current bitmap or icon, if either one is loaded
			*/
			if (DIBitmapHandle != (HANDLE)NULL)
			{
				DrawDIBitmap(hdc, 0, 0, DIBitmapHandle);
			}
			else if (BitmapHandle != (HBITMAP)NULL)
			{
				DrawBitmap(hdc, 0, 0, BitmapHandle);
			}
			else if (IconHandle != (HICON)NULL)
			{
				DrawIcon(hdc, 0, 0, IconHandle);
			}

			EndPaint(wnd, (LPPAINTSTRUCT)&ps);
			break;
		}

		case WM_INITMENUPOPUP:
			if (HIWORD(lParam) == 0)
			{
				switch (LOWORD(lParam))
				{
					case 0:	/* File Menu */
						/*
						**	Enable bitmap saving only if a bitmap is loaded
						*/
						if (BitmapHandle != (HBITMAP)NULL || DIBitmapHandle != (HANDLE)NULL)
						{
							EnableMenuItem(wParam, IDM_FILE_SAVEBITMAP, MF_ENABLED);
						}
						else
						{
							EnableMenuItem(wParam, IDM_FILE_SAVEBITMAP, MF_GRAYED);
						}

						/*
						**	Check appropriate save mode
						*/
						CheckMenuItem(wParam, IDM_FILE_FMT_DIB, MF_UNCHECKED);
						CheckMenuItem(wParam, IDM_FILE_FMT_RLE, MF_UNCHECKED);
						CheckMenuItem(wParam, IDM_FILE_FMT_OS2, MF_UNCHECKED);

						CheckMenuItem(wParam, IDM_FILE_FMT_DIB + FileSaveMode, MF_CHECKED);
						break;

					case 1:	/* Edit menu */
						/*
						**	Enable copy menu item if we have a bitmap to copy
						*/
						if (BitmapHandle != (HBITMAP)NULL
							|| DIBitmapHandle != (HANDLE)NULL)
						{
							EnableMenuItem(wParam, IDM_EDIT_COPY, MF_ENABLED);
						}
						else
						{
							EnableMenuItem(wParam, IDM_EDIT_COPY, MF_GRAYED);
						}

						/*
						**	Enable paste menu item if the clipboard has a bitmap to paste
						*/
						EnableMenuItem(wParam, IDM_EDIT_PASTE, MF_GRAYED);
						if (OpenClipboard(MainWindow))
						{
							int	fmt = 0;

							while ((fmt = EnumClipboardFormats(fmt)) != 0)
							{
								if (fmt == CF_BITMAP || fmt == CF_DIB)
								{
									break;
								}
							}
							if (fmt != 0)
							{
								EnableMenuItem(wParam, IDM_EDIT_PASTE, MF_ENABLED);
							}
							CloseClipboard();
						}
						break;

					case 2:	/* Information Menu */
						break;

					case 3:	/* Conversion Menu */
					{
						int	logical, dib, rle, os2;

						logical = dib = rle = os2 = MF_ENABLED;

						if (IconHandle)
						{
							logical = dib = rle = os2 = MF_GRAYED;
						}
						else if (BitmapHandle)
						{
							logical = MF_GRAYED;
						}
						else if (DIBitmapHandle)
						{
							if (DibIsOs2(DIBitmapHandle))
							{
								os2 = MF_GRAYED;
							}
							else if (DibIsCompressed(DIBitmapHandle))
							{
								rle = MF_GRAYED;
							}
							else
							{
								dib = MF_GRAYED;
							}
						}
						else
						{
							logical = dib = rle = os2 = MF_GRAYED;
						}

						EnableMenuItem(wParam, IDM_CONVERT_LOGICAL, logical);
						EnableMenuItem(wParam, IDM_CONVERT_DIB, dib);
						EnableMenuItem(wParam, IDM_CONVERT_RLE, rle);
						EnableMenuItem(wParam, IDM_CONVERT_OS2, os2);
						break;
					}
				}
			}
			break;

		case WM_DESTROY:
			ClearHandles();
			PostQuitMessage(0);
			break;

		default:
			return DefWindowProc(wnd, message, wParam, lParam);
	}

	return FALSE;
}

/*
** BOOL FAR PASCAL
** AboutBoxDialog(
**   HWND dlg,				Handle of dialog
**   WORD message,		Message received from Windows
**   WORD wParam,			word parameter
**   DWORD lParam);		long parameter
**
**    Handle the events related to the About Box.
**
** Modification History:
** 09/06/91  LCW  Created
*/
BOOL FAR PASCAL
AboutBoxDialog(HWND dlg, WORD message, WORD wParam, DWORD lParam)
{
	switch (message)
	{
		case WM_COMMAND:
			switch (wParam)
			{
				case IDOK:
					EndDialog(dlg, 0);
					return TRUE;
			}
	}

	return FALSE;
}

/*
** BOOL FAR PASCAL
** OpenFileDialog(
**   HWND dlg,				Handle of dialog
**   WORD message,		Message received from Windows
**   WORD wParam,			word parameter
**   DWORD lParam);		long parameter
**
**    Handle the getting of a name to use for opening or saving a bitmap
**	or icon.
**
** Modification History:
** 09/06/91  LCW  Created
*/
BOOL FAR PASCAL
OpenFileDialog(HWND dlg, WORD message, WORD wParam, DWORD lParam)
{
	int	rc;
	HWND	wnd;

	switch (message)
	{
		case WM_COMMAND:
			switch (wParam)
			{
				case IDOK:
					wnd = GetDlgItem(dlg, IDD_FILENAME);
					rc = GetWindowTextLength(wnd);
					if (rc > 0 && rc < sizeof(FileName))
					{
						GetWindowText(wnd, (LPSTR)FileName, sizeof(FileName));
					}
					else
					{
						FileName[0] = '\0';
					}
					EndDialog(dlg, 1);
					return TRUE;

				case IDCANCEL:
					EndDialog(dlg, 0);
					return TRUE;
			}
			return FALSE;

		case WM_INITDIALOG:
			wnd = GetDlgItem(dlg, IDD_FILENAME);
			SendMessage(wnd, EM_LIMITTEXT, sizeof(FileName), 0L);
			SetWindowText(wnd, (LPSTR)FileName);
			SetFocus(wnd);
			return TRUE;
	}

	return FALSE;
}

/*
** void
** ErrorBox(char *msg);		pointer to message to be displayed
**
**    This function displays a simple error message in a Windows message box
**	to inform the user of a problem.
**
** Modification History:
** 09/06/91  LCW  Created
*/
void
ErrorBox(char *msg)
{
	MessageBox(MainWindow, (LPSTR)msg, NULL, MB_OK | MB_ICONSTOP | MB_TASKMODAL);
}

/*
**	Modification History
**	--------------------
**	$lgb$
** 10/15/91     Larry Widing   Initial version for Win Tech Journal Article.
** 11/12/91     Larry Widing   Added support for bitmap conversion.
**	$lge$
*/
