/*
 EPSHeader

   File: wildfile.c
   Author: J. Kercheval
   Created: Thu, 03/14/1991  21:49:02
*/
/*
 EPSRevision History

   J. Kercheval  Sat, 03/16/1991  15:48:38  add and use FileInfo struct
   J. Kercheval  Sat, 03/16/1991  20:30:52  add status byte and '.' checking
   J. Kercheval  Sat, 03/16/1991  22:50:42  _FA_NORMAL files unique, similar to _FA_READONLY behavior
   J. Kercheval  Thu, 03/28/1991  21:23:37  move DOT_LAST to find_firstfile
   J. Kercheval  Sun, 03/31/1991  14:31:15  change '/' to '\\' in path
   D. Kirschbaum Mon, 05/13/1991  16:42:00  Minor Turbo C changes v1.14
   F.C. Smith    Sat, 01/04/1992  (late)    allow d: in splitpath(),
                                            required d:\ previously.
   J. Kercheval  Mon, 01/06/1992  22:00:08  use general match code (ifdef'd)
*/

#include <stdlib.h>
#include <string.h>

#include "wildfile.h"
#include "match.h"

#ifndef __TURBOC__
#define findfirst(n, a, s) _dos_findfirst(n, s, a)
#define findnext _dos_findnext
#else                           /* __TURBOC__ */
#include <ctype.h>              /* toupper() v1.14 */
#endif

#define DOT_LAST 0x01           /* set status byte if last char in pattern is
                                 * '.' */

void splitpath(FileInfo * ff);


/*----------------------------------------------------------------------------
 *
 * find_firstfile()
 *
 ---------------------------------------------------------------------------*/

BOOLEAN find_firstfile(FileInfo * ff)
{
    char path[MAXPATH + 1];     /* found file and path */
    char *c;                    /* temporary pointer variable */

    int error_return;           /* error return from is_valid_pattern */

    BOOLEAN found;              /* looping flag */
    BOOLEAN dot_found;          /* '.' character checking */

    /* empty strings are bummers, aren't they? */
    if (!*(ff->file_pattern)) {
        return FALSE;
    }

    /* convert to upper case */
    c = ff->file_pattern;
    while (*c) {
        *c = (char) toupper(*c);

        /* if the last pattern char is '.' then set status byte correctly */
        if (*c == '.')
            ff->status |= DOT_LAST;
        else
            ff->status &= ~DOT_LAST;

        c++;
    }

    /* check for valid pattern string */
    if (!is_valid_pattern(ff->file_pattern, &error_return)) {
        return FALSE;
    }

    /* copy the path portion of the pattern to file_path */
    splitpath(ff);

    /* create the search path */
    strcpy(path, ff->file_path);
    strcat(path, "*.*");

    /* check that the search path will not be too long */
    if (strlen(path) > MAXPATH) {
        return FALSE;
    }

    /* obtain the first file conforming to path */
    if (findfirst(path, &(ff->file), (ff->file_attributes) & ~_FA_NORMAL)) {
        return (FALSE);
    }

    /* if the file name is empty we want it not */
    if (!*(ff->file.name)) {
        return (FALSE);
    }

    /* init looping variable */
    found = FALSE;

    /* Build path and file name needed for validation */
    strcpy(path, ff->file_path);
    strcat(path, ff->file.name);

    /* Check for '.' if required */
    if (ff->status & DOT_LAST) {

        /* init looping variables */
        dot_found = FALSE;
        c = ff->file.name;

        /* see if file has extension and give it '.' if not */
        while (*c) {
            if (*c == '.')
                dot_found = TRUE;
            c++;
        }
        if (!dot_found)
            strcat(path, ".");
    }

    /* Verify name and attribute wanted for files */
    if (match(ff->file_pattern, path)) {

        if (!ff->file.attributes) {
            if (ff->file_attributes & _FA_NORMAL) {
                found = TRUE;
            }
        }
        else {
            if (ff->file_attributes & ff->file.attributes) {
                found = TRUE;
            }
        }
    }

    while (!found) {

        /* obtain the next file comforming to path */
        if (findnext(&(ff->file))) {
            return (FALSE);
        }

        /* if the file name is empty we want it not */
        if (!*(ff->file.name)) {
            return (FALSE);
        }

        /* Build path and file name needed for validation */
        strcpy(path, ff->file_path);
        strcat(path, ff->file.name);

        /* Check for '.' if required */
        if (ff->status & DOT_LAST) {

            /* init looping variables */
            dot_found = FALSE;
            c = ff->file.name;

            /* see if file has extension and give it '.' if not */
            while (*c) {
                if (*c == '.')
                    dot_found = TRUE;
                c++;
            }
            if (!dot_found)
                strcat(path, ".");
        }

        /* Verify name and attribute wanted for files */
        if (match(ff->file_pattern, path)) {

            if (!ff->file.attributes) {
                if (ff->file_attributes & _FA_NORMAL) {
                    found = TRUE;
                }
            }
            else {
                if (ff->file_attributes & ff->file.attributes) {
                    found = TRUE;
                }
            }
        }
    }

    return (TRUE);
}


/*----------------------------------------------------------------------------
 *
 * splitpath() is a helper function to isolate the pattern from the path.
 * The parameter r gets the path portion of the pattern string p.
 *
 ---------------------------------------------------------------------------*/

void splitpath(FileInfo * ff)
{
    char *last_slash, *c, oldchar;

    /* init last_slash and c */
    c = last_slash = ff->file_pattern;
    last_slash--;

    /* loop through and find the beginning of the pattern or end of string */
    while (*c && *c != '[' && *c != '*' && *c != '?') {

        /* change '/' to '\' */
        if (*c == '/') {
            *c = '\\';
        }

        /* FCS 01-04-92 add the test for ':' */
        /* if c is a '\' then remember where it is */
        if (*c == '\\' || *c == ':') {
            last_slash = c;
        }

        /* move to next char in pattern */
        c++;
    }

    /* move to next char beyond '\' or to beginning of string */
    last_slash++;

    /* store character and make it end of string for now */
    oldchar = *last_slash;
    *last_slash = '\0';

    /* copy the beginning of the path to return value */
    strcpy(ff->file_path, ff->file_pattern);

    /* restore character */
    *last_slash = oldchar;
}


/*----------------------------------------------------------------------------
 *
 * find_nextfile()
 *
 ---------------------------------------------------------------------------*/

BOOLEAN find_nextfile(FileInfo * ff)
{
    char path[MAXPATH + 1];     /* full filename and path for pattern
                                 * checking */
    char *c;                    /* tempory pointer variable */

    BOOLEAN found;              /* looping flag */
    BOOLEAN dot_found;          /* '.' character checking */

    found = FALSE;

    while (!found) {

        /* obtain the next file comforming to path */
        if (findnext(&(ff->file))) {
            return (FALSE);
        }

        /* if the file name is empty we want it not */
        if (!*(ff->file.name)) {
            return (FALSE);
        }

        /* Build path and file name needed for validation */
        strcpy(path, ff->file_path);
        strcat(path, ff->file.name);

        /* Check for '.' if required */
        if (ff->status & DOT_LAST) {

            /* init looping variables */
            dot_found = FALSE;
            c = ff->file.name;

            /* see if file has extension and give it '.' if not */
            while (*c) {
                if (*c == '.')
                    dot_found = TRUE;
                c++;
            }
            if (!dot_found)
                strcat(path, ".");
        }

        /* Verify name and attribute wanted for files */
        if (match(ff->file_pattern, path)) {

            if (!ff->file.attributes) {
                if (ff->file_attributes & _FA_NORMAL) {
                    found = TRUE;
                }
            }
            else {
                if (ff->file_attributes & ff->file.attributes) {
                    found = TRUE;
                }
            }
        }
    }

    return (TRUE);
}


#ifdef TEST

/*
 * The test loop expects the first input parameter on the command line to be
 * the attributes desired in hex format (ie. 0x3f) and the following
 * parameters to be the search path/file name.  Will output the names of all
 * files which match the attributes wanted and the search path/file names. 
 */

#include <stdio.h>

int main(int argc, char *argv[])
{
    FileInfo ff;

    unsigned int tmp;

    /* parse the attribute from the command line */
    sscanf(argv[1], "%x", &tmp);
    ff.file_attributes = (unsigned char) tmp;

    /* trace down the argument list until all file specs are parsed */
    argc--;
    argv++;
    argc--;
    argv++;
    while (argc > 0) {
        strcpy(ff.file_pattern, *argv);

        if (find_firstfile(&ff)) {
            printf("File: %s%s\n", ff.file_path, ff.file.name);
        }

        while (find_nextfile(&ff)) {
            printf("File: %s%s\n", ff.file_path, ff.file.name);
        }
        argc--;
        argv++;
    }
    return (0);
}

#endif
