/*                    --- term.c ---
**
**  This example code is a Windows terminal emulator. It can transfer
**  files using ASCII, XMODEM, and YMODEM protocols. This program is
**  provided as an axample of the use of the PCL4W library.
**
**  In order to compile TERM, you will need PCL4W.LIB, PCL4W.DLL, and
**  PCL4W.H from the PCL4W distribution disk.
**
**  The script compiler BUILDER and the windows script interpreter WSI.C
**  are bonus programs (not shareware) for registered users.
**
**  For a list of files that comprise the TERM application, see
**  TERM._M_ or TERM._B_  (Microsoft & Borland makefiles).
**
*/


#include "windows.h"
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "commdlg.h"
#include "term.h"
#include "info.h"
#include "pcl4w.h"
#include "sioerror.h"
#include "ascii.h"
#include "term_io.h"
#include "asdriver.h"
#include "xydriver.h"
#include "expect.h"
#include "config.h"
#include "paint.h"
#include "line.h"
#include "modem_io.h"
#include "miscell.h"
#include "about.h"
#include "accept.h"
#include "constants.h"
#if SCRIPTS
#include "wsi.h"
#endif

/* public globals */
HWND hMainWnd;            /* main window handle */
HWND hInfoWnd;            /* popup handle */
HANDLE hInstance;         /* program instance */
int OnLineFlag = FALSE;   /* TRUE: online */
int FatalFlag = FALSE;    /* TRUE: fatal error */
int DebugFlag = TRUE;     /* TRUE: debug on */
char XferState = ' ';     /* 'X' = XMODEM */
                          /* 'Y' = YMODEM */
                          /* 'I' = INIT_MODEM functions */
                          /* 'H' = HANGUP_MODEM functions */
                          /* 'A' = ASCII transfer */
                          /* 'S' = Script Interpreter (wsi) */
char Configuration[24] = {""};  /* configuration */

/* miscellaneous functions */
void ErrorCheck(int,char *);

#define MAXFILENAME 256
#define POPWIDTH    200
#define POPHEIGHT   100

/* private */
static int WinWidth = 8 * NCOLS;
static int WinHeight = 12 * NROWS + 48;
static int DebugValue = 0;
static char NCGchar = 'C';
static int LastPacket;
static int LastNAKcount = 0;
static long HideTime = 0L;
static RECT MainRect;
static OPENFILENAME ofn;
static char FilterAll[128] = "All Files (*.*)\0*.*\0";
#if SCRIPTS
static char FilterSB[128] = "Script Binaries (*.sb)\0*.sb\0";
#endif
static char FileName[MAXFILENAME];
static char FileTitle[MAXFILENAME];
static int ThePort;
static int Mstate = 0;

void SayNoAT(void)
{
 DisplayLine("AT COMMAND SET not enabled\n");
}

void SayNoScripts(void)
{
 DisplayLine("SCRIPTS not enabled\n");
}

/*
** PostMainHandle() is required only for the
** Shareware version of PCL4W.
*/


void GoScript(int Flag)
{int i;
 if(!ExpectOnLine()) return;
 if(FatalFlag) ErrorMessage("Fatal Error");
 else
   {
#if SCRIPTS
    EnableBreakButton();
    ofn.lpstrFilter = FilterSB;
    if(GetOpenFileName((LPOPENFILENAME)&ofn))
      {/* start script */
       i = wsiStart(ThePort,FileName,Flag);
       if(i<0) SaySiError(i);
       else
         {XferState = 'S';
          SetTitle();
         }
      }
#else
    SayNoScripts();
#endif
   }
}

#if __cplusplus
extern "C" void FAR PASCAL PostMainHandle(HWND);
#else
extern void FAR PASCAL PostMainHandle(HWND);
#endif

int PASCAL WinMain(HANDLE hInst,HANDLE hPrevInstance,
                   LPSTR lpCmdLine,int nCmdShow)
{WNDCLASS  wc1, wc2;
 MSG msg;
 BOOL Result1;
 BOOL Result2;
 if(!hPrevInstance)
   {/* register main window class */
    wc1.style = CS_HREDRAW | CS_VREDRAW;
    wc1.lpfnWndProc = MainWndProc;
    wc1.cbClsExtra = 0;
    wc1.cbWndExtra = 0;
    wc1.hInstance = hInst;
    wc1.hIcon = LoadIcon(hInst, "TermIcon");
    wc1.hCursor = LoadCursor(NULL, IDC_ARROW);
    wc1.hbrBackground = GetStockObject(WHITE_BRUSH);
    wc1.lpszMenuName =  "TermMenu";
    wc1.lpszClassName = "TermWClass";
    Result1 = RegisterClass(&wc1);

    /* register popup window class */
    wc2.style = CS_HREDRAW | CS_VREDRAW | CS_PARENTDC;
    wc2.lpfnWndProc = InfoWndProc;
    wc2.cbClsExtra = 0;
    wc2.cbWndExtra = 0;
    wc2.hInstance = hInst;
    wc2.hIcon = NULL;
    wc2.hCursor = LoadCursor(NULL, IDC_ARROW);
    wc2.hbrBackground = GetStockObject(WHITE_BRUSH);
    wc2.lpszMenuName =  NULL;
    wc2.lpszClassName = "InfoWClass";
    Result2 = RegisterClass(&wc2);

    if((!Result1)||(!Result2)) return FALSE;
   }

 /* create main window */
 hInstance = hInst;
 hMainWnd = CreateWindow(
        "TermWClass",     "Term",          WS_OVERLAPPEDWINDOW,
        CW_USEDEFAULT,    CW_USEDEFAULT,
        WinWidth,         WinHeight,
        NULL,             NULL,
        hInstance,        NULL);

 /* fill in non-variant fields of OPENFILENAME struct. */
 ofn.lStructSize       = sizeof(OPENFILENAME);
 ofn.hwndOwner         = hMainWnd;
 ofn.lpstrFilter       = FilterAll;
 ofn.lpstrCustomFilter = NULL;
 ofn.nMaxCustFilter    = 0;
 ofn.nFilterIndex      = 1;
 ofn.lpstrFile         = FileName;
 ofn.nMaxFile          = MAXFILENAME;
 ofn.lpstrInitialDir   = NULL;
 ofn.lpstrFileTitle    = FileTitle;
 ofn.nMaxFileTitle     = MAXFILENAME;
 ofn.lpstrTitle        = NULL;
 ofn.lpstrDefExt       = "TXT";
 ofn.Flags             = OFN_NOVALIDATE;


 ShowWindow(hMainWnd, nCmdShow);
 UpdateWindow(hMainWnd);

 /* window control loop */

 while(GetMessage(&msg,NULL,NULL,NULL))
   {
    TranslateMessage(&msg);
    DispatchMessage(&msg);
   }
 return (msg.wParam);
} /* end WinMain */

long FAR PASCAL MainWndProc(HWND hWindow,UINT message,
  WPARAM wParam,LPARAM lParam)
{int i;
 UINT idTimer;
 HDC hDC;
 PAINTSTRUCT ps;
 int KeyChar;
 int ThisPacket;
 int ThisNAKcount;
 long ThisByteCount;
 HMENU hMenu;
 static FARPROC lpProcAbout;
 static FARPROC lpProcAccept;
 char Temp[40];  /* !!! */

 hMainWnd = hWindow;
 switch (message)
    {case WM_USER:
         DialogBox(hInstance,"AcceptBox",hMainWnd,lpProcAccept);
         InvalidateRect(hMainWnd,NULL,TRUE);  /* force paint */
         break;

     case WM_COMMAND:
         switch(wParam)
           {case MSG_ABOUT:
              DialogBox(hInstance,"AboutBox",hMainWnd,lpProcAbout);
              break;

            case MSG_BREAK:
              DisableBreakButton();
              MessageBeep(0);
              switch(XferState)
                {case 'A':
                 case 'X':
                 case 'Y':
                   InfoStatus("Aborting");
                   ShowWindow(hInfoWnd,SW_SHOW);
                   SendMessage(hInfoWnd,WM_USER,0,0L);
                   xyAbort();
                   /* window will hide after 2 seconds */
                   HideTime = GetTickCount() + 2000L;
                   break;
                 case 'I':
                 case 'H':
                 case 'S':
                   MessageBeep(0);
                   break;
                 default:
                   break;
                }
              XferState = ' ';
              SetTitle();
              break;

            case MSG_LOAD:
              if(!ExpectNoXfer()) break;
              if(!ExpectOffLine()) break;
              LoadConfig();
              break;

            case MSG_SAVE:
              if(!ExpectNoXfer()) break;
              if(!ExpectOffLine()) break;
              SaveConfig();
              SetTitle();
              break;

            case MSG_ONLINE:
              if(!ExpectOffLine()) break;
              if(FatalFlag) ErrorMessage("Fatal Error");
              else
                {/* try to go on-line */

/*
** You must call PostMainHandle() before attemping to go online.
** This is required only for the Shareware version of PCL4W.
*/
                 PostMainHandle(hMainWnd);

                 GoOnLine();
                 if(!OnLineFlag) break;
                 ThePort = GetPort();
                }
              break;
            case MSG_SCRIPT_DEBUG:
              GoScript(TRUE);
              break;
            case MSG_SCRIPT:
              GoScript(FALSE);
              break;
            case MSG_HANGUP_MODEM:
              if(!ExpectOnLine()) break;
              if(FatalFlag) ErrorMessage("Fatal Error");
              else
                {
#if AT_COMMAND_SET
                 EnableBreakButton();
                 /* start 1st HANGHUP_MODEM function */
                 XferState = 'H';
                 Mstate = 1;
                 SetTitle();
#else
                 SayNoAT();
#endif
                }
              break;
            case MSG_INIT_MODEM:
              if(!ExpectOnLine()) break;
              if(FatalFlag) ErrorMessage("Fatal Error");
              else
                {
#if AT_COMMAND_SET
                 EnableBreakButton();
                 /* start 1st INIT_MODEM function */
                 XferState = 'I';
                 Mstate = 1;
                 SetTitle();
#else
                 SayNoAT();
#endif
                }
              break;
            case MSG_OFFLINE:
              if(!ExpectNoXfer()) break;
              GoOffLine();
              break;

            case MSG_EXIT:
              if(!ExpectNoXfer()) break;
              GoOffLine();
              KillTimer(hMainWnd,idTimer);
              TermDone();
              PostQuitMessage(0);
              break;

            case MSG_1200:
              SetBaud(Baud1200);
              break;

            case MSG_2400:
              SetBaud(Baud2400);
              break;

            case MSG_4800:
              SetBaud(Baud4800);
              break;

            case MSG_9600:
              SetBaud(Baud9600);
              break;

            case MSG_19200:
              SetBaud(Baud19200);
              break;

            case MSG_38400:
              SetBaud(Baud38400);
              break;

            case MSG_57600:
              SetBaud(Baud57600);
              break;

            case MSG_115200:
              SetBaud(Baud115200);
              break;

            case MSG_COM1:
            case MSG_COM2:
            case MSG_COM3:
            case MSG_COM4:
            case MSG_COM5:
            case MSG_COM6:
            case MSG_COM7:
            case MSG_COM8:
            case MSG_COM9:
            case MSG_COM10:
            case MSG_COM11:
            case MSG_COM12:
            case MSG_COM13:
            case MSG_COM14:
            case MSG_COM15:
            case MSG_COM16:
              SetPort(COM1+wParam-MSG_COM1);
              break;

            case MSG_DEBUG:

              sprintf(Temp,"# Tx Interrupts = %d\n", SioInfo('T') );
              DisplayLine(Temp);
              sprintf(Temp,"# Rx Interrupts = %d\n", SioInfo('R') );
              DisplayLine(Temp);
              sprintf(Temp,"SioDebug(%d) = %x\n",ThePort,SioDebug(ThePort) );
              DisplayLine(Temp);
              for(i=0;i<7;i++)
                {sprintf(Temp,"UART%d = %x\n", i,SioRead(ThePort,i) );
                 DisplayLine(Temp);
                }
              InvalidateRect(hMainWnd,NULL,TRUE);
              break;

            case MSG_NONE:
              SetParity(NoParity);
              break;

            case MSG_EVEN:
              SetParity(EvenParity);
              break;

            case MSG_ODD:
              SetParity(OddParity);
              break;

            case MSG_1_SB:
              SetStopBits(OneStopBit);
              break;

            case MSG_2_SB:
              SetStopBits(TwoStopBits);
              break;

            case MSG_7_DB:
              SetWordLength(WordLength7);
              break;

            case MSG_8_DB:
              SetWordLength(WordLength8);
              break;

            case MSG_ASCII_TX:
              if(!ExpectNoXfer()) break;
              if(!ExpectOnLine()) break;
              ofn.lpstrFilter = FilterAll;
              if(GetOpenFileName ((LPOPENFILENAME)&ofn))
                {/* initialize xyDriver */
                 ascInit(ThePort);
                 ascStartTx(FileName,5,ESC,TRUE);
                 WindowAS(FileName);
                 XferState = 'A';
                 SetTitle();
                 SetWindowText(hInfoWnd,"ASCII Send");
                 break;
                }
              break;

            case MSG_ASCII_RX:
              if(!ExpectNoXfer()) break;
              if(!ExpectOnLine()) break;
              ofn.lpstrFilter = FilterAll;
              if(GetOpenFileName ((LPOPENFILENAME)&ofn))
                {/* initialize xyDriver */
                 ascInit(ThePort);
                 WindowAS(FileName);
                 ascStartRx(FileName,(2<<(3+RXBUFFERCODE)),ESC,TRUE);
                 XferState = 'A';
                 SetTitle();
                 SetWindowText(hInfoWnd,"ASCII Receive");
                 break;
                }
              break;

            case MSG_XMODEM_TX:
              if(!ExpectNoXfer()) break;
              if(!ExpectOnLine()) break;
              ofn.lpstrFilter = FilterAll;
              if(GetOpenFileName((LPOPENFILENAME)&ofn))
                {xyInit(ThePort);
                 xyStartTx(FileName,FALSE,FALSE);
                 WindowXY(FileName);
                 LastPacket = -1;
                 LastNAKcount = 0;
                 SetTitle();
                 SetWindowText(hInfoWnd,"XMODEM Send");
                 XferState = 'X';
                }
              break;

            case MSG_XMODEM_RX:
              if(!ExpectNoXfer()) break;
              if(!ExpectOnLine()) break;
              ofn.lpstrFilter = FilterAll;
              if(GetOpenFileName((LPOPENFILENAME)&ofn))
                {xyInit(ThePort);
                 NCGchar = NAK;
                 xyStartRx(FileName,NCGchar,FALSE);
                 WindowXY(FileName);
                 LastPacket = -1;
                 LastNAKcount = 0;
                 SetTitle();
                 SetWindowText(hInfoWnd,"XMODEM Receive");
                 XferState = 'X';
                }
              break;

            case MSG_YMODEM_TX:
              if(!ExpectNoXfer()) break;
              if(!ExpectOnLine()) break;
              ofn.lpstrFilter = FilterAll;
              if(GetOpenFileName((LPOPENFILENAME)&ofn))
                {xyInit(ThePort);
                 xyStartTx(FileName,TRUE,TRUE);
                 WindowXY(FileName);
                 LastPacket = -1;
                 LastNAKcount = 0;
                 SetTitle();
                 SetWindowText(hInfoWnd,"YMODEM Send");
                 XferState = 'Y';
                 break;
                }
              break;

            case MSG_YMODEM_RX:
              if(!ExpectNoXfer()) break;
              if(!ExpectOnLine()) break;
              /* don't need filename for YMODEM receive */
              xyInit(ThePort);
              NCGchar = 'C';
              xyStartRx("",NCGchar,TRUE);
              WindowXY("");
              LastPacket = -1;
              LastNAKcount = 0;
              SetTitle();
              SetWindowText(hInfoWnd,"YMODEM Receive");
              XferState = 'Y';
              break;

            default:
              return (DefWindowProc(hMainWnd, message, wParam, lParam));
           }
         break;

    case WM_CREATE:

       /* create popup Info window */
       GetWindowRect(hMainWnd,&MainRect);
       hInfoWnd = CreateWindow(
          "InfoWClass", "Info",
          WS_POPUP | WS_BORDER | WS_CAPTION | WS_THICKFRAME,
          MainRect.left + (MainRect.right-MainRect.left-POPWIDTH)/2,
          MainRect.top + (MainRect.bottom-MainRect.top-POPHEIGHT)/2,
          POPWIDTH, POPHEIGHT,
          hMainWnd,  NULL,
          hInstance, NULL);
      UpdateWindow(hInfoWnd);
      /* check "OFFLINE" menu item */
      hMenu = GetMenu(hMainWnd);
      CheckMenuItem(hMenu,MSG_OFFLINE,MF_BYCOMMAND | MF_CHECKED);
      /* create AboutDlgProc() thunk */
      lpProcAbout = MakeProcInstance(AboutDlgProc, hInstance);
      /* create AcceptDlgProc() thunk */
      lpProcAccept = MakeProcInstance(AcceptDlgProc, hInstance);
      /* grey the BREAK button */
      EnableMenuItem(hMenu,MSG_BREAK,MF_BYCOMMAND|MF_DISABLED|MF_GRAYED);
      /* init configuration */
      InitConfig();
      LoadConfig();
      /* initialize paint module */
      InitPaint();
      /* start timer */
      idTimer = SetTimer(hMainWnd,1,125,NULL);
      if(idTimer==0)
         {ErrorMessage("No timers remaining !");
          FatalFlag = TRUE;
         }


#if 1
 /*** Standard Configuration (DEFAULT) ***/
 strncpy(Configuration,"Standard PC Config.",20);
 SioIRQ(COM1,IRQ4);
 SioIRQ(COM2,IRQ3);
 SioIRQ(COM3,IRQ4);
 SioIRQ(COM4,IRQ3);
 SioUART(COM1,0x3F8);
 SioUART(COM2,0x2F8);
 SioUART(COM3,0x3E8);
 SioUART(COM4,0x2E8);
#endif

#if 0
 /*** Custom Configuration: DigiBoard PC/8 ***/
 strncpy(Configuration,"DigiBoard PC/8",20);
 /* use 0x140 for odd IRQs & 0x141 for even IRQs */
 SioPorts(8,COM1,0x140,DIGIBOARD);
 for(i=COM1;i<=COM8;i++)
   {/* set DigiBoard UART addresses */
    ErrorCheck( SioUART(i,0x100+8*i), "SioUART" );
    /* set DigiBoard for IRQ5 */
    ErrorCheck( SioIRQ(i,IRQ5), "SioIRQ" );
   }
#endif

#if 0
 /*** Custom Configuration: BOCA 16 port dumbcard ***/
 strncpy(Configuration,"BOCA Board BB2016",20);
 /* use base port + 7 */
 SioPorts(16,COM1,0x100+7,BOCABOARD);
 for(i=COM1;i<=COM16;i++)
   {/* set BOCA Board UART addresses */
    ErrorCheck( SioUART(i,0x100+8*i), "SioUART" );
    /* set BOCABoard for IRQ15 */
    ErrorCheck( SioIRQ(i,IRQ15), "SioIRQ" );
   }
#endif

      TermInit();

      break;

    case WM_CHAR:
      if(OnLineFlag&&(XferState==' '))
        {KeyChar = wParam;
         PutChar(ThePort, (char)KeyChar);
        }
      break;

    case WM_TIMER:
      /* flush display buffer */
      FlushDisplay();
      /* time to hide the popup window ? */
      if(HideTime > 0L)
        {if(GetTickCount() > HideTime)
          {HideTime = 0L;
           MessageBeep(0);
           ShowWindow(hInfoWnd,SW_HIDE);
           DisableBreakButton();
          }
        }
      /* fatal error ? */
      if(FatalFlag)
        {xyAbort();
         XferState = ' ';
         SetTitle();
         break;
        }
      /* xfer in progess ? */
      switch(XferState)
        {case 'A':
           if(ascDriver())
             {/* asDriver is idle */
              MessageBeep(0);
              XferState = ' ';
              SetTitle();
              InfoStatus("Completed");
              SendMessage(hInfoWnd,WM_USER,0,0L);
              HideTime = GetTickCount() + 2000L;
             }
           else
             {/* update # bytes */
              ThisByteCount = ascGetCharCount();
              ThisPacket = (int) (ThisByteCount / 100L);
              if(ThisPacket!=LastPacket)
                {LastPacket = ThisPacket;
                 InfoBytes((int)ThisByteCount);
                 SendMessage(hInfoWnd,WM_USER,0,0L);
                }
             }
           break;
         case 'X':
         case 'Y':
           if(xyDriver()==IDLE)
             {/* xyDriver is idle */
              MessageBeep(0);
              XferState = ' ';
              SetTitle();
              InfoStatus("Completed");
              SendMessage(hInfoWnd,WM_USER,0,0L);
              HideTime = GetTickCount() + 3000L;
             }
           else
             {/* update packet number */
              ThisPacket = xyGetPacket();
              if(ThisPacket!=LastPacket)
                {LastPacket = ThisPacket;
                 InfoPacket(ThisPacket);
                 SendMessage(hInfoWnd,WM_USER,0,0L);
                }
              /* update error count */
              ThisNAKcount = xyGetNAKs();
              if(ThisNAKcount != LastNAKcount)
                {LastNAKcount = ThisNAKcount;
                 InfoErrors(ThisNAKcount);
                 SendMessage(hInfoWnd,WM_USER,0,0L);
                }
             }
           break;
         case 'S':  /* WSI functions (script interpreter) */
#if SCRIPTS
           if(wsiDriver()!=IDLE) break;
           XferState = ' ';
           SetTitle();
           DisableBreakButton();
#else
           SayNoScripts();
#endif
           break;
         case 'I':  /* INIT_MODEM functions */
#if AT_COMMAND_SET
           if(ModemDriver()!=IDLE) break;
           /* run next INIT MODEM function */
           switch(Mstate)
             {case 1:
                 ModemSendTo(ThePort,0.200,"!!AT E1 S7=60 S11=60 V1 X1 Q0!!");
                 break;
              case 2:
                 DisplayLine("Waiting for OK...");
                 ModemWaitFor(ThePort,10.0,FALSE,"OK");
                 break;
              case 3:
                 /* get result of above WaitFor */
                 if(ModemResult()) DisplayLine("\n");
                 else DisplayLine("OK not found\n");
              default:
                 /* no more functions to run */
                 XferState = ' ';
                 SetTitle();
                 DisableBreakButton();
                 break;
             }
           Mstate++;
#else
           SayNoAT();
#endif
           break;

         case 'H':  /* HANGUP_MODEM functions */
#if AT_COMMAND_SET
           if(ModemDriver()!=IDLE) break;
           /* run next HANGUP MODEM function */
           switch(Mstate)
             {case 1:
                 ModemQuiet(ThePort,10.0);
                 break;
              case 2:
                 ModemHangup(ThePort);
                 break;
              case 3:
                 ModemSendTo(ThePort,0.200,"!ATH0!");
                 break;
              default:
                 /* no more functions to run */
                 XferState = ' ';
                 SetTitle();
                 DisableBreakButton();
                 break;
             }
           Mstate++;
#else
           SayNoAT();
#endif
           break;

         default:
           /* NOT doing xfer */
           if(!OnLineFlag) break;
           /* fetch line & echo it */
           PortEcho(ThePort,750);
           break;
        } /* end switch(Xfer) */
      break;

    case WM_SETFOCUS:
      /* create client area caret */
      CreateCaret(hMainWnd,NULL,3,10);
      SetCaretPos(GetXposition(),GetYposition());
      ShowCaret(hMainWnd);
      ShowCaret(hMainWnd);
      break;

    case WM_KILLFOCUS:
      DestroyCaret();
      break;

    case WM_SIZE:
      WinWidth = LOWORD(lParam);
      WinHeight = HIWORD(lParam);
      break;

    case WM_PAINT:
      HideCaret(hMainWnd);
      hDC = BeginPaint(hMainWnd, &ps);
      SetMapMode(hDC,MM_ANISOTROPIC);
      SelectObject(hDC, GetStockObject(OEM_FIXED_FONT) );
      PaintMain(hDC,&ps);
      EndPaint(hMainWnd,&ps);
      SetCaretPos(GetXposition(),GetYposition());
      ShowCaret(hMainWnd);
      break;

    case WM_DESTROY:
      GoOffLine();
      if(idTimer) KillTimer(hMainWnd,idTimer);
      PostQuitMessage(0);
      break;

    default:
      return (DefWindowProc(hMainWnd, message, wParam, lParam));
   }
 return (NULL);
} /* end MainWndProc */

void ErrorCheck(int Code,char *Text)
{/* trap PCL error codes */
 if(Code<0)
     {SioError(Code,Text);
      SioDone(GetPort());
      FatalFlag = TRUE;
     }
} /* end ErrorCheck */
