/*
 * Program to run an interesting self-reproducing cellular automata.
 * This can run either under X, or on terminals using ANSI escape sequences.
 *
 * This automata was designed by Christopher G. Langton at the University
 * of Michigan, and is described in his article "Self-Reproduction in
 * Cellular Automata", published in the book "Cellular Automata" in 1984.
 *
 * As described in the above article, this automata is "realistic" and
 * "non-trivial", in the following ways:
 *
 * 	The reproduction is not simply due to the particular rules used in
 *	the automata.  Many automata can reproduce patterns, but such
 *	behavior is a simple mathematical property of the rules, and is
 *	not based on the starting pattern.  In this automata, the correct
 *	starting pattern is essential, otherwise nothing interesting happens.
 *	True, the rules are VERY convenient for this pattern, but the rules
 *	themselves do not make the reproduction occur.  It is the rules
 *	acting on this special pattern which enables reproduction (just
 *	like our own universe!)
 *
 *	Like real organisms, this pattern contains "DNA" which directs the
 *	reproduction.  The "DNA" is processed in two ways, by replicating
 *	it and by transcribing it.  This is necessary for "true" reproduction.
 *
 *	Like bacteria, the reproduction doesn't simply create another copy
 *	of the starting pattern.  It creates two identical copies of
 *	itself, thus both the parent and the child can continue to make
 *	new copies of themselves.
 *
 * There are two questions I have about this automata:
 *
 * The first one is whether larger patterns can be made with these rules
 * which can also reproduce, and whether or not there are any interesting
 * "mutations" of the DNA in such patterns to make other interesting things.
 * It seems to me that it should be possible to "double" the size of the
 * pattern, for example.
 *
 * The second one is whether a simple change to the rules will allow the
 * pattern to run as it does now, EXCEPT that when the pattern "dies"
 * because of overcrowding, it would totally disappear, so that another
 * copy of the pattern could reuse its space.  This would me more interesting.
 *
 * I hope you enjoy watching this run!  If you can answer either of the
 * above questions, please let me know.
 *	- David I. Bell -
 *	dbell@pdact.pd.necisa.oz.au	
 */

#include <stdio.h>


/*
 * Define X to compile for X, otherwise will compile for ANSI terminals.
 */
#ifdef	X
#define	ROWS	600
#define	COLS	800
#else
#define	ROWS	24
#define	COLS	79
#endif


#define	MIN(a,b) 	(((a) < (b)) ? (a) : (b))
#define	MAX(a,b) 	(((a) > (b)) ? (a) : (b))
#define	STATES		8
#define	STATESIZE	(STATES * STATES * STATES * STATES * STATES)
#define	STATEINDEX(a,b,c,d,e)	\
	(((((a) * STATES + (b)) * STATES + (c)) * STATES + (d)) * STATES + (e))


/*
 * The rules for the automata:  The automata is run on a square grid, with
 * a neighborhood of the nearest four orthagonal cells (left, right, up,
 * and down).  Each cell can be in one of 8 states, numbered from 0 to 7.
 * State 0 is the "empty" or "off" state (and is the same as blanks).
 * In the following table, the entries are in the following format:
 *	"cwxyz n".
 * Here c is the old state of the current cell, w, x, y, and z are the
 * states of the four orthagonal neighbors of the current cell in clockwise
 * order, and n is the new state of the current cell.  Each rule is used
 * for all four starting points of the four orthagonal cells.  For example,
 * the rule "01725 5" means that a cell in state 0, surrounded by cells
 * in states 1, 7, 2, and 5, in that clockwise order, changes to state 5.
 * For all rules not given in this table, the cell changes to state 0.
 */
static	char *rules[] = {
	"00001 2",
	"00006 3",
	"00007 1",
	"00011 2",
	"00012 2",
	"00013 2",
	"00021 2",
	"00026 2",
	"00027 2",
	"00052 5",
	"00062 2",
	"00072 2",
	"00102 2",
	"00212 5",
	"00232 2",
	"00522 2",
	"01232 1",
	"01242 1",
	"01252 5",
	"01262 1",
	"01272 1",
	"01275 1",
	"01422 1",
	"01432 1",
	"01442 1",
	"01472 1",
	"01625 1",
	"01722 1",
	"01725 5",
	"01752 1",
	"01762 1",
	"01772 1",
	"02527 1",
	"10001 1",
	"10006 1",
	"10007 7",
	"10011 1",
	"10012 1",
	"10021 1",
	"10024 4",
	"10027 7",
	"10051 1",
	"10101 1",
	"10111 1",
	"10124 4",
	"10127 7",
	"10202 6",
	"10212 1",
	"10221 1",
	"10224 4",
	"10226 3",
	"10227 7",
	"10232 7",
	"10242 4",
	"10262 6",
	"10264 4",
	"10267 7",
	"10272 7",
	"10542 7",
	"11112 1",
	"11122 1",
	"11124 4",
	"11125 1",
	"11126 1",
	"11127 7",
	"11152 2",
	"11212 1",
	"11222 1",
	"11224 4",
	"11225 1",
	"11227 7",
	"11232 1",
	"11242 4",
	"11262 1",
	"11272 7",
	"11322 1",
	"12224 4",
	"12227 7",
	"12243 4",
	"12254 7",
	"12324 4",
	"12327 7",
	"12425 5",
	"12426 7",
	"12527 5",
	"20001 2",
	"20002 2",
	"20004 2",
	"20007 1",
	"20012 2",
	"20015 2",
	"20021 2",
	"20022 2",
	"20023 2",
	"20024 2",
	"20026 2",
	"20027 2",
	"20032 6",
	"20042 3",
	"20051 7",
	"20052 2",
	"20057 5",
	"20072 2",
	"20102 2",
	"20112 2",
	"20122 2",
	"20142 2",
	"20172 2",
	"20202 2",
	"20203 2",
	"20205 2",
	"20207 3",
	"20212 2",
	"20215 2",
	"20221 2",
	"20222 2",
	"20227 2",
	"20232 1",
	"20242 2",
	"20245 2",
	"20255 2",
	"20262 2",
	"20272 2",
	"20312 2",
	"20321 6",
	"20322 6",
	"20342 2",
	"20422 2",
	"20512 2",
	"20521 2",
	"20522 2",
	"20552 1",
	"20572 5",
	"20622 2",
	"20672 2",
	"20712 2",
	"20722 2",
	"20742 2",
	"20772 2",
	"21122 2",
	"21126 1",
	"21222 2",
	"21224 2",
	"21226 2",
	"21227 2",
	"21422 2",
	"21522 2",
	"21622 2",
	"21722 2",
	"22227 2",
	"22244 2",
	"22246 2",
	"22276 2",
	"22277 2",
	"30001 3",
	"30002 2",
	"30004 1",
	"30007 6",
	"30012 3",
	"30042 1",
	"30062 2",
	"30102 1",
	"30251 1",
	"40222 1",
	"40232 6",
	"40322 1",
	"50002 2",
	"50021 5",
	"50022 5",
	"50023 2",
	"50027 2",
	"50202 2",
	"50212 2",
	"50215 2",
	"50224 4",
	"50272 2",
	"51212 2",
	"51242 2",
	"51272 2",
	"60001 1",
	"60002 1",
	"61212 5",
	"61213 1",
	"61222 5",
	"70007 7",
	"70222 1",
	"70225 1",
	"70232 1",
	"70252 5",
	NULL
};


/*
 * The magic starting pattern which reproduces itself.
 * The cells in state 2 are the "skin" or "sheath" of the pattern.
 * The inner cells in states 0, 1, 4, and 7 are the "DNA" of the
 * pattern, and directs the construction of the replicated pattern.
 * The "DNA" cycles around the loop, and is copied and sent to the
 * end of the "arm", where it directs new growth.
 */
static	char	*init[] = {
	" 22222222",
	"2170140142",
	"2022222202",
	"272    212",
	"212    212",
	"202    212",
	"272    212",
	"21222222122222",
	"207107107111112",
	" 2222222222222",
	NULL
};


typedef	struct {
	char	pre[COLS+1];
	char	cells[ROWS][COLS+1];
	char	post[COLS+1];
	int	minrow;
	int	maxrow;
	int	mincol;
	int	maxcol;
} DATA;


static	int	width;
static	int	height;
static	DATA	b1;
static	DATA	b2;
static	DATA	*curb;
static	char	states[STATESIZE];


static	void	x_init();
static	void	x_flush();
static	void	x_draw_pixel();
static	void	printgen();
static	void	rungen();
static	void	initstates();


main(argc, argv)
	char	**argv;
{
	int	row, col;

	char	*rp;
	char	**rpp;

	x_init(argc, argv);
	initstates();
	memset(&b1, '0', sizeof(b1));
	memset(&b2, '0', sizeof(b2));
	row = height / 2;
	b1.minrow = row;
	b1.maxrow = row;
	b1.mincol = width / 2;
	b1.maxcol = width / 2;

	/*
	 * Set the initial pattern.
	 */
	rpp = init;
	while (*rpp) {
		rp = *rpp++;
		col = width / 2;
		while (*rp) {
			if (*rp != ' ') {
				b1.cells[row][col] = *rp;
				if (b1.maxcol < col)
					b1.maxcol = col;
			}
			col++;
			rp++;
		}
		b1.maxrow = row++;
	}

	b2.minrow = b1.minrow;
	b2.maxrow = b1.maxrow;
	b2.mincol = b1.mincol;
	b2.maxcol = b1.maxcol;

	curb = &b1;
	x_flush();

	for (;;) {
		printgen(&b2, &b1);
		rungen(&b1, &b2);
		printgen(&b1, &b2);
		rungen(&b2, &b1);
	}
}


/*
 * Print a configuration by comparing the old data with the new data.
 */
static void
printgen(old, new)
	DATA	*old, *new;
{
	register char	*ocp;
	register char	*ncp;
	int	row, col;
	int	minrow, maxrow, mincol, maxcol;

	minrow = MIN(old->minrow, new->minrow);
	maxrow = MAX(old->maxrow, new->maxrow);
	mincol = MIN(old->mincol, new->mincol);
	maxcol = MAX(old->maxcol, new->maxcol);

	for (row = minrow; row <= maxrow; row++) {
		ocp = &old->cells[row][mincol];
		ncp = &new->cells[row][mincol];
		for (col = mincol; col <= maxcol; col++) {
			if (*ocp != *ncp)
				x_draw_pixel(col, row, *ncp - '0');
			ocp++;
			ncp++;
		}
	}
	curb = new;
	x_flush();
}


/*
 * Run one generation of the automata.
 */
static void
rungen(old, new)
	DATA	*old, *new;
{
	register char	*ocp;
	char	*ncp;
	int	c0, c1, c2, c3, c4;
	int	row, col, ch;
	int	minrow, maxrow, mincol, maxcol;

	minrow = old->minrow - 1;
	maxrow = old->maxrow + 1;
	mincol = old->mincol - 1;
	maxcol = old->maxcol + 1;
	minrow = MAX(minrow, 0);
	maxrow = MIN(maxrow, height - 1);
	mincol = MAX(mincol, 0);
	maxcol = MIN(maxcol, width - 1);

	for (row = minrow; row <= maxrow; row++) {
		ocp = &old->cells[row][mincol];
		ncp = &new->cells[row][mincol];
		for (col = mincol; col <= maxcol; col++) {
			c0 = ocp[0] - '0';
			c1 = ocp[-1] - '0';
			c2 = ocp[-(COLS+1)] - '0';
			c3 = ocp[1] - '0';
			c4 = ocp[COLS+1] - '0';
			ch = states[STATEINDEX(c0, c1, c2, c3, c4)];
			if (ch != *ocp++) {
				if (new->minrow > row)
					new->minrow = row;
				if (new->maxrow < row)
					new->maxrow = row;
				if (new->mincol > col)
					new->mincol = col;
				if (new->maxcol < col)
					new->maxcol = col;
			}
			*ncp++ = ch;
		}
	}
}


/*
 * Initialize the state table.
 */
static void
initstates()
{
	char	**rpp;
	char	*rp;
	int	c0, c1, c2, c3, c4, dest;

	memset(states, '0', STATESIZE);
	for (rpp = rules; *rpp; rpp++) {
		rp = *rpp;
		c0 = *rp++ - '0';
		c1 = *rp++ - '0';
		c2 = *rp++ - '0';
		c3 = *rp++ - '0';
		c4 = *rp++ - '0';
		dest = rp[1];
		states[STATEINDEX(c0, c1, c2, c3, c4)] = dest;
		states[STATEINDEX(c0, c2, c3, c4, c1)] = dest;
		states[STATEINDEX(c0, c3, c4, c1, c2)] = dest;
		states[STATEINDEX(c0, c4, c1, c2, c3)] = dest;
	}
}


#ifndef	X

/*
 * Code for doing output to an ANSI terminal.
 */
static	int	oldrow;
static	int	oldcol;


static void
x_init()
{
	fputs("\033[H\033[J", stdout);
	fflush(stdout);
	height = ROWS;
	width = COLS;
}


static void
x_draw_pixel(col, row, value)
{
	int	ch;

	if ((col != oldcol) || (row != oldrow)) {
		putchar('\033');
		putchar('[');
		if (row >= 10) {
			ch = (row / 10) + '0';
			putchar(ch);
			ch = (row % 10) + '0';
		} else
			ch = row + '0';
		putchar(ch);
		putchar(';');
		col++;
		if (col >= 10) {
			ch = (col / 10) + '0';
			putchar(ch);
			ch = (col % 10) + '0';
		} else
			ch = col + '0';
		col--;
		putchar(ch);
		putchar('H');
	}
	if (value)
		ch = value + '0';
	else
		ch = ' ';
	putchar(ch);
	oldrow = row;
	oldcol = col + 1;
}


static void
x_flush()
{
	fputs("\033[H", stdout);
	fflush(stdout);
	oldrow = 0;
	oldcol = 0;
}


#else


/*
 * Code for doing output to an X display.
 */
#include <X11/Xlib.h>

#define	SLOP	50	/* pixels to reserve for window managers */

static	Display	*display;
static	Screen	*screen;
static	Window	window;
static	GC	gc;
static	int	black;
static	int	white;
static	int	oldvalue;
static	int	scale = 4;

extern	char	*getenv();


static void
x_init(argc, argv)
	char	**argv;
{
	char		*name;
	int		cells;
	XGCValues	gcvalues;

	if (argc > 1)
		scale = atoi(argv[1]);
	if ((scale <= 0) || (scale > 10)) {
		fprintf(stderr, "Bad scale value\n");
		exit(1);
	}

	name = getenv("DISPLAY");
	if (name == NULL) {
		fprintf(stderr, "DISPLAY not set\n");
		exit(1);
	}

	display = XOpenDisplay(name);
	if (display == NULL) {
		fprintf(stderr, "Cannot open display \"%s\"\n", name);
		exit(1);
	}

	screen = ScreenOfDisplay(display, DefaultScreen(display));

	if (CellsOfScreen(screen) < STATES) {
		fprintf(stderr, "This program needs %d colors\n", STATES);
		exit(1);
	}

	height = HeightOfScreen(screen) - SLOP;
	if (height > ROWS)
		height = ROWS;
	width = WidthOfScreen(screen) - SLOP;
	if (width > COLS)
		width = COLS;
	black = BlackPixelOfScreen(screen);
	white = WhitePixelOfScreen(screen);

	window = XCreateSimpleWindow(display, RootWindowOfScreen(screen),
		0, 0, width, height, 1, white, black);

	gc = XCreateGC(display, window, 0, &gcvalues);

	XSelectInput(display, window, ExposureMask);

	XMapWindow(display, window);

	width /= scale;
	height /= scale;

	oldvalue = -1;
}


static void
x_draw_pixel(col, row, value)
{
	XPoint	*pp;
	XPoint	points[9];
	int	ri, ci;

	/*
	 * Map the states 0 and 2 to black and white specially.
	 * The other states are not so important, and any color will do.
	 */
	if (value == 0)
		value = black;
	else if (value == black)
		value = 0;
	else if (value == 2)
		value = white;
	else if (value == white)
		value = 2;

	if (value != oldvalue) {
		XSetForeground(display, gc, value);
		oldvalue = value;
	}

	if (scale == 1) {
		XDrawPoint(display, window, gc, col, row);
	} else if ((scale == 2) || (scale == 3)) {
		pp = points;
		for (ri = 0; ri < scale; ri++) {
			for (ci = 0; ci < scale; ci++) {
				pp->x = col * scale + ci;
				pp->y = row * scale + ri;
				pp++;
			}
		}
		XDrawPoints(display, window, gc, points, scale*scale,
			CoordModeOrigin);
	} else {
		XFillRectangle(display, window, gc, col * scale, row * scale,
			scale, scale);
	}
}


static void
x_flush()
{
	XEvent	event;
	int	value;
	int	row;
	int	col;

	if (!XCheckMaskEvent(display, ExposureMask, &event))
		return;

	/*
	 * Draw the whole pattern for an exposure.
	 */
	for (row = curb->minrow; row <= curb->maxrow; row++) {
		for (col = curb->mincol; col <= curb->maxcol; col++) {
			value = curb->cells[row][col] - '0';
			if (value)
				x_draw_pixel(col, row, value);
		}
	}
}

#endif

/* END CODE */
