program  Cxf; uses cx;

{
   Copyright (c) 1990-1992 Eugene Nelson

   This program demonstrates the suggested usage of the Cx Data Compression
   Library.  It will compress and decompress a file of any size, containing
   any type of data.  The compressed data is stored with 16 bit CRC's to
   help in detecting errors.  It stores the file as blocks of data which
   have the following form:              

      BLOCK
      ----------------------------------------------------
      2 bytes - size of uncompressed block (USIZE)
      2 bytes - size of compressed block   (CSIZE)
      2 bytes - 16 bit CRC                 (CRC)
      CSIZE   - data                       (DATA)
      ----------------------------------------------------

   If a block cannot be compressed, the original data is stored, and
   USIZE and CSIZE will be the same.  The CRC is computed on the
   compressed data.

   2 bytes of 0 are stored at the end of the compressed file to indicate
   the end of file.
}


{-------------------------------------------------------------------------}
procedure QUIT(s: string);
begin
   writeln('');
   writeln(s);
   Halt;
end;

{-------------------------------------------------------------------------}
procedure usage;
begin
   writeln('usage: cxf 1|2|3|t|d infile outfile');
   writeln('   1|2|3   - compress infile to outfile using method 1,2 or 3');
   writeln('   d       - decompress infile to outfile');
   writeln('   t       - test integrity of infile');
   writeln('');
   writeln('   Examples:');
   writeln('      cxf 1 cxf.exe x.x');
   writeln('      cxf t x.x');
   writeln('      cxf d x.x y.y');
   Halt;
end;

{-------------------------------------------------------------------------}
procedure file_compress(dst, src: string; method, size: CXINT);
var
   ifile, ofile: file;
   ifilesize, ofilesize: longint;
   ibuff, obuff, tbuff: pointer;
   j, k, crc: CXINT;

begin
   Assign(ifile, src);
   Reset(ifile, 1);
   ifilesize:= FileSize(ifile);

   Assign(ofile, dst);
   Rewrite(ofile, 1);
   ofilesize:= 0;

   GetMem(ibuff, size);
   GetMem(obuff, size+CX_SLOP);
   GetMem(tbuff, CX_C_MAXTEMP);

   repeat
      write('.');

      BlockRead(ifile, ibuff^, size, j);
      BlockWrite(ofile, j, CXINTSIZE);

      if j <> 0
      then begin
         k:= CX_COMPRESS(method, obuff^, size, ibuff^, j, tbuff^, CX_C_MAXTEMP);
         case k of
            CX_ERR_METHOD:
               QUIT('unknown or unsupported method, evaluation version only allows CX_MEHTOD1');

            CX_ERR_BUFFSIZE:
               QUIT('invallid input or output buffer size');

            CX_ERR_TEMPSIZE:
               QUIT('invalid temporary buffer size');
         end;

         BlockWrite(ofile, k, CXINTSIZE);

         if (k = j)     {if block could not be compressed}
         then begin
            crc:= CX_CRC(ibuff^, k);
            BlockWrite(ofile, crc, CXINTSIZE);
            BlockWrite(ofile, ibuff^, j)
         end
         else begin
            crc:= CX_CRC(obuff^, k);
            BlockWrite(ofile, crc, CXINTSIZE);
            BlockWrite(ofile, obuff^, k);
         end;

         ofilesize:= ofilesize + CXINTSIZE + CXINTSIZE + k;
      end;
   until j = 0;

   FreeMem(ibuff, size);
   FreeMem(obuff, size+CX_SLOP);
   FreeMem(tbuff, CX_C_MAXTEMP);

   ifilesize:= FileSize(ifile);
   ofilesize:= FileSize(ofile);
   Close(ifile);
   Close(ofile);

   writeln('');
   write('In Size:    ');  writeln(ifilesize);
   write('Out Size:   ');  writeln(ofilesize);
   if ifilesize <> 0
   then begin
      write('Percent:    '); writeln((ofilesize*100) div ifilesize);
   end
end;

{-------------------------------------------------------------------------}
procedure file_decompress(dst, src: string);
var
   ifile, ofile: file;
   ibuff, obuff, tbuff: pointer;
   j, k, crc: CXINT;

begin
   Assign(ifile, src);
   Reset(ifile, 1);

   if dst <> ''
   then begin
      Assign(ofile, dst);
      Rewrite(ofile, 1);
   end;

   GetMem(ibuff, CX_MAX_BUFFER+CX_SLOP);
   GetMem(obuff, CX_MAX_BUFFER);
   GetMem(tbuff, CX_D_MINTEMP);

   repeat
      write('.');

      BlockRead(ifile, j, CXINTSIZE);

      if j <> 0
      then begin
         BlockRead(ifile, k, CXINTSIZE);
         BlockREad(ifile, crc, CXINTSIZE);
         BlockRead(ifile, ibuff^, k);

         if CX_CRC(ibuff^, k) <> crc
            then QUIT ('CRC error');

         if j = k
         then begin
            if dst <> ''
               then BlockWrite(ofile, ibuff^, k);
         end
         else begin
            k:= CX_DECOMPRESS(obuff^, CX_MAX_BUFFER, ibuff^, k, tbuff^, CX_D_MINTEMP);
            case k of
               CX_ERR_INVALID:
                  QUIT('data being compressed is corrupt, or was not compressed with Cx.');

               CX_ERR_METHOD:
                  QUIT('unknown or unsupported method, evaluation version only allows CX_MEHTOD1');

               CX_ERR_BUFFSIZE:
                  QUIT('End of Data symbol not found');
   
               CX_ERR_TEMPSIZE:
                  QUIT('invalid temporary buffer size');
            end;

            if k <> j
               then QUIT('data being compressed is corrupt, or was not compressed with Cx.');

            if dst <> ''
               then BlockWrite(ofile, obuff^, j);
         end;
      end;
   until j = 0;

   FreeMem(ibuff, CX_MAX_BUFFER+CX_SLOP);
   FreeMem(obuff, CX_MAX_BUFFER);
   FreeMem(tbuff, CX_D_MINTEMP);

   Close(ifile);
   if dst <> ''
      then Close(ofile);
end;


begin
   if ParamCount < 3
   then begin
      if (ParamStr(1) = 't') or (ParamStr(1) = 'T')
         then file_decompress('', ParamStr(2))
         else usage
   end
   else begin
      if ParamStr(1) = '1'
         then file_compress(ParamStr(3), ParamStr(2), CX_METHOD1, CX_MAX_BUFFER);

      if ParamStr(1) = '2'
         then file_compress(ParamStr(3), ParamStr(2), CX_METHOD2, CX_MAX_BUFFER);

      if ParamStr(1) = '3'
         then file_compress(ParamStr(3), ParamStr(2), CX_METHOD3, CX_MAX_BUFFER);

      if (ParamStr(1) = 'd') or (ParamStr(1) = 'D')
         then file_decompress(ParamStr(3), ParamStr(2));
   end;

   QUIT ('Ok');
end.
