/*  Copyright (C) 1993   Marc Stern  (internet: stern@mble.philips.be)  */

#include "strings.h"
#include <ctype.h>

/***
 *  Function    :   strtoken
 *
 *  Description :   Parse a string to extract the next token.
 *
 *  Decisions   :   The tokens are separated by ' ', '\t' or '\n'.
 *	            Tokens may contain these separators by being quoted with
 *		    one of the following characters : '"`.
 *		    The function returns a pointer to the character following the token.
 *                  Special escape sequences accepted: 
 *                     \b, \f, \n, \r, \t, \e (Esc), \x..., \0..., \...
 *
 *  Parameters  :   in    char *string    string to be tokenized
 *		    out   char *token     string containing token
 *
 *  Return code :   pointer to character following the token.
 *
 *  OS/Compiler :   All
 ***/

char *strtoken( const char *string, char *token )

{ char delimiter;
  int base;

  while ( (*string == ' ') || (*string == '\t') || (*string == '\n') ) string++;

  if( ! *string ) { *token = '\0';
		    return string;
		  }

  if ( (*string == '\"') || (*string == '\'') || (*string == '`') )
     delimiter = *string++;
  else delimiter = '\0';

  while ( *string != delimiter )
	{
          base = 0;

	  switch( *string )
	  {
	    case '\\' : string++;
			switch( *string )
			 {
			   case 'b' : *token++ = '\b';
				      string++;
				      break;

			   case 'f' : *token++ = '\f';
				      string++;
				      break;

			   case 'n' : *token++ = '\n';
				      string++;
				      break;

			   case 'r' : *token++ = '\r';
				      string++;
				      break;

			   case 't' : *token++ = '\t';
				      string++;
				      break;

			   case '\\': *token++ = '\\';
				      string++;
				      break;

			   case '\0': *token = '\\';
				      return string;

			   case '\'': *token++ = '\'';
				      string++;
				      break;

			   case 'x' :
			   case 'X' : base = 16;
			   case '0' : if ( ! base ) base = 8;
			   case '1' :
			   case '2' :
			   case '3' :
			   case '4' :
			   case '5' :
			   case '6' :
			   case '7' : if ( ! base ) base = 10;
                                      { int loop, num;

					for ( loop = 0, num = 0; loop < 4 && isdigit(*string); loop++ )
					    num = num * base + ( *string++ - '0' );
					*token++ = (char) num;
					break;
				      }

			   case 'E' :
			   case 'e' : *token++ = '\033';	/* 033 is ESC */
				      string++;
				      break;

			   default: *token++ = *string++;
				    break;
			 }

			break;

	    case ' '  :
	    case '\t' :
	    case '\n' : if ( ! delimiter ) { *token = '\0';
	    				     return string;
	    				   }
	    		break;

	    default   : *token++ = *string++;
			break;
	  }

	}	/* end of while */

  *token = '\0';
  if ( *string ) string++;		/* at the last delimiter */

  return string;
}
