/**************************************************************************
    Listing 9 - LINE.CPP

    Written by Kevin D. Weeks, April 1990
    Released to the Public Domain
*/

#include "line.hpp"

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
    Constructor - creates an empty instance. all attributes are set to
    zero or to the appropriate BGI defaults.
*/
Line::Line(void)
{
    struct linesettingstype line_type;

    // use BGI default line_style, width, and color
    getlinesettings(&line_type);
	style = (line_styles)line_type.linestyle;
    width = (line_widths)line_type.thickness;
    color = (COLORS)getcolor();
    visible = FALSE;
}

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
    Constructor - since the "copy constructor", Line(const Line &source),
    won't accept a pointer easily so we define a separate pointer
    constructor for convenience' sake.
*/
Line::Line(const Line *source)
{
    start = source->start;
    end = source->end;
    style = source->style;
    width = source->width;
    color = source->color;
    visible = FALSE;
}

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
    Constructor - primary constructor. Copies of all attributes are
    accepted although init_color, init_width, and init_style may be
    defaults.
*/
Line::Line(const Point begin, const Point finish, const COLORS init_color,
           const line_styles init_style, const line_widths init_width)
{
    struct linesettingstype line_type;

    if ((init_style == DEFAULT) || (init_width == DEFAULT))
        getlinesettings(&line_type);

    if (init_style == DEFAULT)
        style = (line_styles)line_type.linestyle;
    else
        style = init_style;

    if (init_width == DEFAULT)
        width = (line_widths)line_type.thickness;
    else
        width = init_width;

    if (init_color == DEFAULT)
        color = (COLORS)getcolor();
    else
        color = init_color;

    start = begin;
    end = finish;
    visible = FALSE;
}

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
    color can be set whether the line is visible or not
*/
void    Line::set_color(const COLORS new_color)
{
    if (new_color == (COLORS)DEFAULT)
        color = (COLORS)getcolor();
    else
        color = new_color;
    if (visible)
    {
        erase();
        draw();
    }
}

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
    style can be set whether the line is visible or not
*/
void    Line::set_style(line_styles new_style)
{
    struct linesettingstype line_type;

    if (new_style == DEFAULT)
    {
        getlinesettings(&line_type);
        new_style = (line_styles)line_type.linestyle;
    }
    if (visible)
    {
        erase();                            // erase using old style
        style = new_style;                  // change style
        draw();                             // draw using new style
    }
    else
        style = new_style;
}

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
    width can be set whether the line is visible or not
*/
void    Line::set_width(line_widths new_width)
{
    struct linesettingstype line_type;

    if (new_width == DEFAULT)
    {
        getlinesettings(&line_type);
        new_width = (line_widths)line_type.thickness;
    }
    if (visible)
    {
        erase();                            // erase using old width
        width = new_width;                  // change style
        draw();                             // draw using new width
    }
    else
        width = new_width;
}

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
    the draw() method first preserves the current BGI settings, then
    modifies defaults for this particular object, and finally restores the
    original settings. had we written the graphics routines from scratch
    the step of preserving and then restoring the settings wouldn't be
    neccessary.
*/
void    Line::draw(void)
{
    struct linesettingstype hold_current;
    int    default_color;

    // don't draw it twice
    if (visible)
        return;

    // preserve current settings
    getlinesettings(&hold_current);
    default_color = getcolor();

    // select color, line style, and width for this object and draw it
    setcolor(color);
    setlinestyle(style,0,width);
    line(start.get_x(),start.get_y(),end.get_x(),end.get_y());

    // restore original settings
    setlinestyle(hold_current.linestyle,hold_current.upattern,
                 hold_current.thickness);
    setcolor(default_color);
    visible = TRUE;
}

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
    the erase method draws over the existing line using the background
    color. as with the draw method, the current BGI settings must be
    preserved and then restored.
*/
void    Line::erase(void)
{
    struct linesettingstype hold_current;
    int    default_color;

    // don't erase it if it isn't visible
    if (!visible)
        return;

    // preserve current settings
    getlinesettings(&hold_current);
    default_color = getcolor();

    // set the forground current color to the background color and set the
    // BGI defaults for this object and draw over it
    setcolor(getbkcolor());
    setlinestyle(style,0,width);
    line(start.get_x(),start.get_y(),end.get_x(),end.get_y());

    // restore the defaults
    setlinestyle(hold_current.linestyle,hold_current.upattern,
                 hold_current.thickness);
    setcolor(default_color);
    visible = FALSE;
}

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
void    Line::move_relative(Point &distance)
{
    erase();
    start += distance;
    end += distance;
    draw();
}

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
void    Line::move_absolute(Point &new_location, Point &reference)
{
    erase();
    if (reference == start)
    {
        end += new_location - start;
        start = new_location;
    }
    else
    {
        start += new_location - end;
        end = new_location;
    }
    draw();
}

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
    this is the private method used by the copy constructor and the =
    operator. Note that the visible attribute is forced to FALSE.
*/
Line    &Line::copy(const Line &source)
{
    start = source.start;
    end = source.end;
    style = source.style;
    width = source.width;
    color = source.color;
    visible = FALSE;
    return *this;
}
