/**************************************************************************
    Listing 10 - POLYGON.HPP

    Written by Kevin D. Weeks, April 1990
    Released to the Public Domain

    This class varies rather sharply from the BGI polygon in that its
    sides are composed of lines which can be manipulated individually.
    In addition, fill() is a method of the standard polygon rather than
    a separate filled polygon type.
*/
#if !defined(POLYGON_HPP)
#define POLYGON_HPP

#include "graphobj.hpp"
#include "point.hpp"
#include "line.hpp"

class Polygon : virtual public Graphics_Object
{
  public:
// Constructors and destructor.
    // notice that a polygon can be declared as either a set of
    // Points or a set of Lines.
    Polygon(void);
    Polygon(const Polygon &source)
    { copy(source); };
    Polygon(const Polygon *source);
    Polygon(const int num_vertices, const Point *vertex,
            const COLORS side_color = (COLORS)DEFAULT,
            const COLORS init_color = (COLORS)DEFAULT,
            const line_styles side_style = (line_styles)DEFAULT,
            const line_widths side_width = (line_widths)DEFAULT,
            const fill_patterns init_pattern = (fill_patterns)DEFAULT);
    Polygon(const int num_sides, const Line *init_side,
            const COLORS side_color = (COLORS)DEFAULT,
            const COLORS init_color = (COLORS)DEFAULT,
            const line_styles side_style = (line_styles)DEFAULT,
            const line_widths side_width = (line_widths)DEFAULT,
            const fill_patterns init_pattern = (fill_patterns)DEFAULT);
    ~Polygon(void);

// Get attribute methods.
    virtual Line            &side(const int side_number);
    virtual int             get_num_sides(void)
                            { return num_sides; };
    virtual fill_patterns   get_pattern(void)
                            { return fill_pattern; };
    virtual COLORS          get_color(void)
                            { return fill_color; };

// Set attribute methods.
    virtual void            set_side_color(COLORS new_color =
                                          (COLORS)DEFAULT);
    virtual void            set_side_style(line_styles new_style =
                                           (line_styles)DEFAULT);
    virtual void            set_side_width(line_widths new_width =
                                           (line_widths)DEFAULT);
    virtual void            set_color(const COLORS new_color);
    virtual void            set_pattern(const fill_patterns new_pattern);

    virtual void            fill(void);     // fill the polygon
    virtual void            empty(void);    // clear the polygon

// Inherited methods.
    virtual void            draw(void);
    virtual void            erase(void);
    virtual void            move_relative(Point &distance);
    virtual void            move_absolute(Point &new_location, Point &reference);

// Operators.
    virtual Polygon         &operator=(const Polygon &source)
                            { return copy(source); };

  protected:
    // these attributes are declared as protected so that descendant
    // classes such as Rectangle, Square, etc may access them directly.
    Line            **sides;
    int             num_sides;
    COLORS          line_color;
    COLORS          fill_color;
    fill_patterns   fill_pattern;
    int             filled;

  private:
    Polygon         &copy(const Polygon &source);
};

#endif
