// ----------------------------------------------------------------------------
// 
//    CALC.CPP
// 
//    Implements classes used in a simple four-function calculator
// 
//    To build a standalone .EXE file, compile with 
//
//          bcc -DTEST calc
//
//    To build a linkable .OBJ, compile with 
//
//          bcc -c calc
//
// ----------------------------------------------------------------------------

#include <process.h>
#include <ctype.h>
#include <iostream.h>
#include <iomanip.h>
#include <stdlib.h>
#include "calc.h"

MemStack::MemStack( size_t sz ) : memBlock( new char[sz] ), size(sz), top(0)
{
}

MemStack::~MemStack()
{
    delete memBlock;
}

void *MemStack::allocate( size_t sz )
{
    char *result = memBlock + top;
    top += sz;
    if( top > size )
        {
        cerr << "Out of memory." << endl;
        exit(1);
        }
    return result;
}

void MemStack::free( void *ptr )
{
    top = (char *)ptr - memBlock;
}

Scanner::Scanner( const char *str ) : input(str), start(str)
{
    nextToken();
}

void Scanner::nextToken()
{
    while( isspace(*input) )
        input++;

    if( isdigit( *input ) )
        {
        char *end;
        lastValue = strtol( input, &end, 10 );
        input = end;
        curTok = Constant;
        }
    else
        {
        curTok = *input;
        if( *input != EOS )
            input++;
        }
}

Scanner *Production::input;
int Production::errorOccurred;
MemStack Production::mem( 1024 );   // this should be more than enough
                                    // for any reasonable expression

void Production::expect( char ch )
{
    if( input->curToken() != ch )
        error( ch );
    input->nextToken();
}

void Production::error( const char *str )
{
    cout << setw( input->curCol()-1 ) << "" << "^" << endl;
    cout << "Error: " << str << endl;
    errorOccurred = 1;
}

void Production::error( char ch )
{
    switch( ch )
        {
        case EOS:
            error( "extra input after expression" );
            break;
        case Constant:
            error( "number expected" );
            break;
        default:
            {
            char buf[] = "  expected.";
            buf[0] = ch;
            error( buf );
            }
        }
}

LongValue::LongValue()
{
    if( errorOccurred )
        return;

    if( input->curToken() == Constant )
        value = input->value();
    else
        error( Constant );
    input->nextToken();
}

long LongValue::eval()
{
    return value;
}

Factor::Factor()
{
    if( errorOccurred )
        return;

    if( input->curToken() == '(' )
        {
        input->nextToken();
        expr = new AddOp;
        expect( ')' );
        }
    else
        expr = new LongValue;
}

long Factor::eval()
{
    return expr->eval();
}

AddOp::AddOp()
{
    if( errorOccurred )
        return;

    left = new MulOp;
    if( isAddOp( input->curToken() ) )
        {
        op = input->curToken();
        input->nextToken();
        right = new AddOp;
        }
    else
        op = None;
}

long AddOp::eval()
{
    long result = left->eval();
    switch( op )
        {
        case '+':
            result += right->eval();
            break;
        case '-':
            result -= right->eval();
            break;
        }
    return result;
}

MulOp::MulOp()
{
    if( errorOccurred )
        return;

    left = new Factor;
    if( isMulOp( input->curToken() ) )
        {
        op = input->curToken();
        input->nextToken();
        right = new MulOp;
        }
    else
        op = None;
}

long MulOp::eval()
{
    long result = left->eval();
    switch( op )
        {
        case '*':
            result *= right->eval();
            break;
        case '/':
            result /= right->eval();
            break;
        }
    return result;
}

Expression::Expression( const char *str )
{
    input = new Scanner(str);
    errorOccurred = 0;
    expr = new AddOp;
    expect( EOS );
}

Expression::~Expression()
{
    delete expr;
    delete input;
}

long Expression::eval()
{
    if( errorOccurred )
        return 0;
    else
        return expr->eval();
}

#if defined( TEST )

const char *getExpression()
{
    cout << endl << "Expression to evaluate(q to quit):" << endl;
    static char buf[128];
    cin.getline( buf, sizeof buf );
    return buf;
}

int main()
{
    const char *buf = getExpression();
    while( tolower(*buf) != 'q' )
        {
        Expression expr( buf );
        if( !expr.error() )
            cout << buf << " = " << expr.eval() << endl;
        buf = getExpression();
        }
    return 0;
}

#endif
