/*
	cl.hpp -- Container Lite v 1.82
	(C) Copyright 1993  John Webster Small
	All rights reserved
*/


#ifndef cl_hpp
#define cl_hpp

#include <limits.h>	// UINT_MAX
#include <stdarg.h>
#include <iostream.h>
#include <iomanip.h>

/*  stream delimiters/manipulators  */

extern char memberStreamDelimiter;

inline ostream& endm(ostream& os)
{ return os << memberStreamDelimiter << flush; }

inline istream& nextm(istream& is)
{ is.get(); return is; }


/*  defined in cl.hpt and cl.hpf  */
extern ostream& operator<<(ostream& os, char ** D);
extern istream& operator>>(istream& is, char ** D);


/*  cl pointer types  */

#define CLblank
#define CLcmP(ID,ITEM)  \
	int (*ID)(const ITEM * D1, const ITEM * D2)
typedef CLcmP(voidCmP,void);
#define CLcmPcast(ID,ITEM)  \
	(CLcmP(CLblank,ITEM)) ID
#define CLcmP0(ITEM)  CLcmPcast(0,ITEM)
#define voidCmP0 ((voidCmP)0)
#define CmPfnc CmP() const

#define CLapplY(ID,ITEM)  \
	void (*ID)(ITEM * D, va_list args)
typedef CLapplY(voidApplY,void);
#define CLapplYcast(ID,ITEM)  \
	(CLapplY(CLblank,ITEM)) ID
#define CLapplY0(ITEM)  CLapplYcast(0,ITEM)
#define voidApplY0 ((voidApplY)0)

typedef struct uniquenessGimmick { int x; }
	* defaultConstructor;
#define defaultConstruct    ((defaultConstructor)0)


/*  cl constants  */

#define CL_MAXNODES	((unsigned) \
			(UINT_MAX/sizeof(void *)))
#define CL_LIMIT	20U
#define CL_DELTA	10U
#define CL_NOTFOUND	CL_MAXNODES

#define CL_SORTED	0x01U
#define CL_BIND_ONLY	0x00U
#define CL_DASSIGN	0x02U
#define CL_DREAD	0x04U
#define CL_DREAD_DEL	0x08U
#define CL_DNEW	0x10U
#define CL_DNEWED	0x20U
#define CL_DDELETE	0x40U
#define CL_DSTORE	0x80U
#define CL_ALL_FLAGS	0xFFU
#define CL_DANDS	CL_DASSIGN|CL_DNEW|\
			CL_DDELETE|CL_DSTORE


class cl {

private:

	void init(unsigned flags = 0U,
		unsigned maxNodes = 0U,
		unsigned limit = 0U,
		unsigned delta =0U);


protected:

	unsigned lowLimit, lowThreshold, first;
	void **  linkS;
	unsigned limit,    delta,   nodes;
	unsigned maxNodes, curNode, flags;
	voidCmP  cmP;

	cl  (defaultConstructor)
			{ init(); }
	void    assign(const cl& b);
	void    destruct();
	virtual voidCmP DcmP(voidCmP cmP)
			{ return cmP; }
	virtual void * Dassign(void *, const void *)
			{ return (void *)0; }
	virtual void * Dnew(const void *)
			{ return (void *)0; }
	virtual void   Ddelete(void * D)
			{ delete D; }
	virtual int    Dattach(void *)
			{ return 1; }
	virtual void   Ddetach(void *)
			{}
	virtual	int    Dput(ostream&, void *)
			{ return 0; }
	virtual	void * Dget(istream&) 
			{ return (void *)0; }
	virtual int    put(ostream& os);
	virtual int    get(istream& is);
	int     load(const char * filename);
	int     save(const char * filename);
	void    vforEach(voidApplY B, va_list args);


public:

/*  Constructors and destructor  */

	cl     (unsigned flags = CL_BIND_ONLY,
		unsigned maxNodes = CL_MAXNODES,
		unsigned limit = CL_LIMIT,
		unsigned delta = CL_DELTA)
		{ init(flags,maxNodes,limit,delta); }
	cl     (void ** argv, unsigned argc = 0U,
		unsigned flags = CL_BIND_ONLY);
	cl&     operator=(const cl& b)
		{ return *this; }
	int     operator==(const cl& b) const;
	int     operator> (const cl& b) const;
	void ** vector(void ** argv = (void **)0,
			unsigned argc = 0U) const;
	virtual ~cl()  { destruct(); }
/*
	You must override this destructor in any
	descendant that overrides Ddelete() or
	Ddetach(), i.e.

	virtual ~DerivedFromCL()
		{ cl::destruct(); }

	This is because base destructors are called
	after vfts are reset to their default values
	for the base level.  This means that
	cl::~cl() calls cl::Ddetach()
	and perhaps cl::Ddelete() instead of
	the intended DerivedFromCL::Ddetach()
	or DerivedFromCL::Ddelete().
*/


/*  Housekeeping Primitives  */

	unsigned Limit() const { return limit; }
	unsigned setLimit(unsigned newLimit);
	unsigned pack()  { return setLimit(nodes); }
	unsigned Delta() const { return delta; }
	unsigned setDelta(unsigned newDelta
			= CL_DELTA);
	unsigned Nodes() const { return nodes; }
	unsigned MaxNodes() const { return maxNodes; }
	unsigned setMaxNodes(unsigned newMaxNodes
			= CL_MAXNODES);
	unsigned vacancy() const
			{ return maxNodes - nodes; }
	unsigned vacancyNonElastic() const
			{ return limit - nodes; }
	unsigned Flags(unsigned flags = CL_ALL_FLAGS)
			const
		    { return (this->flags & flags); }
	unsigned setFlags(unsigned flags)
		    { return (this->flags |= flags); }
	unsigned resetFlags(unsigned flags)
		    { return (this->flags &= ~flags); }
	cl&  operator<<(cl& (*manipulator)
			(cl&))
			{ return (manipulator?
				(*manipulator)(*this)
				: *this); }

/*  Elastic Array Primitives  */

	void *   atIns(unsigned n, void * D);
	void *   atInsNew(unsigned n, const void * D);
	void *   atRmv(unsigned n);
	void     allRmv();
	int      atDel(unsigned n);
	void *   atDelAsg(unsigned n, void * D);
	int      allDel();
	void     allClr();
	void *   atPut(unsigned n, void * D);
	void *   atPutNew(unsigned n, const void * D);
	void *   atPutAsg(unsigned n, const void * D);
	void *   atGet(unsigned n) const;
	void *   operator[](unsigned n) const
			{ return atGet(n); }
	void *   atGetAsg(unsigned n, void * D);
	void *   atXchg(unsigned n, void * D);
	unsigned index(const void * D) const;
	void     forEach(voidApplY B, ...)
			{ va_list args;
			va_start(args,B);
			vforEach(B,args);	
			va_end(args); }


/*  Stack - Deque - Queue Primitives  */

	void *   push(void * D)
			{ return atIns(0U,D); }
	void *   pushNew(const void * D)
			{ return atInsNew(0U,D); }
	void *   pop()  { return atRmv(0U); }
	cl&      operator>>(void *& D)
			{ D = atRmv(0U);
			return *this; }
	int      popDel()  { return atDel(0U); }
	void *   popDelAsg(void * D)
			{ return atDelAsg(0U,D); }
	void *   top() const { return atGet(0U); }
	void *   topAsg(void * D)
			{ return atGetAsg(0U,D); }
	void *   insQ(void * D)
			{ return atIns(nodes,D); }
	cl&      operator<<(void * D)
		    { atIns(nodes,D); return *this; }
	void *   insQNew(const void * D)
			{ return atInsNew(nodes,D); }
	void *   unQ()  { return atRmv(nodes-1); }
	int      unQDel()  { return atDel(nodes-1); }
	void *   unQDelAsg(void * D)
			{ return atDelAsg(nodes-1,D); }
	void *   rear() const
			{ return atGet(nodes-1); }
	void *   rearAsg(void * D)
			{ return atGetAsg(nodes-1,D); }

/*  List (single and double linked) Primitives  */

	unsigned CurNode() const;
	int      setCurNode(unsigned n = CL_NOTFOUND);
	void *   ins(void * D);
	void *   insNew(const void * D);
	void *   rmv();
	int      del();
	void *   delAsg(void * D);
	void *   put(void * D)
			{ return atPut(curNode,D); }
	void *   putNew(const void * D)
			{ return atPutNew(curNode,D); }
	void *   putAsg(const void * D)
			{ return atPutAsg(curNode,D); }
	void *   get() const { return atGet(curNode); }
	void *   getAsg(void * D)
			{ return atGetAsg(curNode,D); }
	void *   next();
	void *   operator++()  { return next(); }
	void *   nextAsg(void * D);
	void *   prev();
	void *   operator--()  { return prev(); }
	void *   prevAsg(void * D);


/* Priority Q, Set, Bag, Dictionary, Sort Primitives */

	unsigned Sorted() const
		    { return (flags & CL_SORTED); }
	void     unSort()  { flags &= ~CL_SORTED; }
	void     setCmP(voidCmP cmP = voidCmP0)
			{ this->cmP = DcmP(cmP);
			flags &= ~CL_SORTED; }
	voidCmP  CmP() const { return cmP; }
	int      sort(voidCmP cmP = voidCmP0);
	void *   insSort(void * D);
	void *   insSortNew(const void * D);
	void *   insUnique(void * D);
	void *   insUniqueNew(const void * D);
	void *   findFirst(const void * K);
	void *   findNext(const void * K);
	void *   findLast(const void * K);
	void *   findPrev(const void * K);
	unsigned findAll(const void * K);

};	/*  class cl  */


inline ostream& operator<<(ostream& os, cl&)
{ return os; }
inline istream& operator>>(istream& is, cl&)
{ return is; }


typedef void (*GenericFnC)();
#define GenericFnC0  ((GenericFnC)0)

class FunctionRegistry : cl  {

public:
	FunctionRegistry() : cl(CL_DDELETE) {}
	int regFnC(GenericFnC fnC, unsigned id);
	unsigned fnC_2_ID(GenericFnC fnC);
	GenericFnC ID_2_fnC(unsigned id);
	void forget() { allDel(); }
	~FunctionRegistry() {}
};

extern  FunctionRegistry fnCv;

#define ForgetFunctions() fnCv.forget()
#define RegisterFunction(fnC,id)  \
		fnCv.regFnC((GenericFnC)fnC,id)
#define RegisterCmP(cmP,id)  \
		RegisterFunction(cmP,id)
#define fnC2ID(fnC) fnCv.fnC_2_ID((GenericFnC)fnC)
#define ID2fnC(id)  fnCv.ID_2_fnC(id)
#define cmP2ID(cmP) fnC2ID(cmP)
#define ID2cmP(id) ((voidCmP)ID2fnC(id))


#endif  /*  cl_hpp  */
