/*
   Sample module for channeling console output through
   a single task to avoid problems with non-reentrant output
   routines.
*/

#include "task.hpp"
#include <conio.h>
#include <string.h>

#define BUFSIZE   1024     /* Size of the console out buffer */
#define STRLEN    256      /* Max. length of single output string */
#define STACKSIZE 2048     /* Size of stack for output task */

local void far conout (void);
local char conout_stack [STACKSIZE];
local char conout_bufbuf [BUFSIZE];
local char conout_str [STRLEN+1];

local buffer conout_buf(conout_bufbuf, BUFSIZE);
local task conout_task((funcptr) conout, (byteptr) conout_stack,
                       STACKSIZE, PRI_STD + 100, NULL);

/* -------------------------------------------------------------- */

/*
   conout: The console output task. Reads strings from the buffer
           and displays them on the console.
*/

local void far conout (void)
{
   int siz, i;
   register int ch;

   while (1)
      {
      siz = conout_buf.read_buffer (conout_str, STRLEN, 0L);
      for (i = 0; i < siz; i++)
         {
         switch (ch = conout_str [i])
            {
            case '\n':  putch ('\r');
                        putch ('\n');
                        break;
            case 0x07:  sound (2000);
                        t_delay (3L);
                        nosound ();
                        break;
            default:    putch (ch);
            }
         }
      }
}


/*
   init_conout:   Creates buffer and task. Must be called
                  before using any other routine from this module.
*/

void init_conout (void)
{
   conout_task.start_task ();
}


/*
   end_conout: Deletes task and buffer. Should be called before
               terminating CTask.
*/

void end_conout (void)
{
   conout_task.task::~task();
   conout_buf.buffer::~buffer;
}

/* -------------------------------------------------------------- */

/*
   tprintf:    Buffered replacement for printf/cprintf.
*/

int tprintf (char *format, char *argptr)
{
   char buf [256];
   int res;

   if ((res = vsprintf (buf, format, &argptr)) > 0)
      if (conout_buf.write_buffer (buf, res, 0L) < 0)
         res = 0;
   return res;
}


/*
   tputs:      Buffered replacement for puts.
*/

int tputs (char *buf)
{
   return (conout_buf.write_buffer (buf, strlen (buf), 0L) < 0) ? -1 : 0;
}


/*
   tputch:     Buffered replacement for putch.
*/

int tputch (int ch)
{
   return (conout_buf.write_buffer (&ch, 1, 0L) < 0) ? EOF : ch;
}

