/*
   CPPTask - A Multitasking Kernel For C++

   Version 1.0 08-12-91

   Ported by Rich Smith from:

   Public Domain Software written by
      Thomas Wagner
      Patschkauer Weg 31
      D-1000 Berlin 33
      West Germany

   NOTE: None of the Buffer routines my be used from inside an interrupt
         handler. The routines are relatively slow, since they use
         the normal word pipe and resource calls. Optimization would be 
         possible by using block moves and internally handling resources.

   TSKBUF.CPP - Buffered Message handling routines.

   Subroutines:
        buffer::buffer
        buffer::~buffer
        buffer::read_buffer
        buffer::c_read_buffer
        buffer::write_buffer
        buffer::c_write_buffer
        buffer::check_buffer

*/

#include <stdio.h>

#include "task.hpp"
#include "tsklocal.hpp"


/*
   buffer
*/
buffer::buffer (farptr pbuf, word bufsize)
:pip(pbuf, bufsize), buf_write(), buf_read()
{
   flags = 0;
   msgcnt = 0;
}


/*
   ~buffer
*/
buffer::~buffer ()
{
   msgcnt = 0;
}


/*
   read_buffer
*/
int far buffer::read_buffer (farptr msg, int size, dword timeout)
{
   int i, len, l1, l2;
   word w;

   if ((i = buf_read.request_resource (timeout)) < 0)
      return i;

   if ((len = pip.read_wpipe (timeout)) < 0)
      {
      buf_read.release_resource ();
      return len;
      }

   l1 = (len < size) ? len : size;

   for (i = 0; i < l1; i++)
      {
      if (!(i & 1))
         w = pip.read_wpipe (0L);
      else
         w = w >> 8;
      ((byteptr)msg) [i] = (byte)w;
      }

   l2 = (len + 1) >> 1;
   for (i = (l1 + 1) >> 1; i < l2; i++)
      pip.read_wpipe (0L);
   if (l1 < size)
      ((byteptr)msg) [l1] = 0;

   msgcnt--;

   buf_read.release_resource ();
   return len;
}


/*
   c_read_buffer
*/
int far buffer::c_read_buffer (farptr msg, int size)
{
   int res;
   CRITICAL;

   res = -1;
   C_ENTER;

   if (pip.get_filled() && buf_read.get_state())
      {
      buf_read.request_resource (0L);
      C_LEAVE;
      res = read_buffer (msg, size, 0L);
      }
   else
      C_LEAVE;

   return res;
}


/*
   tsk_wrbuf
*/
int buffer::tsk_wrbuf (word w, dword timeout)
{
   int i;

   if ((i = pip.write_wpipe (w, timeout)) < 0)
      buf_write.release_resource ();
   return i;
}


/*
   write_buffer
*/
int far buffer::write_buffer (farptr msg, int size, dword timeout)
{
   int i, res, l2;

   if (size < 0 || (word)((size + 3) >> 1) > pip.get_bufsize())
      return -3;

   if ((i = buf_write.request_resource (timeout)) < 0)
      return i;

   if ((i = tsk_wrbuf ((word)size, timeout)) < 0)
      return i;

   l2 = (size + 1) >> 1;

   for (i = 0; i < l2; i++)
      if ((res = tsk_wrbuf (((wordptr)msg) [i], timeout)) < 0)
         return res;

   msgcnt++;

   buf_write.release_resource ();
   return size;
}


/*
   c_write_buffer
*/
int far buffer::c_write_buffer (farptr msg, int size)
{
   int res;
   CRITICAL;

   if (size < 0 || (word)((size + 3) >> 1) > pip.get_bufsize())
      return -3;

   C_ENTER;
   res = -1;

   if (pip.wpipe_free () >= ((size + 3) >> 1) && buf_write.get_state())
      {
      buf_write.request_resource (0L);
      C_LEAVE;
      res = write_buffer (msg, size, 0L);
      }
   else
      C_LEAVE;

   return res;
}


/*
   check_buffer
*/
word far buffer::check_buffer (void)
{
   return msgcnt;
}

