/*
   CPPTask - A Multitasking Kernel For C++

   Version 1.0 08-12-91

   Ported by Rich Smith from:

   Public Domain Software written by
      Thomas Wagner
      Patschkauer Weg 31
      D-1000 Berlin 33
      West Germany

   TSKCNT.CPP - Counter handling routines.

   Subroutines:
        counter::counter
        counter::~counter
        counter::set_flag
        counter::clear_counter
        counter::wait_counter_set
        counter::wait_counter_clear
        counter::inc_counter
        counter::check_counter

*/

#include <stdio.h>

#include "task.hpp"
#include "tsklocal.hpp"

/*
   counter - initialises counter.
*/

counter::counter ()
{
   flags = 0;

   wait_set = wait_clear = NULL;
   state = 0;
}


/*
   ~counter - kills all processes waiting for counter
*/

counter::~counter ()
{
   CRITICAL;

   C_ENTER;
   tsk_kill_queue (&wait_set);
   tsk_kill_queue (&wait_clear);
   state = 0L;
   C_LEAVE;

}


/*
   clear_counter  - Sets counter to zero. All tasks waiting for
                    Counter zero are made eligible.
*/

void far counter::clear_counter (void)
{
   CRITICAL;

   C_ENTER;
   state = 0L;
   while (wait_clear != NULL)
      wait_clear = wait_clear->tsk_runable ();
   C_LEAVE;
}

/*
   wait_counter_set  - Wait until counter is != 0. If counter is != 0 on
                       entry, the counter is decremented and the task
                       continues to run. If the counter is decremented to 
                       zero, tasks waiting for zero are made eligible.
*/

int far counter::wait_counter_set (dword timeout)
{
   CRITICAL;

   C_ENTER;
   if (state)
      {
      if (!--state)
         while (wait_clear != NULL)
            wait_clear = wait_clear->tsk_runable ();
      C_LEAVE;
      return 0;
      }

   tsk_current->set_retptr(NULL);
   tsk_wait (&wait_set, timeout);
   return (int)tsk_current->get_retptr();
}

/*
   wait_counter_clear - Wait until counter is == 0. If counter is == 0 on
                       entry, the task continues to run.
*/

int far counter::wait_counter_clear (dword timeout)
{
   CRITICAL;

   C_ENTER;
   if (!state)
      {
      C_LEAVE;
      return 0;
      }

   tsk_current->set_retptr(NULL);
   tsk_wait (&wait_clear, timeout);
   return (int)tsk_current->get_retptr();
}


/*
   inc_counter - Increment counter. If there are tasks waiting for the
                 set state, the first task in the queue is made eligible.
*/

void far counter::inc_counter (void)
{
   tcbptr curr;

   CRITICAL;

   C_ENTER;
   if ((curr = wait_set) == NULL)
      {
      state++;
      C_LEAVE;
      return;
      }
   wait_set = curr->tsk_runable ();
   C_LEAVE;
}


/*
   check_counter - return current counter state.
*/

dword far counter::check_counter (void)
{
   return state;
}



void far asm_inc_counter(counterptr cnt)
{
   cnt->inc_counter();
}
