/****************************************************************
* FILE:	dissolve.c
* DESC:	This file accepts two PCX files and does a dissolve
*		between them.
* 
* HISTORY:	Created	 7/03/1994
* LAST CHANGED: 7/03/1994
* 
*	Copyright (c) 1994 by Scott Anderson
*
****************************************************************/

/* ----------------------INCLUDES----------------------------- */

#include <conio.h>
#include <stdio.h>
#include <io.h>
#include <math.h>
#include <graph.h>
#include <malloc.h>
#include <memory.h>
#include <string.h>

#include "define.h"

/* ----------------------DEFINES------------------------------ */

#define DISSOLVE_TWEENS	1

/* ----------------------PROTOTYPES--------------------------- */

int	getColor(PICTURE *pic, int x, int y, COLOR *color);
int		tweenDissolve(PICTURE *src, PICTURE *dst);

/* ----------------------EXTERNALS---------------------------- */

/**** color routines ****/
extern int		closestColor(int r, int g, int b, PALETTE *palPtr);
extern void		collapseColors(PALETTE *palPtr);

/**** other variables ****/ 
/* ID of palette currently being displayed */
extern int	 	CurrentPal;

extern char		*OutFilename;

/* set from last picture loaded */
extern int		Xmin, Ymin, Xmax, Ymax;

/***** variables used to compute intermediate images ****/
/* number of colors in tweened image before reduction*/
extern int	 	Ncolors;

/* r, g, b frequency counter array */
extern unsigned int far Freq[MAX_COMP][MAX_COMP][MAX_COMP];

/* tweened images red, grn, and blu components*/
extern unsigned char far Red[MAX_WIDE][MAX_TALL];
extern unsigned char far Grn[MAX_WIDE][MAX_TALL];
extern unsigned char far Blu[MAX_WIDE][MAX_TALL];

extern PALETTE TweenPal;			/* resulting palette */

/* ----------------------GLOBAL DATA--------------------------- */

PICTURE *Src;		/* source & destination picture pointers */
PICTURE *Dst;

int		Tweens;

/*****************************************************************
* FUNC: main (int argc, char *argv[])
* 
* DESC: Read in a filename to load
*****************************************************************/

main (int argc, char *argv[])
{
	int		segment;

	/* load the pcx file if one is given */
	if ((3 > argc) || (argc > 5)) {
		printf("Usage: dissolve <source> <dest> [<steps> [<output>]]\n\n");
		printf("Where: <source> is the source PCX filename\n");
		printf("       <dest>   is the destination filename\n");
		printf("       <steps>  is the optional sequence size\n");
		printf("                (the max is %d, the default is %d)\n",
								MAX_TWEENS, DISSOLVE_TWEENS+2);
		printf("       <output> is the optional output filename\n");
		printf("                (defaults to no output)\n");
		printf("Note:  The output filename can be at most %d characters long.\n",
										MAX_NAME_SIZE);
		printf("       The PCX extension is added automatically, so don't\n");
		printf("       include it in the filename.\n");
		printf("       Morph only accepts PCX files with %d X %d resolution\n",
										MAX_WIDE, MAX_TALL);
		printf("       and %d colors.\n", COLORS);
		exit(0);
	}
	if (argc > 3) {
		/* subtract two from the sequence count (for the source
			and target) to get the tween count */
		Tweens = clip (atoi(argv[3]) - 2, 1, MAX_TWEENS);
		if (argc > 4)
			OutFilename = argv[4];
	}
	else
		Tweens = DISSOLVE_TWEENS;
	printf("Loading the file %s\n", argv[1]);
	Src = loadPicture(argv[1]);
	if (Src == NULL)
		quit(MEMORY_ERR, "");
	printf("Loading the file %s\n", argv[2]);
	Dst = loadPicture(argv[2]);
	if (Dst == NULL)
		quit(MEMORY_ERR, "");

	setGraphicsMode();
	tweenDissolve(Src, Dst);
	setTextMode();
}

/*****************************************************************
* FUNC: int	tweenDissolve(PICTURE *src, PICTURE *dst)
* 
* DESC: calculate a pixel to plot, from the warping function
*****************************************************************/

#define TOTAL_WEIGHT		(100)	/* Good for up to 99 tweens */

tweenDissolve(PICTURE *src, PICTURE *dst)
{
	int color;
	int	x,y;
	COLOR scolor, dcolor;
	int t, i, p;
	int r, g, b;
	unsigned int srcweight, srcpaletteindex;
	unsigned int dstweight, dstpaletteindex;

	displayPicture(src);
	saveScreen(&src->pal);

	/* src is on screen, now tween to the target */
	for (t = 1; t <= Tweens; t++) {
		dstweight = t * TOTAL_WEIGHT / (Tweens+1);
		srcweight = TOTAL_WEIGHT - dstweight;

		/* Zero out the buffers */
		initFreq();
		_fmemset(Red, 0, sizeof Red);
		_fmemset(Grn, 0, sizeof Grn);
		_fmemset(Blu, 0, sizeof Blu);

		/* Go through the screen positions */
		for (y = Ymin; y <= Ymax; y++)	{
			if (quitCheck())
				quit(0, "");
			for (x = Xmin; x <= Xmax; x++)	{	
				getColor(src, x, y, &scolor);
				getColor(dst, x, y, &dcolor);
				r = (scolor.r * srcweight +	dcolor.r * dstweight)
								/ TOTAL_WEIGHT;
				g = (scolor.g * srcweight +	dcolor.g * dstweight)
								/ TOTAL_WEIGHT;
				b = (scolor.b * srcweight +	dcolor.b * dstweight)
								/ TOTAL_WEIGHT;
				if (Freq[r][g][b] == 0)		/* A new color */
					Ncolors++;
				/* Keep it to one byte */
				if (Freq[r][g][b] < MAX_FREQ)
					Freq[r][g][b]++;
				/* put the RGB components of each pixel into a
										temporary screen buffer */
				Red[x][y] = r;
				Grn[x][y] = g;
				Blu[x][y] = b;
			}
		}
		collapseColors(&TweenPal);
		setPalette(&TweenPal);

		for (y = Ymin; y <= Ymax; y++)	{
			if (quitCheck())
				quit(0, "");
			for (x = Xmin; x <= Xmax; x++)	{
				color = closestColor(	Red[x][y],
										Grn[x][y],
										Blu[x][y],
										&TweenPal);
				_setcolor (color);
				_setpixel (x, y);
			}
		}
		if (!OutFilename) {	/* no output file name was given */
			beep();
			beep();
			waitForKey();	/* so pause to enjoy the pictures */
		}
		else
			saveScreen(&TweenPal);
	}
	CurrentPal = 0; 		/* force a new palette */
	displayPicture(dst);
	saveScreen(&dst->pal);
}

/*****************************************************************
* FUNC: int	getColor(PICTURE *pic, int x, int y, COLOR *color)
* 
* DESC: Return the index and the RGB color at the given
*		x,y coordinate.
*****************************************************************/

int
getColor(PICTURE *picture, int x, int y, COLOR *color)
{
	int		paletteIndex;

	paletteIndex = PIXEL (picture, x, y);
	color->r = picture->pal.c[paletteIndex].r;
	color->g = picture->pal.c[paletteIndex].g;
	color->b = picture->pal.c[paletteIndex].b;
	return (paletteIndex);
}


