
// Matv - A Simple Matrix Class

// Version: 2.0
// Author: Mark Von Tress, Ph.D.
// Date: 09/17/94

// Copyright(c) Mark Von Tress 1994


// DISCLAIMER: THIS PROGRAM IS PROVIDED AS IS, WITHOUT ANY
// WARRANTY, EXPRESSED OR IMPLIED, INCLUDING BUT NOT LIMITED
// TO FITNESS FOR A PARTICULAR PURPOSE. THE AUTHOR DISCLAIMS
// ALL LIABILITY FOR DIRECT OR CONSEQUENTIAL DAMAGES RESULTING
// FROM USE OF THIS PROGRAM.


#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <math.h>
#include <iostream.h>
#include <iomanip.h>
#include <fstream.h>


/////////////////////  If your compiler has trouble problems with deleting temporary matices
#ifdef __SC__
#define SCProblemWithTemporaries  CanDelete = TTRUE;
#else
#define SCProblemWithTemporaries
#endif

#ifdef __ZTC__
#define SCProblemWithTemporaries  CanDelete = TTRUE;
#else
#define SCProblemWithTemporaries
#endif


/////////////////////// select checking and environment options
// uncomment the next lines to disable garbage checking
//#ifndef NO_CHECKING
//#define NO_CHECKING
//#endif

//if you are using a dos extender or 32 bit environment,
//uncomment the next #define
//to have extra long arrays (vectors with up to 32768*32768 - 2 elements)
//#define USE_EXTENDER

// to use a huge vectors uncomment the next line
//#define USE_HUGE
#ifdef USE_HUGE
#ifdef _MSC_VER
#define HUGE_PTR __huge
#endif
#ifdef __BCPLUSPLUS__
#define HUGE_PTR huge
#endif
#ifdef __TCPLUSPLUS__
#define HUGE_PTR huge
#endif
#else
// default is undefined
#define HUGE_PTR
#endif

////////////////////// select precision

//to use double precision, uncomment the next line
//#define USE_DOUBLES

//to use complex variables, uncomment the next line
//#define USE_COMPLEX


#ifdef USE_DOUBLES
#define MODELFOUND
#define FLOAT double
#define LFLOAT long double
FLOAT conj( FLOAT x );
#endif

#ifdef USE_COMPLEX
#define MODELFOUND
#ifdef _MSC_VER
#error Microsoft C++ does not have a complex number object as of this date
#endif
#define FLOAT complex
#define LFLOAT complex
#include "complex.h"
// note that compile fails when the borland speed optimization option
// "inline intrinsic functions" is turned on
double fabs( complex x);
#endif

#ifndef MODELFOUND
#define FLOAT float
#define LFLOAT double
FLOAT conj( FLOAT x );
#endif

#undef MODELFOUND



#ifdef USE_EXTENDER
#define BIGINT unsigned long
#else
#define BIGINT int
#endif


////////////////// vector base signature

#ifndef SIGNATURE
#define SIGNATURE 0x5a5a
#endif

////////////////////////////////// compile matv.h only once
#ifndef _MATVHEADERREAD
#define _MATVHEADERREAD

//////////////////////////////////////////////////// vector
typedef enum bboolean { FFALSE, TTRUE } bboolean;

class Vector
{
private :
	FLOAT HUGE_PTR* c;  // vector of floats
	BIGINT n;           // number of floats
	bboolean CanDelete; // Gives permission to delete c - initially true
	bboolean CanAlias;  // Can be aliased - initially true
	long signature;     // for heap checking
	FLOAT HUGE_PTR *SetupVectors(BIGINT n);
	void PurgeVectors(void);

public :
	Vector(void);
	Vector(BIGINT n);
	Vector(BIGINT n, FLOAT *x);
	Vector(int r, int c);
	Vector(int r, int c, FLOAT *x);
	Vector(const Vector &a);
	Vector &operator = (const Vector &a);
	~ Vector();

	FLOAT &operator() (BIGINT i) const;
	BIGINT rows(void) { return n; }

	static FLOAT Nrerror(const char *errormsg);
	void Garbage(const char *errormsg) const;

	friend ostream &operator << (ostream &s, Vector &v);
	void release() { if (CanAlias == TTRUE) CanDelete = FFALSE;
						  SCProblemWithTemporaries
						}
	void CannotAlias() { CanAlias = FFALSE; }

};

/////////////////////////////////// matrix

class Matrix : public Vector
{
public :
	int r, c;

	// indexing - 1d or 2d
	FLOAT &operator() (int i) const;
	FLOAT &operator() (int i, int j) const;

	// constructors and destructor
	Matrix(void) : Vector(1), r(1), c(1) { }
	Matrix(int rr, int cc) : Vector( rr, cc ), r(rr), c(cc) { }
	Matrix(int rr, int cc, FLOAT *x) : Vector( rr, cc, x), r(rr), c(cc) { }
	Matrix(const Matrix &a) : Vector(a), r(a.r), c(a.c) { }
	Matrix &operator = (const Matrix &a);
	~ Matrix() { }

	// io
	friend ostream &operator << (ostream &s, const Matrix &m);
	void show(const char *str);
	void show( void ){ show(" "); }

	// addition
	friend Matrix operator + (const Matrix &a, const Matrix &b);
	friend Matrix operator + (const Matrix &a, FLOAT b);
	friend Matrix operator + (FLOAT b, const Matrix &a);
	Matrix &operator += ( const  Matrix &a );
	Matrix &operator += ( FLOAT b );

	// subtraction
	friend Matrix operator - ( const Matrix &a,  const Matrix &b);
	friend Matrix operator - ( const Matrix &a, FLOAT b);
	friend Matrix operator - (FLOAT b,  const Matrix &a);
	Matrix &operator -= ( const  Matrix &a );
	Matrix &operator -= ( FLOAT b );
	friend Matrix operator - ( const Matrix &a); // unary minus

	// multiplication
	friend Matrix operator * ( const Matrix &a,  const Matrix &b);
	friend Matrix operator * ( const Matrix &a, FLOAT b);
	friend Matrix operator * (FLOAT b,  const Matrix &a);
	Matrix &operator *= (  const Matrix &a );
	Matrix &operator *= ( FLOAT b );

	friend Matrix operator % ( const Matrix &a, const  Matrix &b); // emult
	friend Matrix operator % (FLOAT b, const  Matrix &a);
	friend Matrix operator % ( const Matrix &a, FLOAT b);
	Matrix &operator %= ( const  Matrix &a );
	Matrix &operator %= ( FLOAT b );

	// elementwise division
	friend Matrix operator / ( const Matrix &a, const  Matrix &b);
	friend Matrix operator / ( const Matrix &a, FLOAT b);
	friend Matrix operator / (FLOAT b, const  Matrix &a);
	Matrix &operator /= ( const  Matrix &a );
	Matrix &operator /= ( FLOAT b );
};

///////////////////////////// display control

int Setwid( int w );                      // set display width
int Setdec( int d );                      // set decimals
int Getwid( void  );                      // report display width
int Getdec( void  );                      // report decimals

////////////////////////// file io

Matrix Reada(char *filename);
void Writea(char *filename, const Matrix &a, const char *filetitle = "A_Matrix");


/////////////////////////// sorting, subsetting, and conatenation
//   sort    order == 0 =>  sort col c and exchange rows
//           order != 0 =>  sort row c and exchange cols
//           does not compile for complex matrices
//
//   submat  r,c is bottom right corner,   lr,lc is upper left corner
//
//   Ch   horozontal concatenation
//   Cv   vertical concatentation
//


Matrix MSort( Matrix &a, int col, int order=0);
Matrix Submat(Matrix &a, int lr, int lc, int r, int c);
Matrix Ch( Matrix &a, Matrix &b );
Matrix Cv( Matrix &a, Matrix &b );

/////////////////////////// special matrices
//
//   Tran    transpose of matrix
//   Mexp    elementwise exponential
//   Mabs    elementwise absolute value
//   Mlog    elementwise log
//   Msqrt   elementwise sqrt
//   Trace   trace of square matrix
//   Ident   identity matrix
//   Helm    Helmert matrix
//   Fill    constant matrix
//   Index   build an index matrix: if lower>upper step downward
//

Matrix Tran(Matrix &ROp );
Matrix Mexp(Matrix &ROp );
Matrix Mabs(Matrix &ROp );
Matrix Mlog(Matrix &ROp );
Matrix Msqrt(Matrix &ROp );
FLOAT Trace(Matrix &ROp );
Matrix Ident(int n );
Matrix Helm( int n );
Matrix Fill(int r, int cc, FLOAT d = 0);
Matrix Index( int lower, int upper, int step = 1);


/////////////////////////// inversion and decomposition
//
//     Sweep       sweep on diagonal elements k1 to k2
//     Inv         matrix inverse using sweep(1,ROp.c,ROp)
//     Kron        Kroniker (direct) product
//     Det         Determinant
//     Cholesky    upper triangular S where ROp = S'S, for symmetric ROp
//     QR          ROp( rxc, r>c ) = Q(rxc) R(cxc), and Q'Q = Ident(c)
//                 makeq = TTRUE => make Q, it is garbage otherwise
//     Svd         ROp = USTran(V),
//                 where U'U = I, V'V=I, S= Column of singular values
//                 makeu = TTrue => Compute U, it is garbage otherwise
//                 makev = TTrue => Compute V, it is garbage otherwise
//     Ginv        See Svd, Ginv(ROp) = V Inv(Diag(S+)) Tran(U)
//     Fft         Almost fast Fourier transform, can be slow if
//                 n > 1000 or prime factors of n > 19, does not require
//                 n to be a power of 2
//                 INZEE = TTRUE => compute fft
//                 INZEE = FFALSE=> compute ifft
//
//    QR, Svd, Ginv, FFt not yet available for Complex matrices
//

Matrix Sweep(int k1, int k2, Matrix &a); // g2 sweep
Matrix Inv(Matrix &a);                   // invert - G2 sweep
Matrix Kron( Matrix &a, Matrix &b );
FLOAT Det( Matrix &ROp );
Matrix Cholesky(Matrix& ROp);
void QR( Matrix& ROp, Matrix& Q, Matrix& R, bboolean makeq = TTRUE);
int Svd( Matrix &A, Matrix &U, Matrix &S, Matrix &V,
			 bboolean makeu = TTRUE, bboolean makev = TTRUE);
Matrix Ginv( Matrix& ROp );
Matrix Fft( Matrix &ROp, bboolean INZEE = TTRUE);


////////////////////////// reshaping functions
// none of these functions compile for complex matrices

Matrix Vec( Matrix& ROp );      // send matrix to vector
										  // stack columns into an (r*c)x1
Matrix Vecdiag( Matrix& ROp );  // extract diagonal into a vector
Matrix Diag( Matrix& ROp );     // create a diagonal matrix from
										  // square matrix or column vector
Matrix Shape( Matrix& ROp, int nrows = 1 );       // reshape into nrows

////////////////////////// summation functions
// none of these functions compile for complex matrices

typedef enum margins { ALL, ROWS, COLUMNS } margins; // summation margins

Matrix Sum( Matrix& ROp, margins marg = ALL );    // sums over margins
Matrix Sumsq( Matrix& ROp, margins marg = ALL );  // sum of squares
Matrix Cusum( Matrix& ROp);                       // accumulate along rows


/////////////////////////// max and mins
//
// if marg = ALL      return 3x1   [ maxr, maxc, max element ]
// if marg = ROWS     return 3xc   Tran([ maxr, col=1..c, max element ])
// if marg = COLUMNS  return rx3   [ row=1..r, maxc, max element]
//
// same structure for mins
//
// none of these functions compile for complex matrices

Matrix Mmax( Matrix& ROp, margins marg=ALL );   // matrix max
Matrix Mmin( Matrix& ROp, margins marg=ALL );   //  matrix min

////////////////////////// row and column operations
//
// Crow      c*r1
// Srow      swap r1 with r2
// Lrow      r2 + c*r1
// Ccol      c*c1
// Scol      swap c1 with c2
// Lcol      c2 + c*c1
//
// note default values for row and col are out of range, so they
// must be supplied. This is for error checking.
// default constants perform no changes.
// !!!!!!!!!!!!! these functions operate directly on matrix
//////////////////////////

void Crow( Matrix& ROp, int row=0, double c=1.0);
void Srow( Matrix &ROp, int row1 = 0, int row2 = 0);
void Lrow( Matrix &ROp, int row1 = 0, int row2 = 0, double c=0.0);
void Ccol( Matrix& ROp, int col=0, double c=1.0);
void Scol( Matrix &ROp, int col1 = 0, int col2 = 0);
void Lcol( Matrix &ROp, int col1 = 0, int col2 = 0, double c=0.0);


// end of _MATVHEADERREAD
#endif
