/* MFARHEAP.C - Far heap example #1.
   This program illustrates Microsoft C6.0a far heap memory
   management using _fheapset, _fheapchk, _fmalloc, _fmsize,
   _fmemset, _ffree, and _fheapwalk. It allocates, mainpulates,
   and deallocates two sets of far heap memory blocks, and shows
   some of the details about how far heap management works.

   Copyright (c) 1991 Borland International. All rights reserved.

 */

#include <stdio.h>
#include <conio.h>
#include <malloc.h>
#include <stdlib.h>
#include <time.h>
/* Note 1 - No #defines here since this is a Microsoft C program */

/* Function prototypes and manifest constants */
void heapdump( char fill );
void heap_status( int status );

#define IMAX 10 /* number of heap blocks to allocate */
void main()
{
   void _far *p[10];
   /* Note 2  - Just the heap pointer array p is here. */
   int i, n_alloc;

    /* Fill all current heap blocks with known data. */
    heap_status( _fheapset( 254 ) );

    printf ("Free heap has been initialized.\n");

    /* Check heap consistency and dump it before doing anything else. */
    heapdump( (char)254 );

    /* Check heap status. */
    heap_status( _fheapchk() );
    printf ("Just before allocating blocks.\n");

    /* Now allocate IMAX fixed blocks of memory to affect the far heap,
       and fill them with ones. */
    for( i = 0; i < IMAX; i++ )
    {
    /* Note 3 - Use the heap size directly here. */
    /* Note 4 - Microsoft C even allows you to allocate a heap block
       that is zero bytes in size.
    */
    if( (p[i] = _fmalloc( (size_t)i*10 )) != NULL )
      printf( "Allocated %u bytes on far heap at %Fp\n",
          _fmsize(p[i]), p[i] );
    else
      break;
    }
    n_alloc = i-1;  /* Index of last block successfully allocated */
    /* Note 5 - With Microsoft C6.0, if there is not enough space
       available on the free heap to satisfy a given call, the
       heap manager will grow the heap when possible.  The
       conditions that allow that are: free memory is available
       from the operating system, and the heap will not grow
       beyond the maximum size permissible for the type of heap.

       As a result, you may see more free memory available after
       space is allocated than before.
    */

    printf ("Just after allocating heap blocks.\n" );
    /* Fill all free heap blocks with known data. */
    heap_status( _fheapset( 254 ) );
    printf ("Free heap has been initialized again.\n" );

    /* Fill all allocated heap blocks with different data. */
    for( i = 0; i < n_alloc; i++ )
    _fmemset( p[i], 0xFF, _fmsize( p[i] ) );
    printf ("All allocated heap blocks are now filled with 0xFF.\n" );

    /* Check heap consistency again. */
    heapdump( (char)254 );

    /* Now free up the heap blocks. */
    for( i=n_alloc; i >= 0; i-- )
    {
    printf( "Deallocating %u bytes from far heap at %Fp\n",
        _fmsize( p[i] ), p[i] );
    _ffree( p[i] );
    }

    /* Check heap consistency after releasing memory. */
    heapdump( (char)254 );

    /* Now AGAIN allocate IMAX fixed blocks of memory to
       affect the far heap, and fill them with ones. This
       will see whether free heap gets concatenated together
       to satisfy a given request.
    */
    printf ("Just before allocating blocks second time.\n");
    for( i = 0; i < IMAX; i++ )
    {
    if( (p[i] = _fmalloc( (size_t)i*10 )) != NULL )  /* farmalloc */
      printf( "Allocated %u bytes on far heap at %Fp\n",
          _fmsize(p[i]), p[i] );
    else
      break;
    }
    n_alloc = i-1;  /* Index of last block successfully allocated */
    printf ("Just after allocating heap blocks second time.\n" );
    /* Fill all free heap blocks with known data. */
    heap_status( _fheapset( 254 ) );
    printf ("Free heap has been initialized again.\n" );

    /* Fill all allocated heap blocks with different data. */
    for( i = 0; i < n_alloc; i++ )
    _fmemset( p[i], 0, _fmsize( p[i] ) );
    printf ("All allocated heap blocks are now filled with 0.\n" );

    /* Check heap consistency again. */
    heapdump( (char)254 );

    /* Now free up the heap blocks. */
    for( i=n_alloc; i >= 0; i-- )
    {
    printf( "Deallocating %u bytes from far heap at %Fp\n",
        _fmsize( p[i] ), p[i] );
    _ffree( p[i] );
    }

    /* Final check of heap consistency. */
    heapdump( (char)254 );

    exit(0);
}

void heapdump( char fill )
/* heapdump walks through all heap entries, checks consistency
   of free blocks, and displays information about each heap block.
*/
{
    struct _heapinfo hi;
    int hstate, i;
    char _far *p;

    printf("\n-- Current Heap Blocks --\nStatus    Size   Address\n");
    hi._pentry = NULL;      /* Initialize to get first heap entry */
    while( ( hstate = _fheapwalk( &hi )) == _HEAPOK )
    {
    /* Note 6 - If the _useflag field in the _heapinfo structure
       = _USEDENTRY, then the heap block is "used", not "free".
       The _size field is an unsigned integer, and it reports exactly
       the number of bytes to which the program has access.
     */
    printf( "%s  %10u %Fp ",
        hi._useflag == _USEDENTRY ? "USED" : "FREE",
        hi._size,
        hi._pentry );

    /* Note 7 - For free heap entries, check each byte to see that it
       contains the requested fill character.
     */
    if( hi._useflag != _USEDENTRY )
    {
        for( p = (char _far *)hi._pentry, i = 0; i < hi._size; p++,i++ )
        if( (char)*p != fill )
            break;
        if( i == hi._size )
        printf( "Not changed\n" );
        else
        printf( "Changed\n" );
    }
    else
      printf ("\n");
    }
    heap_status( hstate );     /* Finish up with heap status */
}

void heap_status( int heapstate )
/* heap_status decodes the status returned by far heap
   management functions _fheapwalk, _fheapset, or _fheapchk,
   and displays appropriate text.
*/
{
    /* Note 8 - _HEAPBADPTR, _HEAPBADBEGIN, and _HEAPBADNODE
       are heap error values unique to Microsoft C6.0a.
     */
    switch( heapstate )
    {
    case _HEAPOK:
        printf( "Heap status is OK.\n" );
        break;
    case _HEAPEMPTY:
        printf( "Heap status is empty.\n" );
        break;
    case _HEAPEND:
        printf( "Heap status is at end of heap.\n" );
        break;
    case _HEAPBADPTR:
        printf( "Heap status is bad heap pointer.\n" );
        break;
    case _HEAPBADBEGIN:
        printf( "Heap status is bad start of heap.\n" );
        break;
    case _HEAPBADNODE:
        printf( "Heap status is bad heap node.\n" );
        break;
    }
}
