//--------------------------------------------------------------------------
//
//      THREADS.H: header file for DOS multithreading library.
//      Copyright (c) J.English 1993.
//      Author's address: je@unix.brighton.ac.uk
//
//      Permission is granted to use copy and distribute the
//      information contained in this file provided that this
//      copyright notice is retained intact and that any software
//      or other document incorporating this file or parts thereof
//      makes the source code for the library of which this file
//      is a part freely available.
//
//--------------------------------------------------------------------------
//
//      Note: this library is highly DOS specific and hence non-portable.
//      It also involves the use of assembly language and interrupt
//      functions, so it is also very compiler-specific and will need
//      modification for use with compilers other than Borland C++ 3.0
//      or later.
//
//      Revision history:
//      1.0     March 1993      Initial coding
//
//--------------------------------------------------------------------------

#ifndef __THREADS_H
#define __THREADS_H

#include <stdlib.h>
#include <stdio.h>

//--------------------------------------------------------------------------
//
//      Class DOSThread: abstract DOS multithreading base class.
//
//      Note that you must derive a concrete class from this base class
//      by supplying a main function containing the body of the thread.
//
class DOSThread
{
    friend class DOSThreadManager;
    
  public:
    enum State                            // status codes for threads
        { CREATED, READY, DELAYED, WAITING, SUSPENDED, TERMINATED };
    enum Error                            // result codes for critical errors
        { IGNORE = 0, RETRY = 1, FAIL = 3 };

    DOSThread (unsigned stacksize = 2048);  // constructor
    virtual ~DOSThread ();                  // destructor

    int  status ()                        { return state; }

    int  run ();                          // start thread running
    void terminate ();                    // terminate thread

    static void pause ();                 // schedule another thread
    static void delay (int n);            // delay for "n" clock ticks
    static void timeslice (unsigned n);   // set timeslice to "n" ticks
    static int  userbreak ();             // test if control-break pressed
    static int  cancelbreak ();           // test and reset control-break flag

  protected:
    virtual void  main () = 0;            // main body of thread
    virtual Error DOSerror (int)          { return FAIL; }

    void wait ();                         // wait for thread to terminate

  private:
    DOSThreadManager* const entry;
    char* const             stack;
    volatile State          state;
};

//--------------------------------------------------------------------------
//
//      Class DOSMonitorQueue: event queue for use by monitor classes.
//
//      Note that there are no member functions for this class.  The
//      functions "suspend" and "resume" in class DOSMonitor (below)
//      are the only operations allowed on a monitor queue.
//
class DOSMonitorQueue
{
    friend class DOSMonitor;

  public:
    DOSMonitorQueue ();                   // constructor
    ~DOSMonitorQueue ();                  // destructor

  private:
    DOSThreadManager* const queue;
};


//--------------------------------------------------------------------------
//
//      Class DOSMonitor: thread synchronisation management class
//
//      Note that there are no public members for this class.  You must
//      derive classes from it which use the locking primitives below
//      to protect against re-entry.  You should overload the error
//      handling function "error" to report any errors as necessary;
//      the default action is to exit with the error code as the exit
//      status.
//
class DOSMonitor
{
  protected:
    DOSMonitor ();                        // constructor
    virtual ~DOSMonitor ();               // destructor

    enum ErrorCode {                      // internal error codes:
        NEW_FAIL     = -1,                // ...out of memory
        NO_THREAD    = -2,                // ...no threads running
        LOCK_FAIL    = -3,                // ...thread already holds lock
        UNLOCK_FAIL  = -4,                // ...unlock attempted without lock
        SUSPEND_FAIL = -5,                // ...suspend attempted without lock
        RESUME_FAIL  = -6                 // ...resume attempted without lock
    };
    virtual void error (ErrorCode e)      { exit (e); }

    void lock ();                         // lock monitor against re-entry
    void unlock ();                       // unlock monitor
    void suspend (DOSMonitorQueue& q);    // suspend & await event
    void resume (DOSMonitorQueue& q);     // resume thread waiting for event

  private:
    DOSThreadManager* const lockq;
    volatile DOSThreadManager* lockholder;
};

#endif
