/****************************************************************************
*
*					 MegaVision Application Framework
*
*			A C++ GUI Toolkit for the MegaGraph Graphics Library
*
*					Copyright (C) 1994 SciTech Software.
*							All rights reserved.
*
* Filename:		$RCSfile: tlist.cpp $
* Version:		$Revision: 1.2 $
*
* Language:		C++ 3.0
* Environment:	IBM PC (MS DOS)
*
* Description:	Member functions for the TList class hierarchy.
*
* $Id: tlist.cpp 1.2 1994/03/09 11:50:36 kjb Exp $
*
****************************************************************************/

#include "mvision.hpp"

#pragma	hdrstop

#include "tlist.hpp"
#include "tfontmgr.hpp"
#include "tgroup.hpp"
#include "tmouse.hpp"
#include "tkeys.hpp"

/*----------------------------- Implementation ----------------------------*/

#define	INDENTLEFT	2
#define	INDENTTOP	3

TListBase::TListBase(const TRect& bounds,const TPoint& cellSize,
	TScrollBar *vScroll,TScrollBar *hScroll,ushort flags)
	: TView(bounds), cellSize(cellSize), flags(flags)
/****************************************************************************
*
* Function:		TListBase::TListBase
* Parameters:	bounds		- Bounding box for the list
*				cellSize	- Dimensions in pixels for each list cell
*				vScroll		- Pointer to horizontal scroll bar (NULL if none)
*				hScroll		- Pointer to vertical scroll bar (NULL if none)
*				flags		- Creation flags
*
* Description:	Constructor for the TListBase class. Works out the number
*				of visible rows and columns in the list.
*
****************************************************************************/
{
	options |= ofSelectable | ofFirstClick;

	visible.left() = visible.top() = 0;
	visible.bottom() = (bounds.bottom() - bounds.top() - 2*_MVIS_sysLineWidth)
								/ cellSize.y;
	visible.right() = (bounds.right() - bounds.left() - 2*_MVIS_sysLineWidth)
								/ cellSize.x;
	setHScroll(hScroll);
	setVScroll(vScroll);
	setCurrentCell(0,0);
}

void TListBase::focusCurrent(bool toTop,bool redraw)
/****************************************************************************
*
* Function:		TListBase::focusCurrent
* Parameters:	toTop	- True if the cell should be moved to the top pos.
*				redraw	- True if list should be redrawn
*
* Description:	Focuses the list on the current cell. If toTop is true, the
*				current cell is moved to the top position in the visible
*				window.
*
****************************************************************************/
{
	if (!visible.includes(cursor)) {
		if (cursor.y < visible.top())
			vScrollTo(cursor.y,redraw);
		if (cursor.y >= visible.bottom()) {
			if (toTop) {
				int y = range.bottom() - (visible.bottom() - visible.top());
				vScrollTo(min(cursor.y,y),redraw);
				}
			else
				vScrollTo(cursor.y - (visible.bottom() - visible.top()-1),
					redraw);
			}
		if (cursor.x < visible.left())
			hScrollTo(cursor.x,redraw);
		if (cursor.x >= visible.right())
			hScrollTo(cursor.x - (visible.right() - visible.left()-1),
				redraw);
		}
}

void TListBase::selectRange(int left,int top,int right,int bottom)
/****************************************************************************
*
* Function:		TListBase::selectRange
* Parameters:	left,top,right,bottom	- Range of cells to select
*
* Description:	Selects all the cells that lie within the specified range.
*
****************************************************************************/
{
	int _left   = MAX(range.left(), left);
	int _top    = MAX(range.top(), top);
	int _right  = MIN(range.right(), right);
	int _bottom = MIN(range.bottom(), bottom);
	for (int i = _left; i < _right; i++)
		for (int j = _top; j < _bottom; j++)
			selectCell(i,j);
}

void TListBase::clearRange(int left,int top,int right,int bottom)
/****************************************************************************
*
* Function:		TListBase::clearRange
* Parameters:	left,top,right,bottom	- Range of cells to select
*
* Description:	Clears all of the cells that lie within the specified
*				range.
*
****************************************************************************/
{
	int _left   = MAX(range.left(), left);
	int _top    = MAX(range.top(), top);
	int _right  = MIN(range.right(), right);
	int _bottom = MIN(range.bottom(), bottom);
	for (int i = _left; i < _right; i++)
		for (int j = _top; j < _bottom; j++)
			deselectCell(i,j);
}

void TListBase::clearSelection()
/****************************************************************************
*
* Function:		TListBase::clearSelection
*
* Description:	Clears all of the cells in the list.
*
****************************************************************************/
{
	clearRange(selection);
	selection.topLeft = cursor;
	selection.right() = selection.left()+1;
	selection.bottom() = selection.top()+1;
	flags &= ~lsExtending;
}

void TListBase::selectNext(ushort direction,short count,ushort modifiers,
	bool toTop)
/****************************************************************************
*
* Function:		TListBase::selectNext
* Parameters:	direction	- Flags indicating the direction to move
*				count		- Number of cells to move down
*				modifiers	- Keyboard shift modifiers
*				toTop		- True if the cell should be moved to the top
*
* Description:	Adjusts the selection by the specified number of cells in
*				the specified direction. If the shift modifiers are set,
*				then the selection is extended.
*
****************************************************************************/
{
	TPoint	oldCursor(cursor);
	short	maxv = maxV(),minv = minV();
	short	maxh = maxH(),minh = minH();

	if (direction & lsBelow)
		if ((cursor.y += count) > maxv)
			cursor.y = maxv;
	if (direction & lsAbove)
		if ((cursor.y -= count) < minv)
			cursor.y = minv;
	if (direction & lsRight)
		if ((cursor.x += count) > maxh)
			cursor.x = maxh;
	if (direction & lsLeft)
		if ((cursor.x -= count) < minh)
			cursor.x = minh;

	if (cursor != oldCursor || (flags & lsExtending)) {
		if ((flags & lsMultipleSelect) && (modifiers & mdShift)) {
			if (cursor == oldCursor)
				return;

			if (direction & lsLeft) {
				if (flags & lsExtendRight) {
					// We are currently extending in the opposite direction,
					// so clear all of the cells from the old cursor position
					// to one above the new cursor position. If the selection
					// is only one high, then turn off the extending flags.

					if (cursor.x <= selection.left()) {
						flags &= ~lsExtendHoriz;
						selection.right() = selection.left()+1;
						selection.left() = cursor.x;
						if (selection.left() != selection.right()-1) {
							flags |= lsExtendLeft;
							selectRange(selection);
							}
						}
					else
						selection.right() = cursor.x+1;
					clearRange(selection.right(),selection.top(),
						oldCursor.x+1,selection.bottom());
					}
				else {
					// We are currently extending the selection in the same
					// direction, or have just started to extend the selection

					flags |= lsExtendLeft;
					selection.left() = cursor.x;
					selectRange(cursor.x,selection.top(),
						oldCursor.x,selection.bottom());
					}
				}

			if (direction & lsRight) {
				if (flags & lsExtendLeft) {
					if (cursor.x >= selection.right()-1) {
						flags &= ~lsExtendHoriz;
						selection.left() = selection.right()-1;
						selection.right() = cursor.x+1;
						if (selection.left() != selection.right()-1) {
							flags |= lsExtendRight;
							selectRange(selection);
							}
						}
					else
						selection.left() = cursor.x;
					clearRange(oldCursor.x,selection.top(),
						selection.left(),selection.bottom());
					}
				else {
					flags |= lsExtendRight;
					selection.right() = cursor.x+1;
					selectRange(oldCursor.x+1,selection.top(),
						cursor.x+1,selection.bottom());
					}
				}

			if (direction & lsAbove) {
				if (flags & lsExtendDown) {
					if (cursor.y <= selection.top()) {
						flags &= ~lsExtendVert;
						selection.bottom() = selection.top()+1;
						selection.top() = cursor.y;
						if (selection.top() != selection.bottom()-1) {
							flags |= lsExtendUp;
							selectRange(selection);
							}
						}
					else
						selection.bottom() = cursor.y+1;
					clearRange(selection.left(),selection.bottom(),
						selection.right(),oldCursor.y+1);
					}
				else {
					flags |= lsExtendUp;
					selection.top() = cursor.y;
					selectRange(selection.left(),cursor.y,
						selection.right(),oldCursor.y);
					}
				}

			if (direction & lsBelow) {
				if (flags & lsExtendUp) {
					if (cursor.y >= selection.bottom()-1) {
						flags &= ~lsExtendVert;
						selection.top() = selection.bottom()-1;
						selection.bottom() = cursor.y+1;
						if (selection.top() != selection.bottom()-1) {
							flags |= lsExtendDown;
							selectRange(selection);
							}
						}
					else
						selection.top() = cursor.y;
					clearRange(selection.left(),oldCursor.y,
						selection.right(),selection.top());
					}
				else {
					flags |= lsExtendDown;
					selection.bottom() = cursor.y+1;
					selectRange(selection.left(),oldCursor.y+1,
						selection.right(),cursor.y+1);
					}
				}
			dirtyCell(oldCursor);
			}
		else {
			// The selection is not being extended, so clear any previous
			// selection and turn extending off, and reselect the cell
			// under the cursor.

			flags &= ~lsExtending;
			if (!(flags & lsDisjointSelect)) {
				clearSelection();
				selectCell(cursor);
				}
			else {
				dirtyCell(oldCursor);
				dirtyCell(cursor);
				}
			}

		message(owner,evBroadcast,cmListCursorChanged,this);
		refresh();
		}
	focusCurrent(toTop);
}

ushort TListBase::findCellHit(const TPoint& global,TPoint& loc)
/****************************************************************************
*
* Function:		TListBase::findCellHit
* Parameters:	global	- Global mouse location point
*				loc		- Place to store computed cell location
* Returns:		Flags representing where the mouse click occurred
*
* Description:	Determines where the mouse click occurred in reference to
*				the list box. If the mouse click was inside the list box,
*				we compute the location of the cell that was hit and return
*				it in loc.
*
****************************************************************************/
{
	Point p(global);
	globalToLocal(p);

	if (bounds.includes(p)) {
		loc.x = (p.x - (bounds.left()+_MVIS_sysLineWidth)) / cellSize.x;
		loc.y = (p.y - (bounds.top()+_MVIS_sysLineWidth)) / cellSize.y;

		loc += visible.topLeft;		// Make relative to visible cells
		return lsInside;			// The mouse click was inside the list
		}

	// The mouse click was outside of the list, so determine where it
	// occurred and set the relevant flags

	ushort	flags = 0;

	if (p.x < bounds.left()+_MVIS_sysLineWidth)		flags |= lsLeft;
	if (p.x >= bounds.right()-_MVIS_sysLineWidth)	flags |= lsRight;
	if (p.y < bounds.top()+_MVIS_sysLineWidth)		flags |= lsAbove;
	if (p.y >= bounds.bottom()-_MVIS_sysLineWidth)	flags |= lsBelow;

	return flags;
}

void TListBase::handleEvent(TEvent& event,phaseType phase)
/****************************************************************************
*
* Function:		TListBase::handleEvent
* Parameters:	event	- Event to handle
*				phase	- Current phase for the event (pre,focus,post)
*
* Description:	Event handling mechanism for the TListBase class.
*
****************************************************************************/
{
	TView::handleEvent(event,phase);

	switch (event.what) {
		case evMouseDown:
			if (range.isEmpty())
				return;

			if (event.mouse.buttons & mbLeftButton) {
				bool 	oldMove = eventQueue.mouseMove(true);
				ushort	oldRepeat = eventQueue.getAutoRepeat();

				eventQueue.setAutoRepeat(2);

				while (event.what != evMouseUp) {
					switch (event.what) {
						case evMouseDown:
							if (!(event.mouse.modifiers & mdShift)
									&& !(flags & lsDisjointSelect)) {
								clearSelection();
								selectCell(cursor);
								refresh();
								}
							if (event.mouse.doubleClick) {
								message(owner,evBroadcast,cmListItemSelected,this);
								goto doneMouse;
								}
						case evMouseAuto:
						case evMouseMove:
							TPoint	loc;
							ushort	modifiers = mdShift;

							if (event.what == evMouseDown)
								modifiers = event.mouse.modifiers;
							if (flags & lsDisjointSelect)
								modifiers = 0;

							ushort pos = findCellHit(event.where,loc);
							if (pos == lsInside && event.what != evMouseAuto) {
								// Selection within the list

								if (loc == cursor && event.what == evMouseDown) {
									// Post a message to ensure updates
									// occur for the first mouse down event
									// in the list box.

									message(owner,evBroadcast,
										cmListCursorChanged,this);
									}

								if (loc.x < cursor.x)
									selectLeft(cursor.x - loc.x,modifiers);
								else if (loc.x > cursor.x)
									selectRight(loc.x - cursor.x,modifiers);

								if (loc.y < cursor.y)
									selectUp(cursor.y - loc.y,modifiers);
								else if (loc.y > cursor.y)
									selectDown(loc.y - cursor.y,modifiers);
								}
							else if (event.what == evMouseAuto) {
								// Auto selection outside window to scroll

								if (pos & lsAbove)
									selectUp(1,modifiers);
								if (pos & lsBelow)
									selectDown(1,modifiers);
								if (pos & lsLeft)
									selectLeft(1,modifiers);
								if (pos & lsRight)
									selectRight(1,modifiers);
								}
							if (event.what == evMouseDown
									&& flags & lsDisjointSelect) {
								toggleCell(cursor);
								refresh();
								}
							break;
						}
					getEvent(event);
					}

doneMouse:
				clearEvent(event);
				eventQueue.mouseMove(oldMove);
				eventQueue.setAutoRepeat(oldRepeat);
				}
			break;
		case evKeyDown:
		case evKeyAuto:
			if (range.isEmpty())
				return;

			switch (event.key.keyCode) {
				case kbSpace:
					if (event.what == evKeyAuto)
						break;
					if (flags & lsDisjointSelect) {
						toggleCell(cursor);
						refresh();
						}
					break;
				case kbLeft:
					selectLeft(1,event.key.modifiers);
					break;
				case kbRight:
					selectRight(1,event.key.modifiers);
					break;
				case kbUp:
					selectUp(1,event.key.modifiers);
					break;
				case kbDown:
					selectDown(1,event.key.modifiers);
					break;
				case kbHome:
					selectNext(lsLeft | lsAbove,range.bottom(),
						event.key.modifiers);
					break;
				case kbEnd:
					selectNext(lsRight | lsBelow,range.bottom(),
						event.key.modifiers);
					break;
				case kbPgUp:
					selectUp(visible.bottom()-visible.top(),
						event.key.modifiers,true);
					break;
				case kbPgDn:
					selectDown(visible.bottom()-visible.top(),
						event.key.modifiers,true);
				default:
					// Key press is not handled by us, so simply return.
					return;
				}
			clearEvent(event);
			break;
		case evBroadcast:
			if (options & ofSelectable) {
				if (event.message.command == cmScrollBarClicked &&
						(event.message.infoPtr == hScroll ||
						 event.message.infoPtr == vScroll))
					select();
				else if (event.message.command == cmScrollBarChanged) {
					if (range.isEmpty())
						return;
					if (event.message.infoPtr == hScroll)
						hScrollTo(hScroll->getValue());
					else if (event.message.infoPtr == vScroll)
						vScrollTo(vScroll->getValue());
					}
				}
			break;
		}
}

void TListBase::drawCell(int i,int j)
/****************************************************************************
*
* Function:		TListBase::drawCell
* Parameters:	i,j	- Location of the cell to draw
*
* Description:	Draws the cell, clipped to the bounds of the list. This
*				routine assumes that the list if entirely visible.
*
****************************************************************************/
{
	mouse.obscure();

	if (visible.includes(i,j)) {
		// Compute the bounding box for the cell

		short	dx = (i-visible.left()) * cellSize.x;
		short	dy = (j-visible.top()) * cellSize.y;

		TRect r(bounds.left()+_MVIS_sysLineWidth+dx,
				bounds.top()+_MVIS_sysLineWidth+dy,
				bounds.left()+_MVIS_sysLineWidth+dx+cellSize.x,
				bounds.top()+_MVIS_sysLineWidth+dy+cellSize.y);
        r &= TRect(bounds.left()+_MVIS_sysLineWidth,
				   bounds.top()+_MVIS_sysLineWidth,
				   bounds.right()-_MVIS_sysLineWidth,
				   bounds.bottom()-_MVIS_sysLineWidth);
		if (!r.isEmpty()) {
			MGL_setClipRect(r);
			drawCell(i,j,r);
			}
		}

	mouse.unobscure();
}

void TListBase::draw(const TRect& clip)
/****************************************************************************
*
* Function:		TListBase::draw
* Parameters:	clip	- Clipping rectangle to use when drawing
*
* Description:	Draws the representation of the list.
*
****************************************************************************/
{
	mouse.obscure();

	TRect	clipBounds(bounds.left()+_MVIS_sysLineWidth,
					   bounds.top()+_MVIS_sysLineWidth,
					   bounds.right()-_MVIS_sysLineWidth,
					   bounds.bottom()-_MVIS_sysLineWidth);
    clipBounds &= clip;

	// Clear the background for the list

	MGL_setClipRect(clip);
	MGL_setColor(getColor(2));
	MGL_fillRect(bounds);
	MGL_setColor(getColor(1));
	drawRect(bounds);

	// Draw each of the items in the list. Note that we set up the clip
	// rectangle to clip everything to the bounds of the list correctly,
	// and that we only draw those items that are visible and within the
	// range of selectable items.

    TRect   v(visible & range);
	TRect 	start(bounds.left()+_MVIS_sysLineWidth,
				  bounds.top()+_MVIS_sysLineWidth,
				  bounds.left()+_MVIS_sysLineWidth+cellSize.x,
				  bounds.top()+_MVIS_sysLineWidth+cellSize.y);

	for (int i = v.left(); i < v.right(); i++) {
		TRect r(start);
		for (int j = v.top(); j < v.bottom(); j++) {
            TRect c(r); c &= clipBounds;
			if (!c.isEmpty()) {
				MGL_setClipRect(c);
				drawCell(i,j,r);
				}
			r.top() += cellSize.y;	r.bottom() += cellSize.y;
			}
		start.left() += cellSize.x;	start.right() += cellSize.x;
		}

	if (range.isEmpty()) {
		// There is nothing to draw in the list (it is currently empty),
		// but if the list is selected then we should draw a dotted outline
		// around the first empty cell to ensure the user knows when the
		// list is selected.

		if ((state & sfFocused)) {
			attributes attr;
			MGL_getAttributes(&attr);
			MGL_setColor(getColor(3));
			MGL_setPenStyle(BITMAP_PATTERN_TRANSPARENT);
			MGL_setPenBitmapPattern(&GRAY_FILL);
			drawRectCoord(bounds.left()+_MVIS_sysLineWidth,
						  bounds.top()+_MVIS_sysLineWidth,
						  bounds.left()+_MVIS_sysLineWidth+cellSize.x,
						  bounds.top()+_MVIS_sysLineWidth+cellSize.y);
			MGL_restoreAttributes(&attr);
			}
		}

	mouse.unobscure();
}

void TListBase::refresh()
/****************************************************************************
*
* Function:		TListBase::refresh
*
* Description:	Refreshes the list by refreshing each of the visible cells.
*
****************************************************************************/
{
    TRect   v(visible & range);

	mouse.obscure();
	for (int i = v.left(); i < v.right(); i++)
		for (int j = v.top(); j < v.bottom(); j++)
			refreshCell(i,j);
	mouse.unobscure();
}

const syLineSize = 1;

void TListBase::update()
/****************************************************************************
*
* Function:		TListBase::update
*
* Description:	Forces an update for the entire list, by posting a repaint
*				event for the interior portion of the list.
*
****************************************************************************/
{
	PRECONDITION(owner != NULL);
	TRect	r(bounds);
	r.inset(syLineSize,syLineSize);
	owner->repaint(r);
}

void TListBase::vScrollTo(int j,bool redraw)
/****************************************************************************
*
* Function:		TListBase::vScrollTo
* Parameters:	j		- New cell to scroll to in vertical direction
*				redraw	- True if list should be redrawn
*
* Description:	Scrolls the list to start at the new cell 'j' in the
*				vertical direction, redrawing all necessary cells to do
*				this.
*
]****************************************************************************/
{
	if (visible.top() != j) {
		// Setup the viewport for redrawing the list directly, and redraw
		// it.

        TRect   v(visible & range);
		TRect	b(bounds.left()+_MVIS_sysLineWidth,
				  bounds.top()+_MVIS_sysLineWidth,
				  bounds.right()-_MVIS_sysLineWidth,
				  bounds.bottom()-_MVIS_sysLineWidth);
		int 	dy = (v.top() - j) * cellSize.y;

		visible.moveTo(hScroll ? hScroll->getValue() : 0,j);
		vScroll->setValue(j);

		if (redraw) {
			PRECONDITION(owner != NULL);
			setupOwnerViewport();
			mouse.obscure();
			int page = MGL_getActivePage();

			if (j < v.top()) {
                b.bottom() -= dy;
				if (!b.isEmpty())
					MGL_copyImage(b,b.left(),
						bounds.top()+_MVIS_sysLineWidth+dy,page,page);
				for (int y = j; y < v.top(); y++)
					for (int x = v.left(); x < v.right(); x++)
						drawCell(x,y);
				}
			else {
				b.top() -= dy;
				if (!b.isEmpty())
					MGL_copyImage(b,b.left(),bounds.top()+_MVIS_sysLineWidth,
						page,page);
				j -= v.top();
				for (int y = 0; y < j; y++)
					for (int x = v.left(); x < v.right(); x++)
						drawCell(x,v.bottom()+y);
				}
			mouse.unobscure();
			resetViewport();
			}
		}
}

void TListBase::hScrollTo(int i,bool redraw)
/****************************************************************************
*
* Function:		TListBase::hScrollTo
* Parameters:	i		- New cell to scroll to in horizontal direction
*				redraw	- True if list should be redrawn
*
* Description:	Scrolls the list to start at the new cell 'i' in the
*				horizontal direction, redrawing all necessary cells to do
*				this.
*
****************************************************************************/
{
	if (visible.left() != i) {
		// Setup the viewport for redrawing the list directly, and redraw
		// it.

        TRect   v(visible & range);
		TRect	b(bounds.left()+_MVIS_sysLineWidth,
				  bounds.top()+_MVIS_sysLineWidth,
				  bounds.right()-_MVIS_sysLineWidth,
				  bounds.bottom()-_MVIS_sysLineWidth);
		int 	dx = (v.left() - i) * cellSize.x;

		visible.moveTo(i,vScroll ? vScroll->getValue() : 0);
		hScroll->setValue(i);

		if (redraw) {
			PRECONDITION(owner != NULL);
			setupOwnerViewport();
			mouse.obscure();
			int page = MGL_getActivePage();

			if (i < v.left()) {
				b.right() -= dx;
				if (!b.isEmpty())
					MGL_copyImage(b,bounds.left()+_MVIS_sysLineWidth+dx,
						b.top(),page,page);
				for (int y = v.top(); y < v.bottom(); y++)
					for (int x = i; x < v.left(); x++)
						drawCell(x,y);
				}
			else {
				b.left() -= dx;
				if (!b.isEmpty())
					MGL_copyImage(b,bounds.left()+_MVIS_sysLineWidth,
						b.top(),page,page);
				i -= v.left();
				for (int y = v.top(); y < v.bottom(); y++)
					for (int x = 0; x < i; x++)
						drawCell(v.right() + x,y);
				}
			mouse.unobscure();
			resetViewport();
			}
		}
}

void TListBase::setCurrentCell(int i,int j)
/****************************************************************************
*
* Function:		TListBase::setCurrentCell
* Parameters:	i,j	- Location of cell to set
*
* Description:	Sets the cell (i,j) to be the currently active cell.
*
****************************************************************************/
{
	cursor.x = i;
	cursor.y = j;
	selection.topLeft = cursor;
	selection.right() = selection.left()+1;
	selection.bottom() = selection.top()+1;
	focusCurrent(true,false);
}

void TListBase::setHScroll(TScrollBar *h)
/****************************************************************************
*
* Function:		TListBase::setHScroll
* Parameters:	h	- Pointer to the new scroll bar for the list
*
* Description:	Sets the horizontal scroll bar for the list.
*
****************************************************************************/
{
	hScroll = h;
	setHRange(0,0);
	if (hScroll) {
		hScroll->setPageStep(max(1,visible.right() - visible.left()-1));
		hScroll->setArrowStep(1);
		}
}

void TListBase::setVScroll(TScrollBar *v)
/****************************************************************************
*
* Function:		TListBase::setVScroll
* Parameters:	v	- Pointer to the new scroll bar for the list
*
* Description:	Sets the vertical scroll bar for the list.
*
****************************************************************************/
{
	vScroll = v;
	setVRange(0,0);
	if (vScroll) {
		vScroll->setPageStep(max(1,visible.bottom() - visible.top()-1));
		vScroll->setArrowStep(1);
		}
}

void TListBase::setHRange(short min,short max)
/****************************************************************************
*
* Function:		TListBase::setHRange
* Parameters:	min		- Minimum value for the range
*				max		- Maximum value for the range
*
* Description:	Sets the horizontal range for the list, by setting the
*				range of the horizontal scroll bar. The scrollbar value
*				is set to min.
*
****************************************************************************/
{
	if (hScroll) {
		range.left() = min;
		range.right() = max+1;
		if ((max -= (visible.right() - visible.left()-1)) < min)
			max = min;
		hScroll->setRange(min,max);
		hScroll->setValue(min);
		}
}

void TListBase::setVRange(short min,short max)
/****************************************************************************
*
* Function:		TListBase::setVRange
* Parameters:	min		- Minimum value for the range
*				max		- Maximum value for the range
*
* Description:	Sets the vertical range for the list, by setting the
*				range of the vertical scroll bar. The scrollbar value
*				is set to min.
*
****************************************************************************/
{
	if (vScroll) {
		range.top() = min;
		range.bottom() = max+1;
		if ((max -= (visible.bottom() - visible.top()-1)) < min)
			max = min;
		vScroll->setRange(min,max);
		vScroll->setValue(min);
		}
}

TPalette& TListBase::getPalette() const
/****************************************************************************
*
* Function:		TListBase::getPalette
* Returns:		Pointer to the standard palette for List Boxes.
*
****************************************************************************/
{
	static char cpListBase[] = {2,29,30,31,32,33};
	static TPalette palette(cpListBase,sizeof(cpListBase));
	return palette;
}

TList::TList(const TRect& bounds,const TPoint& cellSize,
	const TRect& dataBounds,TScrollBar *vScroll,TScrollBar *hScroll,
	ushort flags)
	: TListBase(bounds,cellSize,vScroll,hScroll,flags),
	  dataBounds(dataBounds)
/****************************************************************************
*
* Function:		TList::TList
* Parameters:	bounds		- Bounding box for the list
*				dataBounds	- Bounds on number of items in the list
*				cellSize	- Dimensions in pixels for each list cell
*				vScroll		- Pointer to horizontal scroll bar (NULL if none)
*				hScroll		- Pointer to vertical scroll bar (NULL if none)
*				flags		- Creation flags
*
* Description:	Constructor for the TList class, for manipulating
*				scrollable lists of text strings.
*
****************************************************************************/
{
	setDataBounds(dataBounds);
}

void TList::clearItems()
/****************************************************************************
*
* Function:		TList::clearItems
*
* Description:	Clears all of the items in the array, to point to nothing.
*
****************************************************************************/
{
	for (int i = 0; i < cells.numberOfItems(); i++) {
		cells[i].text = NULL;
		cells[i].flags = 0;
		}
	setHRange(dataBounds.left(),dataBounds.left());
	setVRange(dataBounds.top(),dataBounds.top());
}

void TList::setDataBounds(const TRect& bounds)
/****************************************************************************
*
* Function:		TList::setDataBounds
* Parameters:	bounds	- New bounds on the number of items in the list
*
* Description:	When this is called, the array is first emptied of all
*				data, then re-sized to fit the new bounds. All data in
*				the list is effectively lost, so will need to be
*				re-inserted.
*
****************************************************************************/
{
	dataBounds = bounds;
	visible.moveTo(dataBounds.topLeft);
	TPoint size(bounds.botRight - bounds.topLeft);
	cells.setCount(max(size.x * size.y,1));
	clearItems();
}

void TList::drawCell(int i,int j,const TRect& bounds)
/****************************************************************************
*
* Function:		TList::drawCell
* Parameters:	i,j		- Index of the cell to draw
*				bounds	- Bounding box to draw the item in
*
* Description:	Draws the cell item within the specified bounds. Note that
*				the appropriate clipping rectangle will already have been
*				set up before this routine is called.
*
****************************************************************************/
{
	CellItem&	cell = findCell(i,j);

	if (cell.text) {
		// If the items is selected and this is the focused view, highlight
		// the item, otherwise clear the background for the item.

		MGL_setColor(getColor(cell.flags & lsSelected ? 5 : 2));
		MGL_fillRect(bounds);

		// Draw the text for the item

		fontManager.useFont(fmSystemFont);
		MGL_setColor(getColor(cell.flags & lsSelected ? 4 : 3));
		MGL_drawStrXY(bounds.left() + INDENTLEFT,
					  bounds.top() + INDENTTOP,cell.text);

		// If the cursor is on the item, and view is focused then draw
		// a dotted outline around the cell.

		if ((state & sfFocused) && cursor.x == i && cursor.y == j) {
			attributes attr;
			MGL_getAttributes(&attr);

			MGL_setColor(getColor(cell.flags & lsSelected ? 6 : 3));
			MGL_setPenStyle(BITMAP_PATTERN_TRANSPARENT);
			MGL_setPenBitmapPattern(&GRAY_FILL);
			drawRect(bounds);
			MGL_restoreAttributes(&attr);
			}
		}
	cell.flags &= ~lsDirty;
}

void TList::refreshCell(int i,int j)
/****************************************************************************
*
* Function:		TList::refreshCell
* Parameters:	i,j	- Index of the cell to refresh
*
* Description:	Refreshes the indexed cell if it is dirty.
*
****************************************************************************/
{
	if (findCell(i,j).flags & lsDirty || (cursor.x == i && cursor.y == j))
		TListBase::drawCell(TPoint(i,j));
}

void TList::selectCell(int i,int j)
/****************************************************************************
*
* Function:		TList::selectCell
* Parameters:	i,j		- Index of the cell to select
*
* Description:	Sets the selected flag for the item. If the cell was not
*				already selected, we set the dirty bit for the cell.
*
****************************************************************************/
{
	CellItem& cell = findCell(i,j);

	if (!(cell.flags & lsSelected))
		cell.flags |= lsDirty;
	cell.flags |= lsSelected;
}

void TList::deselectCell(int i,int j)
/****************************************************************************
*
* Function:		TList::deselectCell
* Parameters:	i,j		- Index of the cell to de-select
*
* Description:	Clears the selected flags for the item. If the cell was
*				already selected, we set the dirty bit for the cell.
*
****************************************************************************/
{
	CellItem& cell = findCell(i,j);

	if (cell.flags & lsSelected)
		cell.flags |= lsDirty;
	cell.flags &= ~lsSelected;
}

void TList::toggleCell(int i,int j)
/****************************************************************************
*
* Function:		TList::toggleCell
* Parameters:	i,j		- Index of cell to toggle
*
* Description:	Toggles the selection flags for the specified cell.
*
****************************************************************************/
{
	CellItem& cell = findCell(i,j);;

	cell.flags ^= lsSelected;
	cell.flags |= lsDirty;
}

void TList::dirtyCell(int i,int j)
/****************************************************************************
*
* Function:		TList::dirtyCell
* Parameters:	i,j		- Index of the cell to dirty
*
* Description:	Sets the dirty bit for the specified cell.
*
****************************************************************************/
{
	findCell(i,j).flags |= lsDirty;
}

void TList::setCell(int i,int j,const char *text)
/****************************************************************************
*
* Function:		TList::setCell
* Parameters:	i,j		- Index of the cell whose text to set
*				text	- Pointer to the text for the cell
*
* Description:	Sets the text for the cell. If this is NULL, nothing is
*				drawn in the cell.
*
****************************************************************************/
{
	CellItem& cell = findCell(i,j);

	cell.text = text;
	cell.flags |= lsDirty;
}

bool TList::getCell(int i,int j,const char*& text)
/****************************************************************************
*
* Function:		TList::getCell
* Parameters:	i,j		- Index of the cell whose text to set
*				text	- Place to store the text for the cell
* Returns:		True if the cell is selected, false if not.
*
* Description:  Finds the text of a cell, returning true if the cell is
*				selected.
*
****************************************************************************/
{
	CellItem& cell = findCell(i,j);

	text = cell.text;
	return (cell.flags & lsSelected);
}
