/****************************************************************************
*
*					 MegaVision Application Framework
*
*			A C++ GUI Toolkit for the MegaGraph Graphics Library
*
*					Copyright (C) 1994 SciTech Software.
*							All rights reserved.
*
* Filename:		$RCSfile: tscrollb.cpp $
* Version:		$Revision: 1.2 $
*
* Language:		C++ 3.0
* Environment:	IBM PC (MS DOS)
*
* Description:	Member functions for the TScrollBar class hierachy. You may
*				make the scroll bar's as wide as you wish, but the standard
*				scroll bar width is 14 pixels.
*
* $Id: tscrollb.cpp 1.2 1994/03/09 11:50:49 kjb Exp $
*
****************************************************************************/

#include "mvision.hpp"

#pragma	hdrstop

#include "tscrollb.hpp"
#include "tgroup.hpp"
#include "tmouse.hpp"

/*----------------------------- Implementation ----------------------------*/

TScrollBar::TScrollBar(const TRect& bounds)
	: TView(bounds), flags(0)
/****************************************************************************
*
* Function:		TScrollBar::TScrollBar
* Parameters:	bounds	- Bounding box for the scroll bar
*
****************************************************************************/
{
	minVal = maxVal = value = 0;
	pageStep = arrowStep = 1;

	minVal = 100;
	maxVal = 200;
	value = 150;
	setBounds(bounds);
}

void TScrollBar::moveTo(int x,int y)
/****************************************************************************
*
* Function:		TScrollBar::moveTo
* Parameters:	x,y	- Position to move the scroll bar to
*
****************************************************************************/
{
	TView::moveTo(x,y);
	setBounds(bounds);
}

void TScrollBar::setBounds(const TRect& bounds)
/****************************************************************************
*
* Function:		TScrollBar::setBounds
* Parameters:	bounds	- New bounding box for the scroll bar
*
* Description:	Changes the bounds for the scroll bar.
*
****************************************************************************/
{
	TView::setBounds(bounds);
	vertical = (size.x < size.y);

	// Compute the location of the left and right arrow buttons

	leftArrow = bounds;
	leftArrow.inset(_MVIS_sysLineWidth,_MVIS_sysLineWidth);
	rightArrow = leftArrow;

	if (vertical) {
		leftArrow.bottom() = leftArrow.top()+(_MVIS_sysScrollBarWidth+1);
		rightArrow.top() = rightArrow.bottom()-(_MVIS_sysScrollBarWidth+1);
		}
	else {
		leftArrow.right() = leftArrow.left()+(_MVIS_sysScrollBarWidth+1);
		rightArrow.left() = rightArrow.right()-(_MVIS_sysScrollBarWidth+1);
		}

	// Compute the location of the thumb

	moveThumb();
}

void TScrollBar::moveThumb()
/****************************************************************************
*
* Function:		TScrollBar::moveThumb
*
* Description:	Computes the current location of the thumb, given the
*				current value of the scroll bar.
*
****************************************************************************/
{
	int	left,right;

	if (vertical) {
		thumb.left() = leftArrow.left();
		thumb.right() = leftArrow.right();
		left = thumb.top() = leftArrow.bottom();
		right = thumb.bottom() = rightArrow.top();
		}
	else {
		thumb.top() = leftArrow.top();
		thumb.bottom() = leftArrow.bottom();
		left = thumb.left() = leftArrow.right();
		right = thumb.right() = rightArrow.left();
		}

	if (maxVal == minVal)
		return;

	left += ((right-left-(_MVIS_sysScrollBarWidth+1)) * (long)(value - minVal))
					/ (long)(maxVal-minVal);
	right = left + (_MVIS_sysScrollBarWidth+1);

	if (vertical) {
		thumb.top() = left;
		thumb.bottom() = right;
		}
	else {
		thumb.left() = left;
		thumb.right() = right;
		}
}

int TScrollBar::getPartHit(const TPoint& global)
/****************************************************************************
*
* Function:		TScrollBar::getPartHit
* Parameters:	global	- Point to test for inclusion (global coords)
* Returns:		Code of part hit, -1 if none hit
*
* Description:	Determines which part of the scroll bar was hit.
*
****************************************************************************/
{
	Point p(global);
	globalToLocal(p);

	if (bounds.includes(p)) {
		if (leftArrow.includes(p))
			return sbLeftArrow;
		if (rightArrow.includes(p))
			return sbRightArrow;

		if (thumb.includes(p))
			return sbThumb;

		if (vertical) {
			if (leftArrow.bottom() <= p.y && p.y < thumb.top())
				return sbPageLeft;
			if (thumb.bottom() <= p.y && p.y < rightArrow.top())
				return sbPageRight;
			}
		else {
			if (leftArrow.right() <= p.x && p.x < thumb.left())
				return sbPageLeft;
			if (thumb.right() <= p.x && p.x < rightArrow.left())
				return sbPageRight;
			}
		}

	return -1;
}

void TScrollBar::changeValue(int part)
/****************************************************************************
*
* Function:		TScrollBar::changeValue
* Parameters:	part	- Scroll bar arrow hit
*
* Description:	Adjusts the scroll bar value depending on which part of
*				the scroll bar was hit.
*
****************************************************************************/
{
	short	oldValue = value;

	switch (part) {
		case sbLeftArrow:
			value -= arrowStep;
			break;
		case sbRightArrow:
			value += arrowStep;
			break;
		case sbPageLeft:
			value -= pageStep;
			break;
		case sbPageRight:
			value += pageStep;
			break;
		}
	if (value < minVal)	value = minVal;
	if (value > maxVal)	value = maxVal;

	if (oldValue != value) {
		moveThumb();
		drawThumb(part);

		// We need to redraw the opposite arrow if the arrow just became
		// active again.

		if (oldValue == minVal)
			drawLeftArrow(false);
		if (oldValue == maxVal)
			drawRightArrow(false);

		message(owner,evBroadcast,cmScrollBarChanged,this);
		}
}

void TScrollBar::changeValue(const TPoint& global)
/****************************************************************************
*
* Function:		TScrollBar::changeValue
* Parameters:	global	- New location of the mouse cursor
*
* Description:	Converts the mouse cursor location into a new value for
*				the scroll bar thumb, repositions the thumb and redraws
*				it.
*
****************************************************************************/
{
	Point p(global);
	globalToLocal(p);

	int		left,right,start;
	short	oldValue = value;

	if (vertical) {
		start = p.y - 7;
		left = leftArrow.bottom();
		right = rightArrow.top();
		}
	else {
		start = p.x - 7;
		left = leftArrow.right();
		right = rightArrow.left();
		}

	if (start < left)		start = left;
	if (start > right-(_MVIS_sysScrollBarWidth+1))
		start = right-(_MVIS_sysScrollBarWidth+1);

	value = ((2*(maxVal-minVal)*(long)(start-left))
			/ (right-left-(_MVIS_sysScrollBarWidth+1))+1)/2;

	if (oldValue != value) {
		moveThumb();
		drawThumb(sbThumb);
		message(owner,evBroadcast,cmScrollBarChanged,this);
		}
}

void TScrollBar::handleEvent(TEvent& event,phaseType phase)
/****************************************************************************
*
* Function:		TScrollBar::handleEvent
* Parameters:	event	- Event to handle
*				phase	- Current phase for the event (pre,focus,post)
*
* Description:	Event handling routine for scroll bars.
*
****************************************************************************/
{
	TView::handleEvent(event,phase);

	flags |= sbInteracting;

	if (event.what == evMouseDown) {
		// Let the owning view know that the scroll bar has been clicked,
		// so that any associated lists etc can select themselves and
		// prepare for scroll bar changed events.

		message(owner,evBroadcast,cmScrollBarClicked,this);
		TEvent e;
		getEvent(e,evRepaint);				// Force repaint

		int part = getPartHit(event.where);
		if (part == -1)
			return;

		bool 	oldMove = eventQueue.mouseMove(true);
		ushort	oldRepeat = eventQueue.getAutoRepeat();
		bool 	down,done = false;

		eventQueue.setAutoRepeat(1);
		MGL_setClipRect(bounds);

		if ((part == sbLeftArrow && value == minVal) ||
			(part == sbRightArrow && value == maxVal) ||
			(minVal == maxVal)) {
			beep();
			return;
			}
		drawPart(part,down = true);
		while (!done) {
			switch (event.what) {
				case evMouseDown:
				case evMouseAuto:
					if (part == getPartHit(event.where))
						changeValue(part);
					break;
				case evMouseUp:
					done = true;
					break;
				case evMouseMove:
					if (part != sbThumb) {
						if (down != (part == getPartHit(event.where)))
							drawPart(part,down = !down);
						}
					break;
				}
			if (part == sbThumb) {
				// Readjust the position of the thumb depending on where the
				// mouse is. Polling is better than using mouse movement
				// events, since we get a lot of movement events which can
				// trigger expensive redraw operations.

				TPoint	pos;
				mouse.pos(pos);
				changeValue(pos);
				}
			else if (part != sbThumb && (value == minVal || value == maxVal))
				done = true;
			getEvent(event);
			}
		if (down)
			drawPart(part,false);

		// Redraw the arrows if they have changed state during the
		// interaction, and we were adjusting the paging state.

		if (part == sbPageLeft || part == sbPageRight || part == sbThumb) {
			if (value == minVal)
				drawLeftArrow(false);
			else if (value == maxVal)
				drawRightArrow(false);
			}

		clearEvent(event);
		eventQueue.mouseMove(oldMove);
		eventQueue.setAutoRepeat(oldRepeat);
		}

	flags &= ~sbInteracting;
}

void TScrollBar::drawPart(int part,bool down)
/****************************************************************************
*
* Function:		TScrollBar::drawPart
* Parameters:	part	- Code of the part to draw
*				down	- True if part is down
*
* Description:	Draws the part in the state 'down'.
*
****************************************************************************/
{
	switch (part) {
		case sbLeftArrow:
			drawLeftArrow(down);
			break;
		case sbRightArrow:
			drawRightArrow(down);
			break;
		case sbPageLeft:
		case sbPageRight:
		case sbThumb:
			drawThumb(down ? part : -1);
			break;
		}
}

void TScrollBar::drawLeftArrow(bool down)
/****************************************************************************
*
* Function:		TScrollBar::drawLeftArrow
* Parameters:	down	- True if arrow is pressed down
*
* Description:	Draw's the left/top arrow portion of the scroll bar. Assumes
*				the outer border for the scroll bar exists, and the
*				border colors are set up.
*
****************************************************************************/
{
	mouse.obscure();
	MGL_setColor(getColor(3));
	MGL_fillRect(leftArrow);
	drawBorder(leftArrow,down ? BDR_INSET : BDR_OUTSET,1);
	MGL_setColor(getColor(value > minVal ? 4 : 5));

	int cx = down + (leftArrow.left() + leftArrow.right()-1)/2;
	int cy = down + (leftArrow.top() + leftArrow.bottom()-1)/2;

	if (vertical) {
		// Draw the top arrow

		MGL_lineCoord(cx-3,cy+1,cx,cy-2);
		MGL_lineCoord(cx+1,cy-2,cx+4,cy+1);
		MGL_lineCoord(cx-2,cy+1,cx,cy-1);
		MGL_lineCoord(cx+1,cy-1,cx+3,cy+1);
		}
	else {
		// Draw the left arrow

		MGL_lineCoord(cx+1,cy-3,cx-2,cy);
		MGL_lineCoord(cx-2,cy+1,cx+1,cy+4);
		MGL_lineCoord(cx+1,cy-2,cx-1,cy);
		MGL_lineCoord(cx-1,cy+1,cx+1,cy+3);
		}
	mouse.unobscure();
	flags &= ~sbDirtyLeft;
}

void TScrollBar::drawRightArrow(bool down)
/****************************************************************************
*
* Function:		TScrollBar::drawRightArrow
* Parameters:	down	- True if arrow is pressed down
*
* Description:	Draw's the right/bottom arrow portion of the scroll bar. Assume
*				the outer border for the scroll bar exists, and the
*				border colors are set up.
*
****************************************************************************/
{
	mouse.obscure();
	MGL_setColor(getColor(3));
	MGL_fillRect(rightArrow);
	drawBorder(rightArrow,down ? BDR_INSET : BDR_OUTSET,1);
	MGL_setColor(getColor(value < maxVal ? 4 : 5));

	int cx = down + (rightArrow.left() + rightArrow.right()-1)/2;
	int cy = down + (rightArrow.top() + rightArrow.bottom()-1)/2;

	if (vertical) {
		// Draw the bottom arrow

		MGL_lineCoord(cx-3,cy-1,cx,cy+2);
		MGL_lineCoord(cx+1,cy+2,cx+4,cy-1);
		MGL_lineCoord(cx-2,cy-1,cx,cy+1);
		MGL_lineCoord(cx+1,cy+1,cx+3,cy-1);
		}
	else {
		// Draw the right arrow

		MGL_lineCoord(cx-1,cy-3,cx+2,cy);
		MGL_lineCoord(cx+2,cy+1,cx-1,cy+4);
		MGL_lineCoord(cx-1,cy-2,cx+1,cy);
		MGL_lineCoord(cx+1,cy+1,cx-1,cy+3);
		}
	mouse.unobscure();
	flags &= ~sbDirtyRight;
}

void TScrollBar::drawThumb(int which)
/****************************************************************************
*
* Function:		TScrollBar::drawThumb
* Parameters:	which	- Which part of bar to highlight.
*
* Description:	Draws the thumb portion of the scroll bar, highlighting
*				the specific portion of the scroll bar.
*
****************************************************************************/
{
	if (maxVal == minVal) {
		mouse.obscure();
		MGL_setColor(getColor(3));
		MGL_fillRect(thumb);
		mouse.unobscure();
		return;
		}

	attributes attr;
	MGL_getAttributes(&attr);

	mouse.obscure();
	MGL_setColor(getColor(3));
	MGL_setPenStyle(BITMAP_PATTERN_OPAQUE);
	MGL_setPenBitmapPattern(&GRAY_FILL);

	if (vertical) {
		MGL_setBackColor(getColor(which == sbPageLeft ? 7 : 6));
		MGL_fillRectCoord(thumb.left(),leftArrow.bottom(),
			thumb.right(),thumb.top());
		MGL_setBackColor(getColor(which == sbPageRight ? 7 : 6));
		MGL_fillRectCoord(thumb.left(),thumb.bottom(),
			thumb.right(),rightArrow.top());
		}
	else {
		MGL_setBackColor(getColor(which == sbPageLeft ? 7 : 6));
		MGL_fillRectCoord(leftArrow.right(),thumb.top(),
			thumb.left(),thumb.bottom());
		MGL_setBackColor(getColor(which == sbPageRight ? 7 : 6));
		MGL_fillRectCoord(thumb.right(),thumb.top(),
			rightArrow.left(),thumb.bottom());
		}

	MGL_setPenStyle(SOLID_PATTERN);
	MGL_fillRect(thumb);
	drawBorder(thumb,(which == sbThumb ? BDR_INSET : BDR_OUTSET),1);

	MGL_restoreAttributes(&attr);
	mouse.unobscure();

	flags &= ~sbDirtyThumb;
}

void TScrollBar::draw(const TRect& clip)
/****************************************************************************
*
* Function:		TScrollBar::draw
* Parameters:	clip	- Clipping rectangle to use when drawing
*
* Description:	Draws the scroll bar in the current state.
*
****************************************************************************/
{
	if (flags & sbInteracting)
		return;

	MGL_setClipRect(clip);

	// Draw the main body of the scroll bar inset into the screen

	MGL_setBorderColors(getColor(1),getColor(2));
	drawBorder(bounds,BDR_INSET,1);

	drawLeftArrow(false);
	drawRightArrow(false);
	drawThumb(-1);
}

void TScrollBar::update()
/****************************************************************************
*
* Function:		TScrollBar::update
*
* Description:	Updates the state of the scroll bar thumb, if the scroll
*				bar is visible.
*
****************************************************************************/
{
	if ((flags & sbInteracting) || !(flags & sbDirty))
		return;

	if ((state & sfVisible) && (state & sfExposed)) {
		setupOwnerViewport();
		if (flags & sbDirtyLeft)
			drawLeftArrow(false);
		if (flags & sbDirtyRight)
			drawRightArrow(false);
		if (flags & sbDirtyThumb)
			drawThumb(-1);
		resetViewport();
		}
}

void TScrollBar::setValue(short v)
/****************************************************************************
*
* Function:		TScrollBar::setValue
* Parameters:	v	- New value for the scroll bar
*
* Description:	Sets the value for the scroll bar, and updates it.
*
****************************************************************************/
{
	if (value == v)
		return;
	if (value < minVal)	value = minVal;		// Clamp to specified range
	if (value > maxVal)	value = maxVal;
	if (value == minVal || v == minVal)
		flags |= sbDirtyLeft;
	if (value == maxVal || v == maxVal)
		flags |= sbDirtyRight;
	flags |= sbDirtyThumb;
	value = v;
	moveThumb();
	update();
}

void TScrollBar::setMinVal(short v)
/****************************************************************************
*
* Function:		TScrollBar::setMinVal
* Parameters:	v	- New minimum value for the scroll bar
*
* Description:	Sets the minimum value for the scroll bar.
*
****************************************************************************/
{
	minVal = v;
	if (value < minVal)	value = minVal;
	moveThumb();
	if (owner)
		owner->repaint(bounds);
}

void TScrollBar::setMaxVal(short v)
/****************************************************************************
*
* Function:		TScrollBar::setMaxVal
* Parameters:	v	- New maximum value for the scroll bar
*
* Description:	Sets the maximum value for the scroll bar.
*
****************************************************************************/
{
	maxVal = v;
	if (value > maxVal)	value = maxVal;
	moveThumb();
	if (owner)
		owner->repaint(bounds);
}

void TScrollBar::setRange(short min,short max)
/****************************************************************************
*
* Function:		TScrollBar::setRange
* Parameters:	min	- New minimum value for scroll bar
*				max	- New maximum value for scroll bar
*
* Description:	Sets the minimum and maximum values for the scroll bar.
*
****************************************************************************/
{
	minVal = min;
	maxVal = max;
	if (value < minVal)	value = minVal;
	if (value > maxVal)	value = maxVal;
	moveThumb();
	if (owner)
		owner->repaint(bounds);
}

TPalette& TScrollBar::getPalette() const
/****************************************************************************
*
* Function:		TScrollBar::getPalette
* Returns:		Pointer to the standard palette for scrollbar's
*
****************************************************************************/
{
	static char cpScrollBar[] = {1,2,12,13,14,15,16};
	static TPalette palette(cpScrollBar,sizeof(cpScrollBar));
	return palette;
}
