/****************************************************************************
*
*					 MegaVision Application Framework
*
*			A C++ GUI Toolkit for the MegaGraph Graphics Library
*
*                   Copyright (C) 1994 SciTech Software.
*							All rights reserved.
*
* Filename:		$RCSfile: tview.hpp $
* Version:		$Revision: 1.2 $
*
* Language:		C++ 3.0
* Environment:	IBM PC (MS DOS)
*
* Description:	Header file for the TView class. This class is the base
*				class for user interface classes in the library.
*
* $Id: tview.hpp 1.2 1994/03/09 11:50:25 kjb Exp $
*
****************************************************************************/

#ifndef	__TVIEW_HPP
#define	__TVIEW_HPP

#ifndef	__MVISION_HPP
#include "mvision.hpp"
#endif

#ifndef	__TRECT_HPP
#include "trect.hpp"
#endif

#ifndef	__TPOINT_HPP
#include "tpoint.hpp"
#endif

#ifndef	__TEVENT_HPP
#include "tevent.hpp"
#endif

#ifndef	__DLIST_HPP
#include "dlist.hpp"
#endif

#ifndef	__TPALETTE_HPP
#include "tpalette.hpp"
#endif

/*------------------------------ Constants --------------------------------*/

const ushort

// TView state masks

	sfVisible		= 0x001,		// Set if the view is visible
	sfActive		= 0x002,		// Set if the view is active
	sfFocused		= 0x004,		// Set if the view is focused
	sfDisabled		= 0x008,		// Set if the view is disabled
	sfModal			= 0x010,		// Set if the view is modal
	sfExposed		= 0x020,		// Set if the view is exposed
	sfLockRepaint	= 0x040,		// Set if repaints are locked

// TView option masks

	ofSelectable	= 0x001,		// The view can be selected
	ofFirstClick	= 0x002,		// First mouse click is processed
	ofPreProcess	= 0x004,		// View recieves pre-focused events
	ofPostProcess	= 0x008,		// View recieves post-focused events
	ofCenterX		= 0x010,		// View is centered in X
	ofCenterY		= 0x020,		// View is centered in Y
	ofCentered		= 0x030,		// View is centered in X and Y

// TView drag mode masks

	dmMoveable		= 0x01,			// Allows the view to move
	dmLimitLoX		= 0x02,			// Lefthand side is limited in movement
	dmLimitHiX		= 0x04,			// Righthand side is limited in movement
	dmLimitLoY		= 0x08,			// Top side is limited in movement
	dmLimitHiY		= 0x10,			// Bottom side is limited in movement
	dmLimitAll		= dmLimitLoX | dmLimitLoY | dmLimitHiX | dmLimitHiY,

// TView grow mode masks

	gmAnchorLoX		= 0x01,			// Lefthand side cannot move
	gmAnchorHiX		= 0x02,			// Righthand side cannot move
	gmAnchorLoY		= 0x04,			// Top side cannot move
	gmAnchorHiY		= 0x08,			// Bottom side cannot move
	gmGrowRel		= 0x10;			// Keep relative side to owner

/*-------------------------- Class definitions ----------------------------*/

class TGroup;

//---------------------------------------------------------------------------
// This class is the base class for all user interface classes in the
// library, providing the necessary interface common to all views. Note
// that all views are derived from ListNode, so all views can be placed
// onto a linked list.
//---------------------------------------------------------------------------

class TView : public DListNode {
protected:
	TRect	bounds;				// Bounding rectangle for the view
	TPoint	size;				// Size of the view
	TGroup	*owner;				// TGroup that owns this view
	ushort	state;				// State of the view
	ushort	options;			// Option flags for the view
	uchar	dragMode;			// Drag mode for view
	uchar	growMode;			// Grow mode for view
	ushort	helpCtx;			// Help context number for the view

			// Phase of event handling for focused events
	enum 	phaseType {phFocused,phPreProcess,phPostProcess};

			// Convert a coordinate from global to local coords
			void globalToLocal(Point& p) const	{ MGL_globalToLocal(&p); };
			void localToGlobal(Point& p) const	{ MGL_localToGlobal(&p); };

			// Draw a rectangle with the current system line width settings.
			// The rectangle will be completely bounded by the rectangle.
			void drawRectCoord(int left,int top,int right,int bottom, int width = 1);
			void drawRect(const TRect& r, int width = 1)
			{ drawRectCoord(r.left(),r.top(),r.right(),r.bottom()); };

			// Draw a border with the current system line width settings
			void drawBorderCoord(int left,int top,int right,int bottom,int style, int width)
			{ MGL_drawBorderCoord(left,top,right,bottom,style,width * _MVIS_sysLineWidth); };
			void drawBorder(const TRect& r,int style, int width)
			{ MGL_drawBorder(r,style,width * _MVIS_sysLineWidth); };

			// Draw a line with the current system line width settings
			void drawLineCoord(int left,int top,int right,int bottom, int width = 1);

public:
			// Constructor
			TView(const TRect& bounds);

			// Virtual destructor
	virtual	~TView();

			// Virtual to determine if the view was constructed correctly
	virtual	bool valid(ushort command);

			// Method to clear the event, flagging it as handled
			void clearEvent(TEvent& event)
				{ event.what = evNothing; event.message.infoPtr = this; };

			// Virtual to handle an event
	virtual	void handleEvent(TEvent& event,phaseType phase = phFocused);

			// Methods to retrieve and post events - eventually calls
			// eventQueue.xxx() if not overridden
	virtual	void putEvent(TEvent& event);
	virtual	bool getEvent(TEvent& event,ushort mask = evAll);
	virtual	bool peekEvent(TEvent& event,ushort mask = evAll);

			// Methods to draw the view
	virtual	void draw(const TRect& clip) = 0;
	virtual	void redraw();

			// Setup/reset the viewport relative to owners viewport
			void setupViewport() const;
			void setupOwnerViewport() const;

			void resetViewport() const	{ MGL_popViewport(); };

			// Methods to show and hide the view
			void show();
			void hide();

			// Methods to enable and disable the view
			void enable();
			void disable();

			// Methods to lock/unlock repaint events
			void lock()		{ state |= sfLockRepaint; };
			void unlock()	{ state &= ~sfLockRepaint; };

			// Method to set the bounds of the view
	virtual	void setBounds(const TRect& bounds);

			// Method to move the view to another location
	virtual	void moveTo(int x,int y);
			void moveTo(const TPoint& p)	{ moveTo(p.x,p.y); };

			// Methods to obtain bounding information
			const TRect& getBounds() const	{ return bounds; };
			void getExtent(TRect& e) const
				{ e.left() = e.top() = 0; e.botRight = size; };
			bool getClipRect(TRect& c) const;
			const TPoint& getSize() const	{ return size; };

			// Methods to invalidate the specified rectangle
	virtual	void invalidRect(TRect& rect,bool global);

			// Methods to request a repaint for the specified rectangle
			void repaint(void)
				{ repaint(TRect(0,0,size.x+1,size.y+1)); };
			void repaint(const TRect& rect);
			void globalRepaint(const TRect& rect);

			// Returns true if the 'global' coordinate is in the view
			bool includes(const Point& p) const;

			// Methods to examine the flags of the view
			ushort getState() const		{ return state; };
			ushort getOptions() const	{ return options; };

			// Method to set a state flag
	virtual	void setState(ushort state,bool set);

			// Method to set an options flag
			void setOption(ushort option,bool set);

			// Method to select the view
			void select();

			// Obtain the delta to convert from global to local coords
	virtual	void findGlobalDelta(TPoint& d);

			// Method to find the currently active modal view
	virtual	TView *findModalView();

			// Method called when idle
	virtual	void idle();

			// Method called to obtain the cursor definition
	virtual	cursor *getCursor(const TPoint& p);

			// Method to return a reference to the view's current palette
	virtual	TPalette& getPalette() const;

			// Method to map a color index to a real color value
			color_t getColor(uchar index);

			// Method to obtain the help context number for the view
	virtual	ushort getHelpCtx() const;

			// Method to set/get the owner for the view
			void setOwner(TGroup* owner)	{ TView::owner = owner; };
			TGroup *getOwner()				{ return TView::owner; };
	};

#endif	// __TVIEW_HPP
