#ifndef _SERIAL_H
#define _SERIAL_H
//============================================================================
//
//  SERIAL.H
//
//  Source include file for class SerialPort
//  Serial++ Library SPP.LIB Ver. 1.0
//
//  Copyright (c) 1991 Cortlandt Technologies, All rights reserved.
//
//	 Cortlandt Technologies, P.O. Box 195, Pleasantville NY 10570
//
//  Please direct all inquiries to the above address,
//  or via Compuserve: 71066,155
//
//============================================================================

#ifndef __STDLIB_H
#include <stdlib.h>
#endif

	enum CommPorts {
						COM_1,
						COM_2,
						CommPorts_Ct
                  };

	enum Baud {                            // divisors for selected rates
             B_50      = 2304,
             B_75      = 1536,
             B_110     = 1047,
             B_134_5   = 857,
             B_150     = 768,
				 B_300     = 384,
				 B_600     = 192,
				 B_1200    = 96,
             B_1800    = 64,
             B_2000    = 58,
				 B_2400    = 48,
             B_3600    = 32,
				 B_4800    = 24,
             B_7200    = 16,
				 B_9600    = 12,
				 B_19200   = 6,
             B_38400   = 3,
				 B_57600   = 2,
				 B_115200  = 1
				 };

	enum Parity {                  // bit pattern for parity set
					P_N = 0,           // none
               P_O = 0x0008,      // odd
					P_E = 0x0018,      // even
               P_M = 0x0028,      // mark (always 1)
               P_S = 0x0038       // space (always 0)
					};

	enum DataBits {                // bit pattern for data bits set
					  D_5 = 0,
					  D_6 = 1,
					  D_7 = 2,
					  D_8 = 3
					  };

	enum StopBits  {               // bit pattern for stop bits set
						S_1,
						S_2
						};

	enum CommRegs {              // Comm port registers:
					  RBR,           // Receive Buffer
					  THR,           // Transmit Hold
					  DLR,           // Divisor LSB
					  DMR,           // Divisor MSB
					  IER,           // Interrupt Enable
					  IIR,           // Interrupt Id
					  LCR,           // Line Control
					  MCR,           // Modem Control
					  LSR,           // Line Status
					  MSR,           // Modem Status
					  IMR,           // Interrupt Mask
					  CommRegs_Ct    // last enum has value == ct
					  };

   enum LSR_Masks {
                  DR       =  0x0001,   // data ready
                  OE       =  0x0002,   // overrun error
                  PE       =  0x0004,   // parity error
                  FE       =  0x0008,   // framing error
                  BI       =  0x0010,   // break indication
                  THRE     =  0x0020,   // THR empty
                  TSRE     =  0x0040    // Transmit Shift empty
                  };

   enum MSR_Masks {
                  DCTS     =  0x0001,   // delta clear to send
                  DDSR     =  0x0002,   // delta data set ready
                  TERI     =  0x0004,   // trailing edge ring ind
                  DDCD     =  0x0008,   // delta data carrier detect
                  CTS      =  0x0010,   // clear to send   
                  DSR      =  0x0020,   // data set ready
                  RI       =  0x0040,   // ring indicator
                  DCD      =  0x0080    // data carrier detect
                  };

   enum MCR_Masks {
                  DTR      =  0x0001,   // data terminal ready
                  RTS      =  0x0002,   // req to sent
                  OUT1     =  0x0004,   // aux user o/p 1
                  OUT2     =  0x0008,   // aux user o/p 2
                  LOOP     =  0x0010    // enable loopback
                  };



	const size_t SERIALPORT_STD_BUF_SIZE = 2048;

//===========================================================================

	struct PortAnchor;

	class SerialPort
		{
		private:

			PortAnchor 				*pa;

			Baud            		speed;
			DataBits        		data_bits;
			StopBits        		stop_bits;
			Parity          		parity;
			CommPorts       		c_port;

			void open(size_t b_size);

			static void close(PortAnchor *anc);
         static void breakCheck(void);

		public:

			SerialPort(CommPorts  cp,
						  Baud       sp=B_1200,
						  DataBits   db=D_8,
						  Parity     ps=P_N,
						  StopBits   sb=S_1,
						  size_t	 	 b_size=SERIALPORT_STD_BUF_SIZE,
                    int        brk_off = 0);

			virtual ~SerialPort(void);

//       input routines
			virtual int inbyte(unsigned char &cc, unsigned int timeout=0);
			virtual void unbyte(unsigned char cc);

			virtual size_t inmem(void *mp,
										size_t msize,
										unsigned int timeout=0);

			virtual char *instr(char *buf,
									  size_t maxlen,
									  unsigned int timeout=0);

//       output routines
			virtual void outbyte(unsigned char cc);

			virtual void outmem(const void *mp, size_t msize);

			virtual void outstr(const char *buf);

//       line status and control routines
         int getLSR(LSR_Masks mm);

         int getMSR(MSR_Masks mm);

			int setMCR(MCR_Masks mm, int state);

			void sendBreak(int dur=750);

//       SerialPort status routines
			int lineOK(void);

			int isOwner(void);

			Baud getSpeed(void);

			DataBits getDataBits(void);

			StopBits getStopBits(void);

			Parity getParity(void);

			int getBreakOff(void);

			CommPorts getCommPort(void);

	};

#endif
