#include <stdlib.h>
#include "gfsounit.h"

Gfso::Gfso(int Ba, int Fa, ColorPak &Cp)
// Initializes the three components of a Gfso: the interior, frame,
// and overall objects. Gfso also supports a new type of frame
// style: windows with CloseBoxes. The dimensions of the close box
// is hard-coded, but looks good in most cases. 
: Fso(Ba, Fa, Cp)
{
  Overall = new Grso;
  Frame   = new Grso;
  Interior = new Grso;
  SaveBuff = NULL;  SwapBuff = NULL;
  SaveSize = 0;
  if (IsCloseable()) {
     // Create a close box and place it within a 16 pixel header
     // at the top of the window 
     CloseBox = new Grso;
     CloseBox->SetSize(11, 11);   // Hard coded 
     HeaderHt = 16; // Frame header height is hard coded 
  }
  else CloseBox = NULL;
}

Gfso::~Gfso(void)
// Frees the swap buffers and the four nested Grso objects 
{
  if (IsSwappable()) {
     delete[SaveSize] SaveBuff;
     delete[SaveSize] SwapBuff;
  }
  if (IsCloseable()) delete CloseBox;
  delete Frame;
  delete Interior;
  delete Overall;
}

void Gfso::SetSize(int W, int H)
// Sets the size of the three components of a Gfso object and 
// its CloseBox if it has one. The size (W,H) is interpreted 
// to be the interior size. Memory for the swap buffers is also 
// allocated if the object is swappable, but only if the first 
// time, or if the new size is bigger than the old. 
{
  unsigned OldSaveSize;

  Interior->SetSize(W, H);
  if (IsCloseable())
     Frame->SetSize(W+Bwd*2, H+HeaderHt+Bwd*2);
     else Frame->SetSize(W+Bwd*2, H+Bwd*2);
  Overall->SetSize(Frame->Wd, Frame->Ht);
  if (IsSwappable()) {
     OldSaveSize = SaveSize;
     SaveSize = imagesize(Frame->Xul,Frame->Yul,Frame->Xlr,Frame->Ylr);
     if (SaveSize > OldSaveSize) { // Allocate 
        if (SaveBuff != NULL) {    // Re-allocate 
           delete[OldSaveSize] SaveBuff;
           delete[OldSaveSize] SwapBuff;
        }
        SaveBuff = new char[SaveSize];
        SwapBuff = new char[SaveSize];
     }
  }
}

void Gfso::SetLocn(int Xl, int Yl)
// Sets the upper-left location of the three Grsos and the 
// CloseBox if this object has one 
{
  Frame->SetLocn(Xl, Yl);
  if (IsCloseable()) {
    Interior->SetLocn(Xl+Bwd, Yl+Bwd+HeaderHt);
    CloseBox->SetLocn(Xl+Bwd+5, Yl+Bwd+2);
  }
  else Interior->SetLocn(Xl+Bwd, Yl+Bwd);
  Overall->SetLocn(Xl, Yl);
}

void Gfso::DrawFrame(char Ba, char Attr)
// Draws this object's frame and its CloseBox if it has one 
{
  int I;

  if (Bwd > 0) {
     if (Ba == 0) Ba = (Bstyle << 4) | Bwd;
     if (Attr == 0) Attr = Colors.Bc;
     Frame->Box(0, 0, Frame->Wd, Frame->Ht, Ba, Attr);
     if (IsCloseable()) {
        // This object has a CloseBox. Display it now 
        Mouse.Hide();
        for (I = 0; I < HeaderHt; I++) {
          if ((I % 2) == 0) 
	     setcolor(0);
	     else setcolor(BackGround(Colors.Wc));
	  moveto(Frame->Xul+Bwd, Frame->Yul+Bwd+I);
	  lineto(Frame->Xlr-Bwd, Frame->Yul+Bwd+I);
	}
	CloseBox->Fill(0, 0, CloseBox->Wd, CloseBox->Ht, ' ', Colors.Wc);
	CloseBox->Box(0, 0, CloseBox->Wd, CloseBox->Ht, Relief+1, Attr);
	Mouse.Show();
     }
  }
}

int Gfso::OnCloseButton(int X, int Y)
// Returns True if the coordinate is within the object's CloseBox. 
// This function returns False if the object does not have a CloseBox. 
{
  return (IsCloseable() && CloseBox->Contains(X, Y));
}

void Gfso::Clear(char Ch, char Attr)
// Clears the interior of this object using the colors in Attr. If
// Attr is zero, then the window color field of this object is used. 
{
    if (Attr > 0) Colors.Wc = Attr;
    Interior->Fill(0, 0, Interior->Wd, Interior->Ht, Ch, Colors.Wc);
}

void Gfso::GetImage(Rect *)
// Gets image from frame buffer and stores it in the save buffer if
// the object is swappable. The clipping rectangle for the
// shadows is not currently used. 
{
  Mouse.Hide();
  if (IsSwappable()) 
     ::getimage(Frame->Xul, Frame->Yul, Frame->Xlr, Frame->Ylr, SaveBuff);
  Mouse.Show();
}

void Gfso::PutImage(Rect *)
// Puts save buffer image onto Frame buffer if the object
// is swappable. The clipping rectangle for the shadows
// is not currently used. 
{
  Mouse.Hide();
  if (IsSwappable())
     ::putimage(Frame->Xul, Frame->Yul, SaveBuff, COPY_PUT);
  Mouse.Show();
}

void Gfso::Swap(Rect *, XfrDirn)
// Swaps the image on the screen.
// Note: the parameters aren't used here. 
{
  char *Tmp;

  if (IsSwappable()) {
     Mouse.Hide();
     ::getimage(Frame->Xul, Frame->Yul, Frame->Xlr, Frame->Ylr, SwapBuff);
     ::putimage(Frame->Xul, Frame->Yul, SaveBuff, COPY_PUT);
     Mouse.Show();
     // Just swap pointers between SwapBuff and SaveBuff 
     Tmp = SaveBuff; SaveBuff = SwapBuff;  SwapBuff = Tmp;
  }
}

int Gfso::TextWidth(char *Str)
// Returns the pixel width of the string. 
// Note the typecast, so that the right width is returned.
{
  return ((Grso *)Interior)->TextWidth(Str);
}

int Gfso::TextHeight(int N)
// Returns the pixel height of N rows of text 
// Note the typecast, so that the right height is returned.
{
  return ((Grso *)Interior)->TextHeight(N);
}

// ---------------- Graphics Skeleton Methods ------------------- 

Gskel::Gskel(ColorPak &Cp)
// Initializes a graphics skeleton object so that it has a
// single pixel border and is swappable
: Fso(0x11, Swappable, Cp) 
{
  Overall = new Grso;
  Frame   = new Grso;
  Interior = new Grso;
}

void Gskel::DrawFrame(char Ba, char Attr)
// Draws the frame of a skeleton object using dashed, 
// exclusive-OR lines 
{ 
  setlinestyle(DASHED_LINE,0,NORM_WIDTH);
  setwritemode(XOR_PUT);
  if (Ba == 0) Ba = (Bstyle << 4) + Bwd;
  if (Attr == 0) Attr = Colors.Bc;
  Box(0, 0, Frame->Wd, Frame->Ht, Ba, Attr);
  setwritemode(COPY_PUT);
  setlinestyle(SOLID_LINE,0,NORM_WIDTH);
}

void Gskel::Swap(Rect *, XfrDirn)
// Swapping graphics skeleton objects is simply accomplished by
// calling DrawFrame again. This works because DrawFrame uses
// exclusive-OR lines. Note: this version doesn't use the parameters. 
{
  DrawFrame(0,0);
}


