// gscroll.cpp: Graphics Scroll Window Class Implementations

#include "math.h"
#include "gscroll.h"

// ----------------------  Slider Type ------------------- 

Slider::Slider(ColorPak &Cp)
// Initialize the object that represents the slider 
: Wso(Relief+BorderWd, Swappable, Cp)
{
  return;
}

void Slider::OnMouseEnter(MsgPkt &M)
// Changes the mouse cursor when it is on the slider 
{
  Wso::OnMouseEnter(M);      // Generally call this first 
  if (((ScrollBar *)Base)->Orientation == HzOrient)
     Mouse.SetGCursor(LeftRightCursor);
     else Mouse.SetGCursor(UpDownCursor);
}

void Slider::OnMouseLeave(MsgPkt &M)
// Restores the mouse cursor when the mouse leaves the slider 
{
  Mouse.SetGCursor(ArrowCursor);
  Wso::OnMouseLeave(M);
}

void Slider::OnMouseDown(MsgPkt &M)
// Moves the slider when the mouse is anywhere within the slider 
{
  SwitchFocus(M);
  BorderHandler(M);
}

void Slider::Move(int X, int Y)
// Moves the slider to a new absolute coordinate 
{
  Wso::Move(X, Y);
  ((ScrollBar *)Base)->SendScrollPosn();
}

void Slider::OnKeyStroke(MsgPkt &M)
// Sends keystrokes to slider bar 
{
  Base->OnKeyStroke(M);
}

// --------------------- ScrollBar Type -------------------- 

ScrollBar::ScrollBar(BarOrient Orient, ColorPak &Cp)
// Initializes a scroll bar object with a slider within it 
: Wso(Recessed+BorderWd, 0x00, Cp)
{
  Orientation = Orient;
  Slide = new Slider(Cp);
  // Allow slider to overlap with the frame of the scroll bar 
  Slide->ClipToFrame = True;
  InhibitMessage = False;  
}

void ScrollBar::SetSize(int W, int H)
// Sets the size of the scroll bar and slider dep}ing 
// on whether it is horizontally or vertically oriented 
{
  int D;
  Wso::SetSize(W,H);
  if (Orientation == HzOrient)
    D = Panel->Frame->Ht - Panel->Bwd * 2;
    else D = Panel->Frame->Wd - Panel->Bwd * 2;
  Slide->SetSize(D, D);
}

void ScrollBar::Open(Iso *B, int X, int Y)
// Opens the scroll bar and the slider within it 
{
  Wso::Open(B, X, Y);
  Slide->Open(this, 0, 0);
}

void ScrollBar::Redraw(void)
// Redraw the bar and the slider 
{
  Wso::Redraw();
  Slide->Redraw();
}

void ScrollBar::OnMouseEnter(MsgPkt &M)
// Changes the mouse cursor when it is over the scroll bar 
{
  Wso::OnMouseEnter(M); // Generally call this first 
  Mouse.SetGCursor(HandCursor);
}

void ScrollBar::OnMouseLeave(MsgPkt &M)
// Restores the mouse cursor when it leaves the scroll bar 
{
  Mouse.SetGCursor(ArrowCursor);
  Wso::OnMouseLeave(M); // Generally call this last 
}

void ScrollBar::OnMouseDown(MsgPkt &M)
// Moves the slider to the location where the mouse was just pressed 
{
  if (Panel->OnInterior(M.Mx, M.My)) {
     Slide->Move(M.Mx, M.My); 
     // Need to do this so you can leave the mouse button down
     // and still move 
     Slide->OnMouseDown(M);   
  }
}

void ScrollBar::SendScrollPosn(void)
// Sends a message to the base window to report the current 
// location of the slider 
{
  int Ip;
  float Rp;
  MsgPkt M;
  if (InhibitMessage) return;
  if (Orientation == HzOrient) {
    // Compute relative position of slider in slidebar 
    Ip = Slide->Panel->Interior->Xul - Panel->Interior->Xul; 
    if (Ip == 0) {
       Rp = 0.0;
    }
    else {
      int X = Panel->Interior->Wd-1-(Slide->Panel->Overall->Wd-1) / 2;
      Rp = float(Ip) / float(X);
    }
    // Send a message to base to update according to the new
    // location of the slider 
    M.RtnCode = HzScrollMove;  M.Code = HzScrollMove;
    M.Focus = Base; // Send message to base 
    M.Mx = floor(Rp * 10000.0); M.My = 0;
  }
  else {
     Ip = SubMgr->Top->Panel->Interior->Yul - Panel->Interior->Yul;
     if (Ip == 0) {
         Rp = 0.0;
     }
     else {
       int Y = Panel->Interior->Ht-1-(Slide->Panel->Overall->Ht-1) / 2;
       Rp = float(Ip)/float(Y);
     }
     M.RtnCode = VtScrollMove;  M.Code = VtScrollMove;
     M.Focus = Base; // Send message to base
     M.My = floor(Rp * 10000.0); M.Mx = 0;
  }
  Base->Dispatch(M);
}

void ScrollBar::RcvScrollPosn(float P, BarOrient Which)
// This method takes the number P, between 0.0 and 1.0, and
// cause the Slider to move to the appropriate position 
{
  int NewPos;
  if (Which == HzOrient) {
    // Compute relative position to move slider within slidebar 
    float Rx = Panel->Interior->Wd-1-(Slide->Panel->Overall->Wd-1)/2;
    NewPos = ceil(P*Rx);
    InhibitMessage = True;
    Slide->Move(Panel->Overall->Xul+NewPos,Panel->Overall->Yul);
    InhibitMessage = False;
  }
  else {
    // Compute relative position to move slider within slidebar 
    float Ry = Panel->Interior->Ht-1-(Slide->Panel->Overall->Ht-1)/2;
    NewPos = ceil(P*Ry);
    InhibitMessage = True;
    Slide->Move(Panel->Overall->Xul,NewPos+Panel->Overall->Yul);
    InhibitMessage = False;
  }
}

void ScrollBar::OnKeyStroke(MsgPkt &M)
// Sends keystrokes to base window 
{
  Base->OnKeyStroke(M);
}

// ------------- ScrollBar Window Type (Swso) ----------- 

Swso::Swso(int Ba, int Fa, ColorPak &Cp)
// Initializes a scroll window with two scroll bars 
// and an interior window 
: Wso(Ba, Fa, Cp)
{
  Window = new Wso(Recessed+BorderWd, 0x00, Cp);
  HzSlide = new ScrollBar(HzOrient, Cp);
  VtSlide = new ScrollBar(VtOrient, Cp);
}

void Swso::SetSize(int W, int H)
// Set the size of the Swso object and the nested scroll bars. 
// Note: we have to relocate the scroll bars too. SetLocn has
// no way of knowing they need to be relocated if the window
// is resized. 
{
  Wso::SetSize(W, H);
  Window->SetSize(W-Panel->Bwd*2-30, H-Panel->Bwd*2-28);
  HzSlide->SetSize(W-Panel->Bwd*2-30, 9); 
  HzSlide->SetLocn(Panel->Bwd+2, Panel->Interior->Ht-
                   HzSlide->Panel->Interior->Ht-
                   HzSlide->Panel->Bwd*2-Panel->Bwd-2, Relc);
  VtSlide->SetSize(9, H-Panel->Bwd*2-28);
  VtSlide->SetLocn(Panel->Interior->Wd-Panel->Bwd-
                   VtSlide->Panel->Interior->Wd-
                   VtSlide->Panel->Bwd*2-2, Panel->Bwd, Relc);
}

void Swso::Open(Iso *B, int X, int Y)
{
  Wso::Open(B, X, Y);
  Window->Open(this, Panel->Bwd+2, Panel->Bwd);
  HzSlide->Open(this, Panel->Bwd+2, Panel->Interior->Ht-
                      HzSlide->Panel->Interior->Ht-
                      HzSlide->Panel->Bwd*2-Panel->Bwd-2);
  VtSlide->Open(this, Panel->Interior->Wd-Panel->Bwd-
                      VtSlide->Panel->Interior->Wd-
                      VtSlide->Panel->Bwd*2-2, Panel->Bwd);
}

void Swso::Redraw()
// Redraw the window, interior window, and scroll bars 
{
  Wso::Redraw();
  Window->Redraw();
  HzSlide->Redraw();
  VtSlide->Redraw();
}

void Swso::Dispatch(MsgPkt &M)
// Traps messages from the scroll bar and sends them to RcvScrollPosn
// to interpret them 
{
  if (M.Code == HzScrollMove) {
     RcvScrollPosn(float(M.Mx) / 10000.0, HzOrient);
     M.RtnCode = M.Code = Idle;
  }
  else if (M.Code == VtScrollMove) {
     RcvScrollPosn(float(M.My) / 10000.0, VtOrient);
     M.RtnCode = M.Code = Idle;
  }
  else Wso::Dispatch(M);
}


